/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */
package com.openexchange.office.rt2.protocol;

import java.util.Collection;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;

import javax.jms.JMSException;
import javax.jms.Message;
import javax.jms.TextMessage;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.openexchange.office.rt2.protocol.value.RT2CliendUidType;
import com.openexchange.office.rt2.protocol.value.RT2DocUidType;
import com.openexchange.office.rt2.protocol.value.RT2MessageType;
import com.openexchange.office.rt2.protocol.value.RT2UnavailableTimeType;

//=============================================================================
public class RT2MessageFactory
{
    private static final Logger log = LoggerFactory.getLogger(RT2MessageFactory.class);

    //-------------------------------------------------------------------------
    private RT2MessageFactory ()
    {}

//    final String sClientUID = RT2MessageGetSet.getClientUID (aRequest);
//    final String sDocUID    = RT2MessageGetSet.getDocUID    (aRequest);


    public static RT2Message newAdminMessage (final RT2MessageType type)
    {
        final RT2Message aMessage = new RT2Message (type);

        impl_defineMessageFlags  (aMessage, type);
        aMessage.newMessageID();

        return aMessage;
    }

    public static RT2Message newBroadcastMessage (final RT2MessageType type)
    {
        final RT2Message aMessage = new RT2Message (type);

        impl_defineMessageFlags  (aMessage, type);
        aMessage.newMessageID();

        return aMessage;
    }

    public static RT2Message newBroadcastMessage(RT2CliendUidType clientUid, RT2DocUidType docUid,
        RT2MessageType type) {
        final RT2Message aMessage = newMessage(type, clientUid, docUid);

        impl_defineMessageFlags  (aMessage, type);
        aMessage.newMessageID();

        return aMessage;
	}

    public static RT2Message newLeaveRequest(final RT2CliendUidType clientUid, final RT2DocUidType docUid) {
        return newMessage(RT2MessageType.REQUEST_LEAVE, clientUid, docUid);
    }

    public static RT2Message newCloseDocRequest(final RT2CliendUidType clientUid, final RT2DocUidType docUid) {
        return newMessage(RT2MessageType.REQUEST_CLOSE_DOC, clientUid, docUid);
    }

    public static RT2Message newSimpleNackRequest(final RT2CliendUidType clientUid, final RT2DocUidType docUid, Collection<Integer> seqNrs) throws JSONException {
        RT2Message res = newMessage(RT2MessageType.NACK_SIMPLE, clientUid, docUid);
		res.getBody().put(RT2Protocol.BODYPART_NACKS, new JSONArray(seqNrs));
        return res;
    }

    public static RT2Message newUnavailabilityRequest(final RT2CliendUidType clientUid, final RT2DocUidType docUid, final RT2UnavailableTimeType unvailableTime) {
        RT2Message res = newMessage(RT2MessageType.REQUEST_UNAVAILABILITY, clientUid, docUid);
        res.setUnvailableTime(unvailableTime);
        return res;
    }

    public static RT2Message newPing() {
        final RT2Message aMessage = new RT2Message(RT2MessageType.PING);
        aMessage.newMessageID();

        return aMessage;
    }

    public static RT2Message newPong() {
        final RT2Message aMessage = new RT2Message(RT2MessageType.PONG);
        aMessage.newMessageID();

        return aMessage;
    }

    //-------------------------------------------------------------------------
    public static RT2Message newMessage (final RT2MessageType type, final RT2CliendUidType clientUid, final RT2DocUidType docUid)
    {
        Validate.notNull(clientUid, "ClientUid is null!");
        Validate.notNull(docUid, "DocUid is null!");

        final RT2Message aMessage = new RT2Message(type);

        impl_defineMessageFlags(aMessage, type);
        aMessage.newMessageID();
        aMessage.setClientUID(clientUid);
        aMessage.setDocUID(docUid);

        return aMessage;
    }

    //-------------------------------------------------------------------------
    public static RT2Message fromJmsMessage(Message jmsMessage) throws JSONException, JMSException {
    	try {
	        String type = jmsMessage.getStringProperty(RT2MessageJmsPostProcessor.HEADER_MSG_TYPE);
	        RT2Message res = new RT2Message(RT2MessageType.valueOfType(type));
	        impl_defineMessageFlags(res, res.getType());
	        TextMessage txtMsg = (TextMessage) jmsMessage;

	        Map<String, Object> header = new HashMap<>();
	        @SuppressWarnings("unchecked")
	        Enumeration<String> enumeration = txtMsg.getPropertyNames();
	        while (enumeration.hasMoreElements()) {
	            String next = enumeration.nextElement();
	            header.put(next, txtMsg.getObjectProperty(next));
	        }
	        res.mergeHeader(header, jmsMessage);
	        String text = txtMsg.getText();
	        try {
	        	res.setBodyString(text);
	        } catch (JSONException ex) {
	        	log.error("Can parse json body {} of jms message {}.", text, jmsMessage, ex);
	        	throw ex;
	        }
	        return res;
    	} catch (JMSException ex) {
    		log.error("Error reading data of jms message {}", jmsMessage, ex);
    		throw ex;
    	}
    }

    //-------------------------------------------------------------------------
    public static RT2Message cloneMessage (final RT2Message aOriginal, final RT2MessageType cloneType)
    {
        final RT2Message aClone = RT2MessageFactory.newMessage(cloneType);

        RT2Message.copyAllHeader(aOriginal, aClone);
        RT2Message.copyBody     (aOriginal, aClone);

        // msg flags are bound to message type.
        // Dont copy it - generate it new for new type always !
        impl_defineMessageFlags  (aClone, cloneType);

        return aClone;
    }

    //-------------------------------------------------------------------------
    public static RT2Message createResponseFromMessage (final RT2Message aMessage, final RT2MessageType responseType)
    {
        final RT2Message aResponse = RT2MessageFactory.cloneMessage(aMessage, responseType);
        aResponse.setHeader("internal.type", aMessage.getType().getResponseType());
        // message id of response has to be the same !
        // but body needs to be fresh and new !
        aResponse.setBody(null);
        aResponse.setHeaderWithoutCheck(RT2Protocol.HEADER_SESSION_ID, null);
        aResponse.setHeaderWithoutCheck(RT2Protocol.HEADER_AUTO_CLOSE, null);
        return aResponse;
    }

    //-------------------------------------------------------------------------
    public static RT2Message createRequestFromMessage (final RT2Message aMessage, final RT2MessageType requestType)
    {
        final RT2Message aRequest = RT2MessageFactory.cloneMessage(aMessage, requestType);
        // new request needs new message id !
        aRequest.newMessageID();
        // but body needs to be fresh and new !
        aRequest.setBody(null);
        return aRequest;
    }

    //-------------------------------------------------------------------------
    private static void impl_defineMessageFlags (final RT2Message aMessage,
                                                 final RT2MessageType     sType   )
    {
        final Integer nFlags4Type = RT2Protocol.get().getFlags4Type(sType.getValue());
        aMessage.setFlags(nFlags4Type);
    }

    //-------------------------------------------------------------------------
    public static RT2Message fromJSONString (final String sJSON)
        throws Exception
    {
        final RT2Message aRTMsg = RT2MessageFactory.fromJSONString(sJSON, /*decode*/true);
        return aRTMsg;
    }

    //-------------------------------------------------------------------------
    public static RT2Message fromJSONString (final String  sJSON  ,
                                             final boolean bDecode)
        throws Exception
    {
        final JSONObject aJSON = new JSONObject ();
        aJSON.parseJSONString(sJSON);

        final RT2Message aRTMsg = RT2MessageFactory.fromJSON(aJSON, bDecode);
        return aRTMsg;
    }

    //-------------------------------------------------------------------------
    public static RT2Message fromJSON (final JSONObject aJSON  ,
                                       final boolean    bDecode)
        throws Exception
    {
        final String      KEY_TYPE   = bDecode ? "t" : "type"  ;
        final String      KEY_HEADER = bDecode ? "h" : "header";
        final String      KEY_BODY   = bDecode ? "b" : "body"  ;

        final RT2Protocol aProtocol  = RT2Protocol.get ();

        String sType = (String)aJSON.getString(KEY_TYPE);
        if (bDecode)
            sType = aProtocol.decode (sType);

        final RT2Message aRTMsg = newMessage(RT2MessageType.valueOfType(sType));

        if (aJSON.hasAndNotNull(KEY_HEADER))
        {
            final Map< String, Object >              lHeader = RT2MessageFactory.impl_JSONObject2Map(aJSON.getJSONObject(KEY_HEADER));
            final Iterator< Entry< String, Object >> rHeader = lHeader.entrySet().iterator();

            while (rHeader.hasNext())
            {
                final Entry< String, Object > aHeader     = rHeader.next();
                      String                  sHeader     = aHeader.getKey();
                final Object                  aValue      = aHeader.getValue();

                if (bDecode)
                    sHeader = aProtocol.decode(sHeader);

                aRTMsg.setHeader(sHeader, aValue);
            }
        }

        final JSONObject aBody = aJSON.optJSONObject(KEY_BODY);
        aRTMsg.setBody(aBody);

        final StringBuffer sValidation = new StringBuffer (256);
        if ( ! aRTMsg.isValid(sValidation))
            throw new Exception ("Msg not valid : "+sValidation.toString ());

        return aRTMsg;
    }

    //-------------------------------------------------------------------------
    private static RT2Message newMessage (final RT2MessageType sType)
    {
        Validate.notEmpty (sType.getValue(), "Invalid argument 'type'.");

        final RT2Message aMessage = new RT2Message (sType);

        impl_defineMessageFlags  (aMessage, sType);
        aMessage.newMessageID();

        return aMessage;
    }

    //-------------------------------------------------------------------------
    public static String toJSONString (final RT2Message aRTMsg)
    {
        final String sJSON = RT2MessageFactory.toJSONString(aRTMsg, /*encode*/true);
        return sJSON;
    }

    //-------------------------------------------------------------------------
    public static JSONObject toJSON (final RT2Message aRTMsg)
        throws Exception
    {
        final JSONObject aJSON = RT2MessageFactory.toJSON(aRTMsg, /*encode*/true);
        return aJSON;
    }

    //-------------------------------------------------------------------------
    public static String toJSONString (final RT2Message aRTMsg ,
                                       final boolean    bEncode)
    {
        final JSONObject aJSON = RT2MessageFactory.toJSON (aRTMsg, bEncode);
        final String     sJSON = aJSON.toString ();
        return sJSON;
    }

    //-------------------------------------------------------------------------
    public static JSONObject toJSON (final RT2Message aRTMsg ,
                                     final boolean    bEncode)
    {
        try {
            final String      KEY_TYPE   = bEncode ? "t" : "type"  ;
            final String      KEY_HEADER = bEncode ? "h" : "header";
            final String      KEY_BODY   = bEncode ? "b" : "body"  ;

            final JSONObject  aJSON      = new JSONObject ();
            final RT2Protocol aProtocol  = RT2Protocol.get ();

            String sType = aRTMsg.getType().getValue();
            if (bEncode)
                sType = aProtocol.encode (sType);
            aJSON.put(KEY_TYPE, sType);

            final JSONObject    aJSONHeader = new JSONObject ();
            final Set< String > lHeader     = aRTMsg.listHeader();
            for (String sHeader : lHeader)
            {
                // filter out 'internal' header so they do not reach the client
                if (StringUtils.startsWith(sHeader, RT2Protocol.HEADER_PREFIX_JMS))
                    continue;

                if (StringUtils.startsWith(sHeader, RT2Protocol.HEADER_PREFIX_INTERNAL))
                    continue;

                // note: value needs to be retrieved from internal map,
                // before we might encode the header name ...
                // otherwise we will get "null" as value always ...
                // because the might encoded header is not known ;-)
                final Object aValue = aRTMsg.getHeader(sHeader);

                if (bEncode)
                    sHeader = aProtocol.encode(sHeader);

                aJSONHeader.put (sHeader, aValue);
            }

            aJSON.put(KEY_HEADER, aJSONHeader);
            aJSON.put(KEY_BODY  , aRTMsg.getBody());

            return aJSON;
        } catch (JSONException ex) {
            log.error(ex.getMessage(), ex);
            throw new RuntimeException(ex);
        }
    }

    //-------------------------------------------------------------------------
    private static Map< String, Object > impl_JSONObject2Map (final JSONObject aJSON)
        throws Exception
    {
        final Map< String, Object > aMap = new HashMap< String, Object > ();
        final Set< Entry< String, Object >> lEntries = aJSON.entrySet();
        if (lEntries == null)
            return aMap;

        final Iterator< Entry< String, Object >> rEntries = lEntries.iterator();
        while (rEntries.hasNext())
        {
            final Entry< String, Object > rEntry = rEntries.next();
            final String                  sEntry = rEntry.getKey();
            final Object                  aValue = rEntry.getValue();
            aMap.put (sEntry, aValue);
        }

        return aMap;
    }

//    //-------------------------------------------------------------------------
//    public static void main (final String[] args)
//        throws Exception
//    {
//        final RT2Message m = new RT2Message ();
//        System.err.println (m.hasFlags ());
//        m.setFlags(18);
//
//        for (int i=0; i<32; ++i)
//            System.err.println (i+" : "+m.hasFlags (i));
//    }
}
