/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */
package com.openexchange.office.rt2.osgi;

import java.io.InputStream;
import java.util.Dictionary;
import java.util.Hashtable;
import java.util.List;
import java.util.Properties;

import org.apache.activemq.jms.pool.PooledConnectionFactory;
import org.osgi.framework.Bundle;
import org.osgi.framework.wiring.BundleWiring;
import org.osgi.service.event.EventConstants;
import org.osgi.service.event.EventHandler;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.jms.core.JmsTemplate;

import com.codahale.metrics.MetricRegistry;
import com.codahale.metrics.jmx.JmxReporter;
import com.hazelcast.core.HazelcastInstance;
import com.hazelcast.core.IAtomicLong;
import com.hazelcast.core.LifecycleEvent.LifecycleState;
import com.hazelcast.core.LifecycleService;
import com.hazelcast.core.Member;
import com.openexchange.ajax.requesthandler.crypto.CryptographicServiceAuthenticationFactory;
import com.openexchange.capabilities.CapabilityService;
import com.openexchange.config.ConfigurationService;
import com.openexchange.context.ContextService;
import com.openexchange.file.storage.composition.IDBasedFileAccessFactory;
import com.openexchange.file.storage.composition.IDBasedFolderAccessFactory;
import com.openexchange.file.storage.composition.crypto.CryptographicAwareIDBasedFileAccessFactory;
import com.openexchange.filemanagement.DistributedFileManagement;
import com.openexchange.filemanagement.ManagedFileManagement;
import com.openexchange.folder.FolderService;
import com.openexchange.guard.api.GuardApi;
import com.openexchange.http.grizzly.service.websocket.WebApplicationService;
import com.openexchange.jslob.JSlobService;
import com.openexchange.management.ManagementService;
import com.openexchange.office.backup.manager.DocumentBackupController;
import com.openexchange.office.document.AdvisoryLockInfoService;
import com.openexchange.office.document.EmptyDocumentCache;
import com.openexchange.office.imagemgr.IResourceManagerFactory;
import com.openexchange.office.rt2.cache.ClusterLockService;
import com.openexchange.office.rt2.cache.RT2HazelcastHelperService;
import com.openexchange.office.rt2.config.RT2ConfigItem;
import com.openexchange.office.rt2.config.RT2Const;
import com.openexchange.office.rt2.core.RT2Constants;
import com.openexchange.office.rt2.core.RT2DocProcessorClientExistsTester;
import com.openexchange.office.rt2.core.RT2DocProcessorExistsTester;
import com.openexchange.office.rt2.core.RT2GarbageCollector;
import com.openexchange.office.rt2.core.RT2NodeInfoService;
import com.openexchange.office.rt2.core.RemoveSessionListenerService;
import com.openexchange.office.rt2.core.control.IRT2NodeHealthManager;
import com.openexchange.office.rt2.core.control.RT2NodeHealthMonitor;
import com.openexchange.office.rt2.core.doc.EmergencySaver;
import com.openexchange.office.rt2.core.doc.IRT2DocProcessorManager;
import com.openexchange.office.rt2.core.doc.RT2BackgroundDocSaveManager;
import com.openexchange.office.rt2.core.doc.RT2DocProcessorManager;
import com.openexchange.office.rt2.core.doc.RT2DocStateNotifier;
import com.openexchange.office.rt2.core.management.RT2BackendManagement;
import com.openexchange.office.rt2.core.management.RT2WebSocketManagement;
import com.openexchange.office.rt2.core.metric.DocProcessorMetricService;
import com.openexchange.office.rt2.core.metric.DocProxyRequestMetricService;
import com.openexchange.office.rt2.core.metric.DocProxyResponseMetricService;
import com.openexchange.office.rt2.core.metric.WebsocketResponseMetricService;
import com.openexchange.office.rt2.core.sequence.QueueProcessorDisposer;
import com.openexchange.office.rt2.hazelcast.RT2DocOnNodeMap;
import com.openexchange.office.rt2.hazelcast.RT2NodeHealth;
import com.openexchange.office.rt2.hazelcast.RT2NodeHealthMap;
import com.openexchange.office.rt2.hazelcast.RT2NodeHealthState;
import com.openexchange.office.rt2.jms.JmsComponentFactory;
import com.openexchange.office.rt2.jms.JmsMessageSender;
import com.openexchange.office.rt2.jms.RT2AdminJmsConsumer;
import com.openexchange.office.rt2.jms.RT2DcsUpdateConsumer;
import com.openexchange.office.rt2.jms.RT2DocProcessorJmsConsumer;
import com.openexchange.office.rt2.jms.RT2DocProcessorJmsConsumerHolder;
import com.openexchange.office.rt2.jms.RT2DocProcessorJmsConsumerHolderImpl;
import com.openexchange.office.rt2.proxy.MessageCounters;
import com.openexchange.office.rt2.proxy.RT2DocInfoRegistry;
import com.openexchange.office.rt2.proxy.RT2DocProxyJmsConsumer;
import com.openexchange.office.rt2.proxy.RT2DocProxyRegistry;
import com.openexchange.office.rt2.ws.RT2SessionCountValidator;
import com.openexchange.office.rt2.ws.RT2SessionValidator;
import com.openexchange.office.rt2.ws.RT2WSApp;
import com.openexchange.office.rt2.ws.RT2WSChannelDisposer;
import com.openexchange.office.rt2.ws.RT2WebSocketListener;
import com.openexchange.office.session.SessionService;
import com.openexchange.office.tools.osgi.ActivatorBase;
import com.openexchange.office.tools.osgi.ServiceLookupRegistry;
import com.openexchange.sessiond.SessiondEventConstants;
import com.openexchange.sessiond.SessiondService;
import com.openexchange.timer.TimerService;
import com.openexchange.user.UserService;

//=============================================================================
public class RT2Activator extends ActivatorBase
{
	//-------------------------------------------------------------------------
    private static final String BUNDLE_ID = "com.openexchange.office.rt2.osgi";

    //-------------------------------------------------------------------------
    private static final Logger log = LoggerFactory.getLogger(RT2Activator.class);

    //-------------------------------------------------------------------------
	public RT2Activator() throws Exception {
		super(BUNDLE_ID);
	}

	//-------------------------------------------------------------------------
	@Override
	protected void listMandatoryServiceImports(final List< Class< ? > > lServices)
		throws Exception
	{
		lServices.add(WebApplicationService.class); // needed to register RT2WSApp instance

		lServices.add (ConfigurationService      .class);
		lServices.add (JSlobService              .class);
		lServices.add (SessiondService           .class);
		lServices.add (UserService               .class);
		lServices.add (ContextService            .class);
		lServices.add (TimerService              .class);
		lServices.add (ManagedFileManagement     .class);
		lServices.add (DistributedFileManagement .class);
		lServices.add (FolderService             .class);
		lServices.add (IDBasedFolderAccessFactory.class);
		lServices.add (IDBasedFileAccessFactory  .class);
		lServices.add (CapabilityService         .class);
		lServices.add (DocumentBackupController  .class);
		lServices.add (EmptyDocumentCache        .class);
		lServices.add (HazelcastInstance         .class);
		lServices.add (IResourceManagerFactory   .class);
		lServices.add (ManagementService         .class);
		lServices.add (RT2NodeHealthMap          .class);
		lServices.add (RT2DocOnNodeMap           .class);
		lServices.add (SessionService            .class);
		lServices.add (AdvisoryLockInfoService   .class);
	}

	//-------------------------------------------------------------------------
	@Override
	protected void listOptionalServiceImports(final List< Class< ? > > lServices)
		throws Exception
	{
        lServices.add(CryptographicAwareIDBasedFileAccessFactory.class);
        lServices.add(CryptographicServiceAuthenticationFactory .class);
        lServices.add(GuardApi                                  .class);
	}

	//-------------------------------------------------------------------------
	@Override
	protected void activate() throws Exception {
	    log.info("activate RT2 env ...");

	    try {
	        final ManagementService managementService = ServiceLookupRegistry.get().getService(ManagementService.class);
	        final RT2ConfigItem     aCfg    = RT2ConfigItem.get();
	        final HazelcastInstance aHzCore = this.getService(HazelcastInstance.class);
	        final RT2HazelcastHelperService hzHelperService = new RT2HazelcastHelperService(aHzCore);

	        final RT2DocOnNodeMap rt2DocOnNodeMap = ServiceLookupRegistry.get().getService(RT2DocOnNodeMap.class);
	        m_aRT2NodeInfo = new RT2NodeInfoService(rt2DocOnNodeMap);
	        registerService(RT2NodeInfoService.class, m_aRT2NodeInfo);

	        //---------------------------------------

		    Properties versionProps = new Properties();
		    for (Bundle bundle : m_aBundleContext.getBundles()) {
		    	if (bundle.getSymbolicName().equals("com.openexchange.office.dependencies")) {
		    		InputStream stream = bundle.adapt(BundleWiring.class).getClassLoader().getResourceAsStream("version.properties");
		    		try {
		    			versionProps.load(stream);
		    		} catch (Exception ex) {
		    			log.info("Cannot determine version because version.properties file is missing!");
		    			versionProps = new Properties();
		    		}
		    	}
		    }

		    log.info("... activate cluster lock service");
		    ClusterLockService clusterLockService = new ClusterLockService(aHzCore);
		    registerService(ClusterLockService.class, clusterLockService);
		    this.m_aClusterLockService = clusterLockService;

	        log.info("... activate metrics library");
	        m_aMetricRegistry = new MetricRegistry();
	        m_aJmxReporter =JmxReporter.forRegistry(m_aMetricRegistry).inDomain("com.openexchange.office.rt2.metrics").build();
	        m_aJmxReporter.start();
	        registerService(MetricRegistry.class, m_aMetricRegistry);
	        DocProcessorMetricService docProcessorMetricService = new DocProcessorMetricService();
	        registerService(DocProcessorMetricService.class, docProcessorMetricService);
	        DocProxyRequestMetricService docProxyRequestMetricService = new DocProxyRequestMetricService();
	        registerService(DocProxyRequestMetricService.class, docProxyRequestMetricService);
	        DocProxyResponseMetricService docProxyResponseMetricService = new DocProxyResponseMetricService();
	        registerService(DocProxyResponseMetricService.class, docProxyResponseMetricService);
	        WebsocketResponseMetricService websocketResponseMetriceService = new WebsocketResponseMetricService();
	        registerService(WebsocketResponseMetricService.class, websocketResponseMetriceService);
	        MessageCounters messageCounters = new MessageCounters(m_aMetricRegistry);
	        registerService(MessageCounters.class, messageCounters);

	        //---------------------------------------

	        log.info("... activate rt2 atomic full-member count");
	        boolean bFullMember = !aHzCore.getCluster().getLocalMember().isLiteMember();
	        if (bFullMember)
	        {
	            final IAtomicLong aClusterMemberCount = aHzCore.getAtomicLong(RT2Constants.RT2_CLUSTER_FULL_MEMBER_COUNT);
	            aClusterMemberCount.incrementAndGet();
	        }

	        //---------------------------------------
			log.info("... create jms component factory");
			m_aJmsComponentFactory = new JmsComponentFactory();
			m_aconnFactory = m_aJmsComponentFactory.getPooledConnectionFactory();
			registerService(PooledConnectionFactory.class, m_aconnFactory);
			JmsTemplate jmsTemplateWithoutTtl = m_aJmsComponentFactory.getJmsTemplate(0);
			JmsTemplate jmsTemplateWithTtl = m_aJmsComponentFactory.getJmsTemplate(60000);

	        //---------------------------------------
			log.info("... create JmsMessageSender");
			m_aJmsMessageSender = new JmsMessageSender(jmsTemplateWithoutTtl, jmsTemplateWithTtl, rt2DocOnNodeMap);
			registerService(JmsMessageSender.class, m_aJmsMessageSender);

	        //---------------------------------------
	        log.info("... activate doc proxy registry");
			final RT2DocProxyRegistry aRT2DocRegistry = new RT2DocProxyRegistry (m_aMetricRegistry, m_aJmsMessageSender);
			registerService(RT2DocProxyRegistry.class, aRT2DocRegistry);
			m_aRT2DocProxyRegistry = aRT2DocRegistry;

	        //---------------------------------------
	        log.info("... activate websocket");
	        int maxSessions = new RT2ConfigItem().getRT2MaxSessionsPerNode();
	        final SessionService sessionService = ServiceLookupRegistry.get().getService(SessionService.class);
	        final RT2SessionCountValidator sessionCountValidator = new RT2SessionCountValidator(sessionService, maxSessions);
	        final RT2SessionValidator sessionValidator = new RT2SessionValidator();
	        final RT2WSApp aWSApp = new RT2WSApp(m_aMetricRegistry, m_aRT2DocProxyRegistry);
	        final RT2WSChannelDisposer channelDisposer = new RT2WSChannelDisposer(m_aRT2DocProxyRegistry, aWSApp, m_aRT2NodeInfo);
	        final RT2WebSocketListener webSocketListener =
	        		new RT2WebSocketListener(m_aRT2NodeInfo, sessionValidator, sessionCountValidator, channelDisposer, m_aRT2DocProxyRegistry,
	        								 aWSApp, websocketResponseMetriceService, docProxyRequestMetricService, docProxyResponseMetricService);
	        aWSApp.setWebSocketListener(webSocketListener);
	        m_aWsChannelDisposer = channelDisposer;

	        registerService(RT2WSChannelDisposer.class, channelDisposer);


			final WebApplicationService aWebServer = getService (WebApplicationService.class);
			aWebServer.registerWebSocketApplication("", RT2Const.RT2_URL_PATTERN, aWSApp, null);

			m_aWSApp = aWSApp;


	        log.info("... active backend management");
	        RT2WebSocketManagement webSocketManagement = new RT2WebSocketManagement(aWSApp, channelDisposer, sessionCountValidator);
	        managementService.registerMBean(webSocketManagement.getObjectName(), webSocketManagement);

	        registerService(RT2WSApp.class, aWSApp);
	        registerService(RT2WebSocketListener.class, webSocketListener);
	        m_aWebSocketListener = webSocketListener;

	        //---------------------------------------

	        LOG.info("... register rt2 node to node health map");
	        final RT2NodeHealthMap   aRT2NodeHealthMap = this.getService(RT2NodeHealthMap.class);
	        final RT2NodeHealthState aNodeHealthState  = new RT2NodeHealthState(m_aRT2NodeInfo.getNodeUUID(),
	                                                                            aCfg.getOXNodeID(),
	                                                                            RT2NodeHealth.RT2_NODE_HEALTH_UP,
	                                                                            RT2NodeHealth.getNodeTypeString(bFullMember),
	                                                                            RT2NodeHealth.RT2_CLEANUP_UUID_EMPTY);
	        aRT2NodeHealthMap.set(m_aRT2NodeInfo.getNodeUUID(), aNodeHealthState);

	        //---------------------------------------
	        log.info("... activate doc info registry");
			final RT2DocInfoRegistry aRT2DocInfoRegistry = new RT2DocInfoRegistry(aHzCore);
			registerService(RT2DocInfoRegistry.class, aRT2DocInfoRegistry);
			m_aRT2DocInfoRegistry = aRT2DocInfoRegistry;

	        //---------------------------------------
	        log.info("... activate doc processor manager");
	        final RT2DocProcessorManager aDocProcManager = new RT2DocProcessorManager ();
	        registerService(IRT2DocProcessorManager.class, aDocProcManager);
	        m_aDocProcessorManager = aDocProcManager;
	        m_aQueueProcessorDisposer = new QueueProcessorDisposer(m_aRT2NodeInfo);
	        registerService(QueueProcessorDisposer.class, m_aQueueProcessorDisposer);

	        //---------------------------------------
	        log.info("... activate RT2GarbageCollector");
	        final RT2GarbageCollector aRT2GC = new RT2GarbageCollector (aHzCore, m_aDocProcessorManager, m_aMetricRegistry, m_aRT2NodeInfo, clusterLockService);
	        aRT2GC.start();
	        registerService(RT2GarbageCollector.class, aRT2GC);
	        m_aRT2GC = aRT2GC;

	        //---------------------------------------
	        log.info("... activate node health monitor");
	        final RT2NodeHealthMonitor aRT2HealthMonitor = new RT2NodeHealthMonitor(m_aRT2NodeInfo, aRT2DocRegistry, clusterLockService, aDocProcManager);

	        aHzCore.getCluster().addMembershipListener(aRT2HealthMonitor);
	        final LifecycleService aLifecycleService = aHzCore.getLifecycleService();
	        aLifecycleService.addLifecycleListener(aRT2HealthMonitor);

	        aRT2HealthMonitor.start(aLifecycleService.isRunning() ? LifecycleState.STARTED : LifecycleState.SHUTDOWN);
	        registerService(IRT2NodeHealthManager.class, aRT2HealthMonitor);
	        m_aNodeHealthMonitor = aRT2HealthMonitor;


	        //---------------------------------------
	        log.info("... activate admin channel");
	        m_aDocProcessorClientExistsTester = new RT2DocProcessorClientExistsTester(m_aJmsMessageSender, m_aRT2DocProxyRegistry, aDocProcManager, m_aRT2NodeInfo, hzHelperService);
	        m_aDocProcessorExistsTester = new RT2DocProcessorExistsTester(m_aJmsMessageSender, m_aRT2DocProxyRegistry, aDocProcManager, m_aRT2NodeInfo, hzHelperService);
	        m_aRT2Admin = new RT2AdminJmsConsumer (jmsTemplateWithoutTtl, aDocProcManager, m_aRT2NodeInfo,  aRT2DocInfoRegistry, m_aRT2DocProxyRegistry, aRT2HealthMonitor,
	        									   m_aDocProcessorClientExistsTester, m_aDocProcessorExistsTester, versionProps, clusterLockService, m_aJmsMessageSender);
	        m_aRT2Admin.receiveAdminMessages();
	        registerService(RT2AdminJmsConsumer.class, m_aRT2Admin);

	        //---------------------------------------
	        log.info("... activate doc processor delegat holder");
	        final RT2DocProcessorJmsConsumer docProcessorJmsConsumer = new RT2DocProcessorJmsConsumer();
	        docProcessorJmsConsumer.startReceiveMessages();
            m_aDocProcessorDelegate = new RT2DocProcessorJmsConsumerHolderImpl(docProcessorJmsConsumer);
            registerService(RT2DocProcessorJmsConsumerHolder.class, m_aDocProcessorDelegate);

	        //---------------------------------------
	        log.info("... activate Doc Proxy Jms Consumer");
	        m_aRT2DocProxyJmsConsumer = new RT2DocProxyJmsConsumer(aRT2DocRegistry, m_aconnFactory);
	        m_aRT2DocProxyJmsConsumer.receiveClientDocMessages();

	        //---------------------------------------
	        log.info("... activate Doc Proxy Jms Consumer");
	        m_aRT2DcsUpdateConsumer = new RT2DcsUpdateConsumer((PooledConnectionFactory) m_aconnFactory, m_aRT2NodeInfo);
	        m_aRT2DcsUpdateConsumer.receiveDcsUpdateMessages();

	        //---------------------------------------
	        log.info("... active backend management");
	        m_aRT2BackendManagement = new RT2BackendManagement(aDocProcManager, aRT2DocRegistry, m_aRT2DocInfoRegistry, m_aJmsComponentFactory.getPooledConnectionFactory(),
	        		                                           aRT2GC, m_aRT2NodeInfo, m_aQueueProcessorDisposer, m_aDocProcessorExistsTester, aHzCore,
	        		                                           clusterLockService);

	        managementService.registerMBean(m_aRT2BackendManagement.getObjectName(), m_aRT2BackendManagement);
	        registerService(RT2BackendManagement.class, m_aRT2BackendManagement);

		    log.info("Registering session removed listener...");
		    RemoveSessionListenerService sessionGoneListener = new RemoveSessionListenerService(m_aJmsMessageSender, m_aRT2NodeInfo);
		    final Dictionary<String, Object> properties = new Hashtable<String, Object>(1);
		    properties.put(EventConstants.EVENT_TOPIC, new String[] { SessiondEventConstants.TOPIC_REMOVE_SESSION});
		    m_aBundleContext.registerService(new String[] { RemoveSessionListenerService.class.getName() , EventHandler.class.getName() }, sessionGoneListener, properties);
		    registerService(RemoveSessionListenerService.class, sessionGoneListener);
		    m_aRemoveSessionListenerService = sessionGoneListener;

	        //---------------------------------------
	        if (aCfg.isBackgroundSaveOn())
	        {
	            log.info("... activate background save");
	            // Register document specific background processor dependent on configuration entries.
	            final RT2BackgroundDocSaveManager aBackgroundDocSave = new RT2BackgroundDocSaveManager();
	            aBackgroundDocSave.start();
	            registerService(RT2BackgroundDocSaveManager.class, aBackgroundDocSave);
	            m_aBackgroundDocSave = aBackgroundDocSave;
		        final RT2DocStateNotifier aDocStateNotifier = new RT2DocStateNotifier(aDocProcManager, aBackgroundDocSave);
		        registerService(RT2DocStateNotifier.class, aDocStateNotifier);
		        m_aDocStateNotifier = aDocStateNotifier;
	        } else {
		        final RT2DocStateNotifier aDocStateNotifier = new RT2DocStateNotifier(aDocProcManager);
		        registerService(RT2DocStateNotifier.class, aDocStateNotifier);
		        m_aDocStateNotifier = aDocStateNotifier;
	        }



	        log.info("RT2 core env started.");
	    } catch (Exception e) {
	        log.error("RT2Activator caught exception on start-up, it's possible that RT2 is not correctly working, please check your configuration & set-up!", e);
	    }
	}

	//-------------------------------------------------------------------------
	@Override
	protected void deactivate() throws Exception {

	    log.info("deactivate RT2 core env ...");

        //---------------------------------------
        final RT2ConfigItem     aCfg    = RT2ConfigItem.get();
        final HazelcastInstance aHzCore = this.getService(HazelcastInstance.class);

        //---------------------------------------
        log.info("... decrease rt2 atomic full-member count");

        boolean bFullMember = !aHzCore.getCluster().getLocalMember().isLiteMember();
        if (bFullMember)
        {
            final IAtomicLong aClusterMemberCount = aHzCore.getAtomicLong(RT2Constants.RT2_CLUSTER_FULL_MEMBER_COUNT);
            aClusterMemberCount.decrementAndGet();
        }

        //---------------------------------------
        log.info("... update rt2 node health map with new shutdown state");

        final Member             aMySelfNode       = aHzCore.getCluster().getLocalMember();
        final String             sNodeUUID         = aMySelfNode.getUuid();
        final RT2NodeHealthMap   aRT2NodeHealthMap = this.getService(RT2NodeHealthMap.class);
              RT2NodeHealthState aNodeHealthState  = new RT2NodeHealthState(sNodeUUID,
                                                                            aCfg.getOXNodeID(),
                                                                            RT2NodeHealth.RT2_NODE_HEALTH_SHUTTING_DOWN,
                                                                            RT2NodeHealth.getNodeTypeString(bFullMember),
                                                                            RT2NodeHealth.RT2_CLEANUP_UUID_EMPTY);
        aRT2NodeHealthMap.set(sNodeUUID, aNodeHealthState);

        //---------------------------------------
        log.info("... broadcast shutdown to all dependent document clients");

        if (m_aDocProcessorManager != null)
            m_aDocProcessorManager.preShutdown();

        //---------------------------------------
        log.info("... deactivate background save and do emergency save");

        final RT2BackgroundDocSaveManager   aBackgroundDocSave = m_aBackgroundDocSave;
        m_aBackgroundDocSave = null;

        // perform emergency save for open & modified documents
        if (aBackgroundDocSave != null)
            aBackgroundDocSave.shutdown(); // will do an emergency save
        else
            EmergencySaver.emergencySave();


        //---------------------------------------
        log.info("... deactivate doc state notifier");

        // closes the singleton that notifies if doc processors are created/disposed
        m_aDocStateNotifier = null;

        //---------------------------------------
        log.info("... deactivate RT2GarbageCollector");

        // stops the garbage collector for stale RT2 instances (doc proxies/processors)
        final RT2GarbageCollector aRT2GC = m_aRT2GC;
        m_aRT2GC = null;
        if (aRT2GC != null)
            aRT2GC.stop ();

        //---------------------------------------
        log.info("... deactivate doc proxy registry");

        // stops the registry for doc proxies
        final RT2DocProxyRegistry aRT2DocRegistry = m_aRT2DocProxyRegistry;
        m_aRT2DocProxyRegistry = null;
        if (aRT2DocRegistry != null)
            aRT2DocRegistry.stop ();

        //---------------------------------------
        log.info("... deactivate doc processor delegate");

        RT2DocProcessorJmsConsumerHolder aDocProcessorDelegate = m_aDocProcessorDelegate;
        m_aDocProcessorDelegate = null;
        if (aDocProcessorDelegate != null) {
            aDocProcessorDelegate.destroy();
        }

        //---------------------------------------
        log.info("... deactivate doc processor manager");

        // closes the singleton to store all opened doc processor on this node
        if (m_aDocProcessorManager != null)
            m_aDocProcessorManager.shutdown();

        if (m_aQueueProcessorDisposer != null)
        	m_aQueueProcessorDisposer.stop();

        //---------------------------------------
        log.info("... deactivate rt2 admin");

        RT2AdminJmsConsumer aRT2Admin = m_aRT2Admin;
        m_aRT2Admin = null;
        if (aRT2Admin != null) {
            aRT2Admin.stop();
        }
        if (m_aDocProcessorClientExistsTester != null) {
        	m_aDocProcessorClientExistsTester.stop();
        }
        if (m_aDocProcessorExistsTester != null) {
        	m_aDocProcessorExistsTester.stop();
        }

        //---------------------------------------
        log.info("... deactivate jms consumer");

        // stop the jms consumer
        RT2DocProxyJmsConsumer aRT2DocProxyJmsConsumer = m_aRT2DocProxyJmsConsumer;
        m_aRT2DocProxyJmsConsumer = null;
        if (aRT2DocProxyJmsConsumer != null) {
            aRT2DocProxyJmsConsumer.stopReceivingClientDocMessages();
        }

        //---------------------------------------
        log.info("... deactivate jms consumer");

        // stop the jms consumer
        RT2DcsUpdateConsumer aRT2DcsUpdateConsumer = m_aRT2DcsUpdateConsumer;
        m_aRT2DcsUpdateConsumer = null;
        if (aRT2DcsUpdateConsumer != null) {
        	aRT2DcsUpdateConsumer.stopReceivingDcsUpdateMessages();
        }
        if (m_aJmsComponentFactory != null) {
        	m_aJmsComponentFactory.stop();
        }

        //---------------------------------------
        log.info("... update rt2 node health map with shutdown completed state");

        // update the HZ node health map and store the final "shutdown" state for
        // this node
        aNodeHealthState = new RT2NodeHealthState(sNodeUUID, aCfg.getOXNodeID(), RT2NodeHealth.RT2_NODE_HEALTH_SHUTDOWN, RT2NodeHealth.getNodeTypeString(bFullMember), RT2NodeHealth.RT2_CLEANUP_UUID_EMPTY);
        aRT2NodeHealthMap.set(sNodeUUID, aNodeHealthState);

        //---------------------------------------
        log.info("... deactivate cluster lock service");
        if (m_aClusterLockService != null) {
        	m_aClusterLockService.stop();
        }

        //---------------------------------------
        log.info("... deactivate node health monitor");
        final RT2NodeHealthMonitor aRT2HealthMonitor = m_aNodeHealthMonitor;

        if (aRT2HealthMonitor != null)
            aRT2HealthMonitor.stop ();

    	//-------------------------------------------------------------------------
        log.info("... deactivate connection factory");
        if (m_aconnFactory != null) {
            m_aconnFactory.stop();
        }

        //-------------------------------------------------------------------------
        log.info("... deactivate JMX reporter");
        if (m_aJmxReporter != null) {
            m_aJmxReporter.stop();
        }

        //-------------------------------------------------------------------------
        log.info("... deactivate websocket");
        deactivateWebsocket();

        //---------------------------------------
        log.info("... remove node from rt2 node health map");
        aRT2NodeHealthMap.remove(sNodeUUID);

	    log.info("RT2 core env deactivated.");
	}

	//-------------------------------------------------------------------------
	private void deactivateWebsocket() throws Exception {

	    log.info("deregister RT2 WebSocket handler ...");

		final WebApplicationService aWebServer = getService (WebApplicationService.class);
		final RT2WSApp              aWSApp     = m_aWSApp;
		aWSApp.stop();
		if (m_aWsChannelDisposer != null) {
			m_aWsChannelDisposer.destroy();
		}
		if (m_aWebSocketListener != null) {
			m_aWebSocketListener.stop();
		}
		aWebServer.unregisterWebSocketApplication(aWSApp);
	}

	//-------------------------------------------------------------------------
	private RT2WSApp m_aWSApp;

    //-------------------------------------------------------------------------
	private MetricRegistry m_aMetricRegistry = null;
	private JmxReporter m_aJmxReporter = null;

    //-------------------------------------------------------------------------
    private RT2NodeInfoService m_aRT2NodeInfo = null;

    //-------------------------------------------------------------------------
    private RT2DocProxyRegistry m_aRT2DocProxyRegistry = null;

    //-------------------------------------------------------------------------
    private RT2DocInfoRegistry m_aRT2DocInfoRegistry = null;

    //-------------------------------------------------------------------------
    private PooledConnectionFactory m_aconnFactory = null;

    //-------------------------------------------------------------------------
    private JmsMessageSender m_aJmsMessageSender;

    //-------------------------------------------------------------------------
    private RT2BackendManagement m_aRT2BackendManagement;

    //-------------------------------------------------------------------------
    private RT2GarbageCollector m_aRT2GC = null;

    //-------------------------------------------------------------------------
    private RT2BackgroundDocSaveManager m_aBackgroundDocSave = null;

    //-------------------------------------------------------------------------
    private RT2DocStateNotifier m_aDocStateNotifier = null;

    //-------------------------------------------------------------------------
    private RT2DocProcessorManager m_aDocProcessorManager = null;

    //-------------------------------------------------------------------------
    private RT2NodeHealthMonitor m_aNodeHealthMonitor = null;

    private RT2DocProcessorJmsConsumerHolder m_aDocProcessorDelegate = null;

    private RT2DocProxyJmsConsumer m_aRT2DocProxyJmsConsumer = null;

    private RT2DcsUpdateConsumer m_aRT2DcsUpdateConsumer = null;

    private RT2AdminJmsConsumer m_aRT2Admin = null;

    private RT2DocProcessorClientExistsTester m_aDocProcessorClientExistsTester = null;

    private RT2DocProcessorExistsTester m_aDocProcessorExistsTester = null;

    private QueueProcessorDisposer m_aQueueProcessorDisposer = null;

    private RT2WSChannelDisposer m_aWsChannelDisposer = null;

    private ClusterLockService m_aClusterLockService = null;

    private JmsComponentFactory m_aJmsComponentFactory = null;

    private RT2WebSocketListener m_aWebSocketListener = null;

    private RemoveSessionListenerService m_aRemoveSessionListenerService = null;
}
