/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.rt2.hazelcast.osgi;

import java.util.List;

import org.apache.commons.lang3.StringUtils;
import org.osgi.framework.BundleException;

import com.hazelcast.config.Config;
import com.hazelcast.core.HazelcastInstance;
import com.hazelcast.core.Member;
import com.openexchange.hazelcast.serialization.CustomPortableFactory;
import com.openexchange.management.ManagementService;
import com.openexchange.office.rt2.hazelcast.HzDocOnNodeMap;
import com.openexchange.office.rt2.hazelcast.HzHelper;
import com.openexchange.office.rt2.hazelcast.HzNodeHealthMap;
import com.openexchange.office.rt2.hazelcast.ManagementHouseKeeper;
import com.openexchange.office.rt2.hazelcast.RT2DocOnNodeMap;
import com.openexchange.office.rt2.hazelcast.RT2NodeHealthMap;
import com.openexchange.office.rt2.hazelcast.serialization.PortableNodeDocsPredicateFactory;
import com.openexchange.office.rt2.hazelcast.serialization.PortableNodeHealthPredicateFactory;
import com.openexchange.office.rt2.hazelcast.serialization.PortableNodeHealthStateFactory;
import com.openexchange.office.rt2.hazelcast.serialization.PortableNodeHealthStatePredicateFactory;
import com.openexchange.office.tools.logging.v2.ELogLevel;
import com.openexchange.office.tools.logging.v2.impl.Logger;
import com.openexchange.office.tools.logging.v2.impl.Slf4JLogger;
import com.openexchange.office.tools.osgi.ActivatorBase;
import com.openexchange.office.tools.osgi.ServiceLookupRegistry;

//=============================================================================
public class RT2HazelcastActivator extends ActivatorBase
{
	//-------------------------------------------------------------------------
    private static final String SERVICE_NAME = "com.openexchange.office.rt2.hazelcast";

    //-------------------------------------------------------------------------
    private static final Logger LOG = Slf4JLogger.create(RT2HazelcastActivator.class);

	//-------------------------------------------------------------------------
    public RT2HazelcastActivator ()
        throws Exception
    {
    	super (SERVICE_NAME);
    }

    //-------------------------------------------------------------------------
    @Override
    protected void listMandatoryServiceImports(final List< Class< ? >> lServices)
        throws Exception
    {
        lServices.add(HazelcastInstance    .class);
        lServices.add(ManagementService    .class);
    }

    //-------------------------------------------------------------------------
    @Override
    protected void listOptionalServiceImports(final List< Class< ? >> lServices)
    	throws Exception
    {
        // no optional services
    }

    //-------------------------------------------------------------------------
    @Override
    protected void activate()
        throws Exception
    {
        final ManagementHouseKeeper managementHouseKeeper = ManagementHouseKeeper.getInstance();
        managementHouseKeeper.initialize(ServiceLookupRegistry.get());

        final HazelcastInstance hazelcastInstance = getService(HazelcastInstance.class);
        final Config config = hazelcastInstance.getConfig();

        //-------------------------------------------------------------------------

        final String sNodeHealthMap = HzHelper.discoverMapName(config, "rt2NodeHealthMap-");
        if(StringUtils.isEmpty(sNodeHealthMap)) {
            final String msg = "Distributed rt2 node health map couldn't be found in hazelcast configuration";
            throw new IllegalStateException(msg, new BundleException(msg, BundleException.ACTIVATOR_ERROR));
        }
        LOG.forLevel(ELogLevel.E_INFO).withMessage("Registered rt2NodeHealthMap to Hazelcast").log();

        m_aNodeHealthMap = new HzNodeHealthMap(sNodeHealthMap);

        managementHouseKeeper.addManagementObject(m_aNodeHealthMap.getManagementObject());
        registerService(RT2NodeHealthMap.class, m_aNodeHealthMap, null);

        //-------------------------------------------------------------------------

        final String sDocOnNodeMap = HzHelper.discoverMapName(config, "rt2DocOnNodeMap-");
        if(StringUtils.isEmpty(sDocOnNodeMap)) {
            final String msg = "Distributed rt2 doc node map couldn't be found in hazelcast configuration";
            throw new IllegalStateException(msg, new BundleException(msg, BundleException.ACTIVATOR_ERROR));
        }
        LOG.forLevel(ELogLevel.E_INFO).withMessage("Registered rt2DocOnNodeMap to Hazelcast").log();
        m_aDocOnNodeMap = new HzDocOnNodeMap(sDocOnNodeMap);

        managementHouseKeeper.addManagementObject(m_aDocOnNodeMap.getManagementObject());
        registerService(RT2DocOnNodeMap.class, m_aDocOnNodeMap, null);

        //-------------------------------------------------------------------------

        registerService(CustomPortableFactory.class, new PortableNodeHealthStateFactory());
        registerService(CustomPortableFactory.class, new PortableNodeHealthPredicateFactory());
        registerService(CustomPortableFactory.class, new PortableNodeDocsPredicateFactory());
        registerService(CustomPortableFactory.class, new PortableNodeHealthStatePredicateFactory());

        LOG.forLevel(ELogLevel.E_INFO).withMessage("Registered custom portable factories to Hazelcast").log();

        managementHouseKeeper.exposeManagementObjects();
    }

    //-------------------------------------------------------------------------
    @Override
    protected void deactivate()
        throws Exception
    {
        ManagementHouseKeeper.getInstance().cleanup();
    }

	//-------------------------------------------------------------------------
	private volatile HzNodeHealthMap m_aNodeHealthMap = null;

	//-------------------------------------------------------------------------
	private volatile HzDocOnNodeMap  m_aDocOnNodeMap  = null;
}
