/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.rt2.ws;

import java.nio.charset.MalformedInputException;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.Executors;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

import org.apache.commons.lang3.tuple.Pair;
import org.glassfish.grizzly.GrizzlyFuture;
import org.glassfish.grizzly.websockets.DataFrame;
import org.glassfish.grizzly.websockets.WebSocket;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.DisposableBean;

import com.openexchange.log.LogProperties;
import com.openexchange.logging.MDCEnabledThreadGroup;
import com.openexchange.office.rt2.config.RT2ConfigItem;
import com.openexchange.office.rt2.core.RT2NodeInfoService;
import com.openexchange.office.rt2.core.RT2ThreadFactoryBuilder;
import com.openexchange.office.rt2.core.metric.DocProxyResponseMetricService;
import com.openexchange.office.rt2.protocol.RT2Message;
import com.openexchange.office.rt2.protocol.RT2MessageFactory;
import com.openexchange.office.rt2.protocol.value.RT2CliendUidType;
import com.openexchange.office.rt2.protocol.value.RT2DocUidType;
import com.openexchange.office.rt2.protocol.value.RT2MessageType;
import com.openexchange.office.rt2.proxy.RT2DocProxy;
import com.openexchange.office.rt2.proxy.RT2DocProxyLogInfo.Direction;
import com.openexchange.office.rt2.proxy.RT2DocProxyRegistry;
import com.openexchange.office.tools.error.ErrorCode;
import com.openexchange.office.tools.osgi.ServiceLookupRegistryService;

public class RT2WSChannelDisposer implements DisposableBean {

	private static final Logger log = LoggerFactory.getLogger(RT2WSChannelDisposer.class);

	private final RT2WSApp webSocketApplication;

	private final RT2DocProxyRegistry docProxyRegistry;

	private final RT2NodeInfoService nodeInfo;

	private final Map<RT2CliendUidType, Pair<RT2DocUidType, RT2ChannelId>> channels = new ConcurrentHashMap<>();

	private final Map<RT2ChannelId, Long> channelsOffline = new ConcurrentHashMap<>();

	private final BlockingQueue<RT2Message> responseQueue = new LinkedBlockingQueue<>();

	private final ScheduledExecutorService channelExistsExecutorService = Executors.newSingleThreadScheduledExecutor(new RT2ThreadFactoryBuilder("RT2ChannelExists-%d").build());

	private final ScheduledExecutorService keepAliveExecutorService = Executors.newSingleThreadScheduledExecutor(new RT2ThreadFactoryBuilder("RT2ChannelKeepAlive-%d").build());

	private final long gcTimeout;

	private DocProxyResponseMetricService docProxyResponseMetricService;

	public RT2WSChannelDisposer(RT2DocProxyRegistry docProxyRegistry, RT2WSApp rt2WsApp, RT2NodeInfoService rt2NodeInfo) {
		this.docProxyRegistry = docProxyRegistry;
		this.webSocketApplication = rt2WsApp;
		this.nodeInfo = rt2NodeInfo;

		gcTimeout = RT2ConfigItem.get().getRT2GCOfflineTresholdInMS();
		channelExistsExecutorService.scheduleAtFixedRate(new RT2WSChannelExistsCheckThread(), (gcTimeout / 10), (gcTimeout / 10), TimeUnit.MILLISECONDS);

        final long nPingFrequencyInMS = RT2ConfigItem.get().getRT2KeepAliveFrequencyInMS();
        if (nPingFrequencyInMS != 0) {
            keepAliveExecutorService.scheduleAtFixedRate(new RT2WSChannelKeepAliveThread(), nPingFrequencyInMS, nPingFrequencyInMS, TimeUnit.MILLISECONDS);
        }

		for (int i=0;i<8;++i) {
			final Thread thread = new Thread(new MDCEnabledThreadGroup(), new ResponseThread(), "ResponseThread-" + i);
			thread.setDaemon(true);
			thread.start();
		}
	}

	@Override
	public void destroy() {
		channelExistsExecutorService.shutdown();
		keepAliveExecutorService.shutdown();
	}

	public Set<Pair<RT2CliendUidType, RT2DocUidType>> getClientDocPairsAssociatedToChannel(RT2ChannelId channelId) {
		final Set<Pair<RT2CliendUidType, RT2DocUidType>> res = channels.entrySet().stream().
				filter(p -> p.getValue().getValue().equals(channelId)).
				map(p -> Pair.of(p.getKey(), p.getValue().getKey())).
				collect(Collectors.toSet());
		return res;
	}

	public Set<RT2CliendUidType> getClientsAssociatedToChannel(RT2ChannelId channelId) {
		final Set<RT2CliendUidType> res = new HashSet<>();
		channels.forEach((clientUid, p) -> {
			if (p.getValue().equals(channelId)) {
				res.add(clientUid);
			}
		});
		return res;
	}

	public Set<RT2DocUidType> getDocsAssociatedToChannel(RT2ChannelId channelId) {
		final Set<RT2DocUidType> res = new HashSet<>();
		channels.forEach((clientUid, p) -> {
			if (p.getValue().equals(channelId)) {
				res.add(p.getKey());
			}
		});
		return res;
	}

	public Map<RT2ChannelId, Long> getChannelsOffline() {
		final Map<RT2ChannelId, Long> offline = new HashMap<>();
		channelsOffline.forEach((channelId, time) -> {
			offline.put(channelId, time);
		});
		return offline;
	}

	public void addDocUidToChannel(RT2EnhDefaultWebSocket channel, RT2CliendUidType clientUid, RT2DocUidType docUid) {
		log.debug("addDocUidToChannel with channel {} and client {} and doc {}", channel.getId(), clientUid, docUid);
		final RT2ChannelId channelId = channel.getId();
		final Pair<RT2DocUidType, RT2ChannelId> docUidToChannelId = Pair.of(docUid, channelId);
		channels.put(clientUid, docUidToChannelId);
	}

	public void addClientToChannel(RT2EnhDefaultWebSocket channel, RT2CliendUidType clientUid) {
		if (!channels.containsKey(clientUid)) {
			log.debug("addClientToChannel with channel {} and client {}", channel.getId(), clientUid);
			final RT2ChannelId channelId = channel.getId();
			final Pair<RT2DocUidType, RT2ChannelId> docUidToChannelId = Pair.of(null, channelId);
			channels.put(clientUid, docUidToChannelId);
		}
	}

	public void removeChannel(RT2EnhDefaultWebSocket channel) {
		final RT2ChannelId channelId = channel.getId();

		log.debug("removeChannel {} from channels with ws {} - let the gc remove the channel mapping later", channel.getId(), channel);

		// ATTENTION: Don't remove the channel here as we have two different behavior scenarios from the
		// Grizzly. First it adds a new web socket and removes the previous one sometimes later (looks like gc). This
		// happens if the client stays offline a little bit longer (about 5 minutes).
		// The second scenario can happen if the client stays offline for about 2 minutes. Grizzly now removes the
		// original web socket and creates a new one for the channel. This would lead to a missing mapping
		// if we would remove it in this method. Therefore we let the gc do the job and remove the channel mapping
		// asynchronously to survive both scenarios.

		if (getClientsAssociatedToChannel(channelId).size() > 0) {
			channelsOffline.putIfAbsent(channelId, System.currentTimeMillis());
		}
	}

	public void removeClientUid(RT2CliendUidType clientUid) {
		log.debug("removeClient {} from channels", clientUid);
		RT2ChannelId channelId = channels.get(clientUid).getValue();		
		channels.remove(clientUid);
		channelsOffline.remove(channelId);
	}

	static int sendBack = 0;
	
 	public void addMessageToResponseQueue(RT2Message msg) {
 		log.debug("sendBack: {}", msg.getHeader());
		this.responseQueue.add(msg);
	}

    public void notifyUnavailability (Direction direction, RT2EnhDefaultWebSocket webSocket, long nTime) {
    	final Set<Pair<RT2CliendUidType, RT2DocUidType>> clientsAssocToSocket = getClientDocPairsAssociatedToChannel(webSocket.getId());
    	clientsAssocToSocket.forEach(p -> {
    		try {
    			log.debug("RT2: ... client '{}' notify unavailability for doc '{}'", p.getKey(), p.getValue());
                final RT2DocProxy aDocProxy = docProxyRegistry.getDocProxy(p.getKey(), p.getValue());
                if (aDocProxy != null) {
                    aDocProxy.notifyUnavailibility(direction, nTime);
                }
    		}
            catch (Throwable ex) {
            	com.openexchange.exception.ExceptionUtils.handleThrowable(ex);
                log.error("... client '"+p.getKey()+"' failed to notify unavailability for doc '"+p.getValue(), ex);
            }
        });
    }

	public Map<RT2CliendUidType, Pair<RT2DocUidType, RT2ChannelId>> getChannels() {
		return new HashMap<>(channels);
	}

	public int getResponseQueueSize() {
		return responseQueue.size();
	}

	class ResponseThread implements Runnable {

		@Override
		public void run() {
            LogProperties.putProperty(LogProperties.Name.RT2_BACKEND_PART, "ResponseThread");
            LogProperties.putProperty(LogProperties.Name.RT2_BACKEND_UID, nodeInfo.getNodeUUID());
			while (true) {
				if (docProxyResponseMetricService == null) {
					docProxyResponseMetricService = ServiceLookupRegistryService.getInstance().getService(DocProxyResponseMetricService.class);
				}
				RT2Message responseMsg = null;
				try {
					responseMsg = responseQueue.poll(1, TimeUnit.SECONDS);
					if (responseMsg != null) {
                        if (responseMsg.getDocUID() != null) {
                            LogProperties.putProperty(LogProperties.Name.RT2_DOC_UID, responseMsg.getDocUID().getValue());
                        }
                        if (responseMsg.getClientUID() != null) {
                            LogProperties.putProperty(LogProperties.Name.RT2_CLIENT_UID, responseMsg.getClientUID().getValue());
                        }
                        LogProperties.putProperty(LogProperties.Name.RT2_BACKEND_PART, "proxy_response_to_ws_client");
                        LogProperties.putProperty(LogProperties.Name.RT2_REQUEST_TYPE, responseMsg.getType().getValue());
                        log.debug("Sending response [{}] to ws client", responseMsg.getHeader());

                        final RT2CliendUidType clientUid = responseMsg.getClientUID();
                        final Pair<RT2DocUidType, RT2ChannelId> docUidToChannelId = channels.get(clientUid);
                        if (docUidToChannelId != null) {
                       		final String sResponse = RT2MessageFactory.toJSONString(responseMsg);
                       		log.debug("impl_sendResponse on channel {}", docUidToChannelId.getValue());
                       		final WebSocket webSocket = webSocketApplication.getWebSocketWithId(docUidToChannelId.getValue());
                       		if (webSocket != null) {
                       			log.debug("websocket instance used {} for sending msg", webSocket);
                       			if (ErrorCode.TOO_MANY_CONNECTIONS_ERROR.equals(responseMsg.getError().getValue())) {
                       				GrizzlyFuture<DataFrame> fut = webSocket.send(sResponse);
                       				fut.get(3, TimeUnit.SECONDS);
                       				webSocket.close(WebSocket.ABNORMAL_CLOSE);
                       			} else {                       				
                       				webSocket.send (sResponse);
                       			}
                       		} else {
                       			log.info("There is no response channel for ClientUid {}.", clientUid);
                       		}
                       		// remove client mapping immediately if client decided to leave
                       		if ((responseMsg.getType() == RT2MessageType.RESPONSE_LEAVE) ||
                       			(responseMsg.getType() == RT2MessageType.RESPONSE_EMERGENCY_LEAVE)) {
                       			removeClientUid(clientUid);
                       		}
                        } else {
                        	log.info("There is no response channel registered for ClientUid {}.", clientUid);
                        }
                        docProxyResponseMetricService.stopTimer(responseMsg.getMessageID());
 					}
				}
	            catch (InterruptedException ex) {
                    // ignored ... as it's expected ... somehow ;-)
                    Thread.currentThread().interrupt();
                }
				catch (Error ex) {
					if ((ex.getCause() != null) && (ex.getCause() instanceof MalformedInputException)) {
						if (responseMsg != null) {
							log.error("Message with MalformedInputException is(notEncoded): {}", RT2MessageFactory.toJSONString(responseMsg, /*encode*/false), ex);
							log.error("Message with MalformedInputException is(encoded): {}", RT2MessageFactory.toJSONString(responseMsg, /*encode*/true), ex);
						}
					}
                    com.openexchange.exception.ExceptionUtils.handleThrowable(ex);
                    log.error(ex.getMessage(), ex);
				}
                catch (Throwable ex) {
                    com.openexchange.exception.ExceptionUtils.handleThrowable(ex);
                    log.error(ex.getMessage(), ex);
                }
			}
		}
	}

	class RT2WSChannelExistsCheckThread implements Runnable {

		@Override
		public void run() {
            LogProperties.putProperty(LogProperties.Name.RT2_BACKEND_PART, "RT2WSChannelExistsCheckThread");
            LogProperties.putProperty(LogProperties.Name.RT2_BACKEND_UID, nodeInfo.getNodeUUID());
			checkIfWebsocketOfDocProxyExists();
		}

		private void checkIfWebsocketOfDocProxyExists() {
			final Set<RT2ChannelId> currentActiveWsIds = webSocketApplication.getCurrentActiveWebsocketIds();
			final List<RT2DocProxy> rt2DocProxies = docProxyRegistry.listAllDocProxies();
			rt2DocProxies.forEach(p -> {
				if (p.isOnline()) {
					final Pair<RT2DocUidType, RT2ChannelId> channelInfo = channels.get(p.getClientUID());
					final RT2ChannelId channelId = channelInfo.getValue();
					final RT2DocUidType docUid = channelInfo.getLeft();
					if (!currentActiveWsIds.contains(channelId)) {
						log.info("No channel with id {} found! DocProxy for docUid {} switch to offline.", channelId, docUid);
						p.goOffline();
						channelsOffline.putIfAbsent(channelId, System.currentTimeMillis());
					}
				}
			});
		}
	}

	class RT2WSChannelKeepAliveThread implements Runnable {

		@Override
		public void run() {
			// DOCS-1396: make sure that an exception doesn't lead to a cancellation
			// of this periodically called implementation (vital keep alive code which
			// triggers ping-messages).
			log.trace("Running keepAliveThread...");
            webSocketApplication.getCurrentActiveWebsocketIds().parallelStream().forEach(id -> {
                LogProperties.putProperty(LogProperties.Name.RT2_BACKEND_PART, "RT2WSChannelKeepAliveThread");
                LogProperties.putProperty(LogProperties.Name.RT2_BACKEND_UID, nodeInfo.getNodeUUID());
            	try {
					final RT2EnhDefaultWebSocket webSocket = webSocketApplication.getWebSocketWithId(id);
					if (webSocket != null) {
						webSocket.sendPing();
						if (webSocket.isAvaible()) {
							long now = System.currentTimeMillis();
							long unavail = webSocket.testIsAlive(now);
							if (unavail > 0l) {
								log.info("Notifiy unavilability for websocket with id {}", webSocket.getId());
								notifyUnavailability(Direction.INTERNAL, webSocket, unavail);
							}
						} else {
							log.debug("Websocket with id {} is unavailable", webSocket.getId());
						}
					}
            	} catch (Throwable t) {
            		com.openexchange.exception.ExceptionUtils.handleThrowable(t);
            		log.warn(t.getMessage(), t);
            	} finally {
            		LogProperties.removeLogProperties();
				}            	
            });
		}
	}
}
