package com.openexchange.office.rt2.proxy;

import java.util.HashMap;
import java.util.Map;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.hazelcast.core.HazelcastInstance;
import com.openexchange.office.rt2.cache.RT2DocInfo;
import com.openexchange.office.rt2.protocol.value.RT2DocUidType;

public class RT2DocInfoRegistry {

	private static final Logger log = LoggerFactory.getLogger(RT2DocInfoRegistry.class);

    //-------------------------------------------------------------------------
    private final HazelcastInstance hzInstance;    
	
    //-------------------------------------------------------------------------
    private final Map< RT2DocUidType, RT2DocInfo > docInfoRegistry = new HashMap<>();

    public RT2DocInfoRegistry(HazelcastInstance hzInstance) {
    	this.hzInstance = hzInstance;
    }
    
    public synchronized void clear() {
        docInfoRegistry.clear ();
    }

	public synchronized RT2DocInfo getDocInfoWithoutCreate(final RT2DocUidType sDocUID) {

		RT2DocInfo aInfo = docInfoRegistry.get(sDocUID);
		return aInfo;
	}
    
    
    //-------------------------------------------------------------------------
	/** @return an object knowing all data about a document (does not matter if it's open or not).
	 *
	 *	@param	sDocUID [IN]
	 *			the unique ID for the document where those data object is requested for.
	 */
	public synchronized RT2DocInfo getDocInfo (final RT2DocUidType sDocUID) {

		RT2DocInfo aInfo = docInfoRegistry.get(sDocUID);
		if (aInfo == null) {
			aInfo = new RT2DocInfo (hzInstance, sDocUID.getValue());
			docInfoRegistry.put (sDocUID, aInfo);
		}
		return aInfo;
	}

    //-------------------------------------------------------------------------
	/** @return an object knowing all data about a document (does not matter if it's open or not).
	 *          The object is NOT added to the internal map if the doc-uid is unknown. The object
	 *          is for information purposes only.
	 *
	 *	@param	sDocUID [IN]
	 *			the unique ID for the document where those data object is requested for.
	 */
	public synchronized RT2DocInfo peekDocInfo (final RT2DocUidType sDocUID) {
        RT2DocInfo aInfo = docInfoRegistry.get(sDocUID);

        if (aInfo == null) {
            aInfo = new RT2DocInfo (hzInstance, sDocUID.getValue());
        }

        return aInfo;
	}

	//-------------------------------------------------------------------------
	/** clean up all resources bound to that document.
	 *
	 *	@param	sDocUID [IN]
	 *			the unique ID for the document where all info object should be removed.
	 */
	public synchronized void freeDocInfos (final RT2DocUidType sDocUID) {
	    log.debug("freeDocInfos for docUid [{}]", sDocUID);
		docInfoRegistry.remove(sDocUID);
	}
    
}
