/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.rt2.jms;

import java.io.IOException;
import java.time.LocalDateTime;
import java.time.temporal.ChronoUnit;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;

import org.apache.activemq.ActiveMQConnectionFactory;
import org.apache.activemq.jms.pool.PooledConnectionFactory;
import org.apache.activemq.transport.TransportListener;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.jms.core.JmsTemplate;

import com.google.common.base.Throwables;
import com.openexchange.office.dcs.registry.config.DCSRegistryItem;
import com.openexchange.office.dcs.registry.database.DCSDatabase;
import com.openexchange.office.dcs.registry.database.DCSDatabaseFactory;
import com.openexchange.office.rt2.config.RT2ConfigItem;
import com.openexchange.office.rt2.core.RT2ThreadFactoryBuilder;
import com.openexchange.office.tools.database.DatabaseException;

public class JmsComponentFactory {

    public static final String DOCUMENT_QUEUE_NAME = "documents-queue-";

    public static final String DB_CLIENT_URL_KEY = "com.openexchange.dcs.client.database";

    private static final Logger log = LoggerFactory.getLogger(JmsComponentFactory.class);

    private LocalDateTime lastErrMsg = null;
    
    private PooledConnectionFactory pooledConnectionFactory = null;

    private final ScheduledExecutorService scheduledExecService = Executors.newScheduledThreadPool(1, new RT2ThreadFactoryBuilder("UrlInSyncWatchDogThread-%d").build());

    public void stop() {
    	scheduledExecService.shutdown();
    }

    public JmsTemplate getJmsTemplate(int ttl) {
        log.debug("Creating JMSTemplate...");
        JmsTemplate jmsTemplate = new JmsTemplate(getPooledConnectionFactory());
        jmsTemplate.setTimeToLive(ttl);
        jmsTemplate.setExplicitQosEnabled(ttl != 0);
        return jmsTemplate;
    }

    public PooledConnectionFactory getPooledConnectionFactory() {
        if (pooledConnectionFactory == null) {
            log.debug("Creating PooledConnectionFactory for connection to JMS Server...");
            pooledConnectionFactory = new PooledConnectionFactory();
            pooledConnectionFactory.setConnectionFactory(getJmsConnectionFactory());
        }
        return pooledConnectionFactory;
    }

    private EnhActiveMQSSLConnectionFactory getJmsConnectionFactory() {
        final EnhActiveMQSSLConnectionFactory jmsConnectionFactory = new EnhActiveMQSSLConnectionFactory();

        final String failOverBrokerUrl = getFailoverBrokerUrl();

        jmsConnectionFactory.setBrokerURL(failOverBrokerUrl);

        if (failOverBrokerUrl.contains("ssl:")) {
            final RT2ConfigItem configItem = RT2ConfigItem.get();

            try {
                // set keystore properties
                jmsConnectionFactory.setKeyStore(configItem.getRT2SSLKeystorePath());
                jmsConnectionFactory.setKeyStorePassword(configItem.getRT2SSLKeystorePassword());

                // set truststore properties
                jmsConnectionFactory.setTrustStore(configItem.getRT2SSLTrustStorePath());
                jmsConnectionFactory.setTrustStorePassword(configItem.getRT2SSLTrustStorePassword());
            } catch (Exception e) {
                log.error(Throwables.getRootCause(e).getMessage());
            }
        }

        jmsConnectionFactory.setTransportListener(new TransportListener() {

            private String lastErrorMsg = null;

            @Override
            public void transportResumed() {
                log.info("Resumed transport connection to dcs with broker url {}", jmsConnectionFactory.getBrokerURL());
            }

            @Override
            public void transportInterupted() {
            }

            @Override
            public void onException(IOException ex) {
                if ((lastErrorMsg == null) || !lastErrorMsg.equals(ex.getMessage())) {
                    log.info("Caught IOException: {}", ex);
                    lastErrorMsg = ex.getMessage();
                }
                jmsConnectionFactory.setBrokerURL(getFailoverBrokerUrl());
            }

            @Override
            public void onCommand(Object cmd) {
                log.debug("onCommand: {}", cmd);
            }
        });
        scheduledExecService.scheduleWithFixedDelay(new UrlInSyncWatchDogThread(jmsConnectionFactory), 1, 1, TimeUnit.MINUTES);
        return jmsConnectionFactory;
    }

    private String getFailoverBrokerUrl() {
    	RT2ConfigItem config = RT2ConfigItem.get();
        List<DCSRegistryItem> dcsRegistryItems = new ArrayList<>(); 
        try {
             final DCSDatabase dcsDatabase = DCSDatabaseFactory.getDcsDatabase(DB_CLIENT_URL_KEY);
             dcsRegistryItems = dcsDatabase.getRegisteredDCSItems();
        } catch (DatabaseException ex) {
        	if (lastErrMsg == null) {
        		lastErrMsg = LocalDateTime.now();
        		log.error("DatabaseException caught - cannot provide broker url, please check configuration & setup!");
        	} else {
        		if (ChronoUnit.MINUTES.between(LocalDateTime.now(), lastErrMsg) > 1) {
        			log.error("DatabaseException caught - cannot provide broker url, please check configuration & setup!");
        			lastErrMsg = null;
        		}
        	}                        
        }

        final StringBuilder res = new StringBuilder(256).append("failover:(");

        for (int i=0;i<dcsRegistryItems.size();++i) {
            final DCSRegistryItem item = dcsRegistryItems.get(i);
            final boolean useSSL = item.isUseSSL();

            res.append(useSSL ? "ssl" : "tcp").append("://").
                append(item.getInterface()).
                append(":").
                append(item.getJMSPort());

            if (useSSL) {
                res.append("?verifyHostName=" + (config.isRT2SSLVerifyHostname() ? "true" : "false"));
            }

            if ((i + 1) < dcsRegistryItems.size()) {
                res.append(",");
            }
        }
        res.append(")?maxReconnectAttempts=3");
        ActiveMQConnectionFactory amqFactory = (ActiveMQConnectionFactory)(pooledConnectionFactory.getConnectionFactory()); 
        if ((amqFactory == null) || !amqFactory.getBrokerURL().equals(res.toString())) {
        	log.info("Broker Url for the connection to DCS is: {}", res);
        } else {
        	log.debug("Broker Url for the connection to DCS is: {}", res);
        }
        return res.toString();
    }

    private class UrlInSyncWatchDogThread implements Runnable {

    	private ActiveMQConnectionFactory activeMqConnectionFactory;

    	public UrlInSyncWatchDogThread(ActiveMQConnectionFactory activeMqConnectionFactory) {
    		this.activeMqConnectionFactory = activeMqConnectionFactory;
		}

		@Override
		public void run() {
			String currentUrl = activeMqConnectionFactory.getBrokerURL();
			String dbBasedUrl = getFailoverBrokerUrl();
			if (!currentUrl.equals(dbBasedUrl)) {
				activeMqConnectionFactory.setBrokerURL(dbBasedUrl);
			}
		}
    }
}
