/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.rt2.core.sequence;

import java.lang.ref.WeakReference;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.stream.Collectors;
import java.util.stream.IntStream;

import org.json.JSONArray;
import org.json.JSONException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.openexchange.exception.ExceptionUtils;
import com.openexchange.log.LogProperties;
import com.openexchange.office.rt2.config.RT2ConfigItem;
import com.openexchange.office.rt2.core.RT2NodeInfoService;
import com.openexchange.office.rt2.exception.RT2ProtocolException;
import com.openexchange.office.rt2.protocol.RT2Message;
import com.openexchange.office.rt2.protocol.RT2MessageFactory;
import com.openexchange.office.rt2.protocol.RT2MessageGetSet;
import com.openexchange.office.rt2.protocol.RT2Protocol;
import com.openexchange.office.rt2.protocol.value.RT2CliendUidType;
import com.openexchange.office.rt2.protocol.value.RT2MessageType;
import com.openexchange.office.tools.common.TimeStampUtils;
import com.openexchange.office.tools.error.ErrorCode;
import com.openexchange.office.tools.osgi.ServiceLookupRegistryService;
import com.openexchange.timer.ScheduledTimerTask;
import com.openexchange.timer.TimerService;

public class MsgBackupAndACKProcessor implements Runnable {
    //-------------------------------------------------------------------------
    private static final Logger log = LoggerFactory.getLogger(MsgBackupAndACKProcessor.class);

    //-------------------------------------------------------------------------
    private static final String STR_ACK_RANGE_SEPARATOR = "-";

    private static final int TIME_FLUSH_ACK = 30000;
    private static final int TIMEOUT_FOR_RECONNECTED_CLIENT = 60000 * 2;
    private static final int TIMEOUT_FOR_BAD_CLIENT = 8 * TIME_FLUSH_ACK;
    private static final int TRESHHOLD_MAX_MSGS_WO_ACK = 2 * (TIME_FLUSH_ACK / 1000);

	//-------------------------------------------------------------------------
	private final WeakReference<SequenceDocProcessor> docProcessor;
	private final RT2NodeInfoService nodeInfoService;
	private final int timeoutValueBadClient;
	private final int threshholdValueMaxMsgs;
	private final int timeoutValueForReconnectClient;
	private Map<RT2CliendUidType, ClientState> clientStates = Collections.synchronizedMap(new HashMap<>());
	private ScheduledTimerTask timerSchedule = null;
	private AtomicBoolean shutdown = new AtomicBoolean(false);
	private Map<RT2CliendUidType, Set<String>> lastStoredMsgsForResend = new HashMap<>();

	//-------------------------------------------------------------------------
	public MsgBackupAndACKProcessor(final SequenceDocProcessor aDocProcessor) {
		this.docProcessor = new WeakReference<SequenceDocProcessor>(aDocProcessor);
		this.nodeInfoService = ServiceLookupRegistryService.getInstance().getOptionalService(RT2NodeInfoService.class);
		this.timeoutValueBadClient = RT2ConfigItem.get().getTimeoutForBadClient(TIMEOUT_FOR_BAD_CLIENT);
		this.threshholdValueMaxMsgs = RT2ConfigItem.get().getMaxMessagesCountWithoutAck(TRESHHOLD_MAX_MSGS_WO_ACK);
		this.timeoutValueForReconnectClient = RT2ConfigItem.get().getTimeoutBeforeHangupReconnectedClient(TIMEOUT_FOR_RECONNECTED_CLIENT); 		
	}

	//-------------------------------------------------------------------------	
	/**
	 * Use only for testing!!!
	 */
	public MsgBackupAndACKProcessor(final SequenceDocProcessor docProc, Map<RT2CliendUidType, Set<String>> lastStoredMsgsForResend) {
		this(docProc);
		this.lastStoredMsgsForResend = lastStoredMsgsForResend;
	}
	
	//-------------------------------------------------------------------------
	public void start() {
		if (timerSchedule == null) {
			int timeFlushAck = RT2ConfigItem.get().getAckResponsePeriod(TIME_FLUSH_ACK);
			timerSchedule = ServiceLookupRegistryService.getInstance().getService(TimerService.class).scheduleWithFixedDelay(this, timeFlushAck, timeFlushAck);
		}
	}

	//-------------------------------------------------------------------------
	public void shutdown() {
		if (!shutdown.getAndSet(true)) {
			if (timerSchedule != null)
				timerSchedule.cancel();
		}
	}

	//-------------------------------------------------------------------------
	public void switchState(RT2CliendUidType sClientUID, boolean online) {
		final ClientState aClientState = clientStates.get(sClientUID);
		if (aClientState != null)
			aClientState.setOnlineState(online);
		else
			log.error("RT2: Online state {} cannot be set for client, client unknown: {}", online, sClientUID);
	}

	//-------------------------------------------------------------------------
	public boolean isClientOnline(RT2CliendUidType sClientUID) {
		boolean result = false;
		final ClientState aClientState = clientStates.get(sClientUID);
		if (aClientState != null)
			result = aClientState.isOnline();
		return result;
	}

	//-------------------------------------------------------------------------
	public void addClient(RT2CliendUidType sClientUID) {
		clientStates.put(sClientUID, new ClientState(sClientUID, true));
	}
	
	//-------------------------------------------------------------------------
	public void removeClient(RT2CliendUidType sClientUID) {
		clientStates.remove(sClientUID);
	}

	//-------------------------------------------------------------------------
	public void updateClientState(RT2Message msg) {
		ClientState clientState = clientStates.get(msg.getClientUID());
		if (clientState != null) {
			switch (msg.getType()) {
				case ACK_ERROR:
				case ACK_SIMPLE:
				case NACK_SIMPLE:
				case REQUEST_ABORT_OPEN:
				case REQUEST_APP_ACTION:
				case REQUEST_APPLY_OPS:
				case REQUEST_CLOSE_DOC:
				case REQUEST_EDITRIGHTS:
				case REQUEST_EMERGENCY_LEAVE:
				case REQUEST_JOIN:
				case REQUEST_LEAVE:
				case REQUEST_OPEN_DOC:
				case REQUEST_RESET:
				case REQUEST_SAVE_DOC:
				case REQUEST_SYNC:
				case REQUEST_SYNC_STABLE:
				case REQUEST_UPDATE_SLIDE: clientState.updateLastMsgReceived();
										   break;
				default: log.debug("Message of type {} received which is an internal message", msg.getType());
			}
		}
	}
	
	//-------------------------------------------------------------------------
	public void backupMessage(final RT2Message aOutMsg) {
		final Integer nOutSeq = RT2MessageGetSet.getSeqNumber(aOutMsg);

		// don't backup messages without seq number!
		if (nOutSeq == null) {
			return;
		}

		log.trace("RT2: Queued message for client {} waiting for ACK: {}", aOutMsg.getClientUID(), nOutSeq);

		final ClientState aClientState = clientStates.get(aOutMsg.getClientUID());
		if (aClientState != null)
			aClientState.backupMessage(aOutMsg);
	}

    //-------------------------------------------------------------------------
    public void removeBackupMessagesForACKs(final RT2Message aAck) throws Exception {
        final ClientState aClientState = clientStates.get(aAck.getClientUID());

        if (aClientState != null) {
        	try {
                final Object ackBody = aAck.getBody().get(RT2Protocol.BODYPART_ACKS);
                // We now support two different syntaxes for ACKs
                if (ackBody instanceof JSONArray) {
                    // 1.) a json arrray with seq-nrs to be ACK: [1, 2, 3, 4, 5, ...]
                    final JSONArray jAckList = (JSONArray)ackBody;
                    log.trace("RT2: Received ACKs : {}" + jAckList.toString() + " for client " + aAck.getClientUID());
                    aClientState.removeStoredMessages(jAckList.asList());
                } else if (ackBody instanceof String) {
                    // 2.) a simple string with a range to be ACK: "1-1000", where both
                    //     values (start & end) are inclusive, means message with this seq-nr
                    //     must be removed. If end < start the range will be ignored, start == end
                    //     means that just one seq-nr is removed
                    try {
                        final String[] parts = ((String)ackBody).split(STR_ACK_RANGE_SEPARATOR);
                        if (parts.length == 2) {
                            int start = Integer.parseInt(parts[0]);
                            int end = Integer.parseInt(parts[1]);
                            aClientState.removeStoredMessages(start, end);
                        } else {
                            throw new RT2ProtocolException("Invalid syntax for ACK message detected: " + ackBody);
                        }
                    } catch (NumberFormatException e) {
                        throw new RT2ProtocolException("Invalid syntax for ACK message detected: " + ackBody, e);
                    }
                } else {
                    throw new RT2ProtocolException("Invalid body type for ACK message detected: " + aAck.getBodyString());
                }
            } catch (JSONException e) {
                throw new RT2ProtocolException("Invalid body format for ACK message detected: " + aAck.getBodyString(), e);
            }
        } else {
            log.debug("RT2: Unknown client UID "+aAck.getClientUID()+" used by ACK message: " + aAck.toString());
        }
    }

	//-------------------------------------------------------------------------
	public List<RT2Message> getMessagesForResent(final RT2CliendUidType sClientUID, final List<Object> aNackList) throws Exception {
		List<RT2Message> aResentList;

		log.trace("RT2: Looking for messages request by client: {} with NACK with seq-nrs: {}", sClientUID, aNackList.toString());

		final ClientState aClientState = clientStates.get(sClientUID);
		if (aClientState != null) {
			aResentList = aClientState.getRequestedMessages(aNackList);
		} else {
			log.debug("RT2: Unknown client UID "+sClientUID+" used by NACK message.");
			aResentList = new ArrayList<RT2Message>();
		}
		return aResentList;
	}

 	//-------------------------------------------------------------------------
	public void addReceivedSeqNr(final RT2CliendUidType clientUID, int start, int end) throws Exception {
		log.trace("RT2: Add received seq-nrs {} - {} for client: {}", start, end, clientUID);

		final ClientState aClientState = clientStates.get(clientUID);
		if (aClientState != null) {
			aClientState.addReceivedSeqNrCol(IntStream.range(start, end+1).boxed().collect(Collectors.toSet()));
		}
		else
			log.debug("RT2: Unknown client UID "+clientUID+" used addReceivedSeqNr().");
	}

	//-------------------------------------------------------------------------
	public Map<RT2CliendUidType, Integer> getClientStatesMsgBackupSize() {
		final Map<RT2CliendUidType, Integer> res = new HashMap<>();
		synchronized (clientStates) {
			clientStates.forEach((k,v) -> res.put(k, v.getMsgBackupSize()));
		}
		return res;
	}
		
	//-------------------------------------------------------------------------	
	public Map<RT2CliendUidType, Set<String>> getLastStoredMsgsForResend() {
		return new HashMap<>(lastStoredMsgsForResend);
	}

	//-------------------------------------------------------------------------
	@Override
	public void run() {
		try {
			if (docProcessor.get() == null) {
				shutdown();
				return;
			}
	        LogProperties.putProperty(LogProperties.Name.RT2_BACKEND_PART, "MsgBackupAndACKProcessor");
	        if (nodeInfoService != null) {
	        	LogProperties.putProperty(LogProperties.Name.RT2_BACKEND_UID, nodeInfoService.getNodeUUID());
	        }
            LogProperties.putProperty(LogProperties.Name.RT2_DOC_UID, docProcessor.get().getDocUID().getValue());          

			final Map<RT2CliendUidType, Set<Integer>> clientsAckList = new HashMap<>();
			final Map<RT2CliendUidType, ClientAckState> clientsAckStateMap = new HashMap<>();
			collectionClientInfo(clientsAckList, clientsAckStateMap);

			if (!this.shutdown.get()) {
				final SequenceDocProcessor aSeqDocProcessor = docProcessor.get();
				if (aSeqDocProcessor != null) {
					sendAcksToClients(clientsAckList, aSeqDocProcessor);
					checkBadClientBehaviour(clientsAckStateMap, aSeqDocProcessor);
				}
			}
		}
		catch (Throwable t) {
			ExceptionUtils.handleThrowable(t);
			log.error("RT2: Throwable received while processing pending ACKs!", t);
		}
	}

	//-------------------------------------------------------------------------	
	private void checkBadClientBehaviour(final Map<RT2CliendUidType, ClientAckState> clientsAckStateMap, final SequenceDocProcessor aSeqDocProcessor) {
		// protection: check for clients which don't send acks although they send msgs -
		// these client must be hang-up to protect us from oom situations (we have to
		// store msgs for resend until we receive an ack) 
		for (final RT2CliendUidType clientUID : clientsAckStateMap.keySet())
		{
			final ClientAckState clientAckState = clientsAckStateMap.get(clientUID);
			if (isBadClientACKBehaviour(clientAckState)) {
				log.warn("RT2: Detected a client that sends messages without correct ack handling - hang-up this client {}", clientUID);
				aSeqDocProcessor.hangUpBadClient(clientUID, ErrorCode.GENERAL_MALICIOUS_ACK_HANDLING_ERROR);
			}
		}
	}

	//-------------------------------------------------------------------------	
	private void sendAcksToClients(final Map<RT2CliendUidType, Set<Integer>> clientsAckList, final SequenceDocProcessor aSeqDocProcessor) {
		// check if we need to send the next lazy ACKs for a client
		// send the lazy ack lists to the clients
		for (final RT2CliendUidType sClientUID : clientsAckList.keySet()) {
			final Set<Integer> aClientAckList = clientsAckList.get(sClientUID);
			try {
				final RT2Message aAckResponse = RT2MessageFactory.newMessage(RT2MessageType.ACK_SIMPLE, sClientUID, aSeqDocProcessor.getDocUID());
				aAckResponse.getBody().put(RT2Protocol.BODYPART_ACKS, new JSONArray(aClientAckList));
				aSeqDocProcessor.sendMessageWOSeqNrTo(sClientUID, aAckResponse);
			}
			catch (JSONException t) {				
				ExceptionUtils.handleThrowable(t);
				log.error("RT2: JSONException received while sending ACKs - client state unknown!", t);
				aSeqDocProcessor.sendErrorResponseToClient(sClientUID, null, null, t);
			}
		}
	}

	//-------------------------------------------------------------------------	
	private void collectionClientInfo(final Map<RT2CliendUidType, Set<Integer>> clientsAckList,
			final Map<RT2CliendUidType, ClientAckState> clientsAckStateMap) {
		if (!this.shutdown.get()) {
			synchronized (clientStates) {
				for (final RT2CliendUidType sClientUID : clientStates.keySet()) {
					final ClientState  aClientState = clientStates.get(sClientUID);
					final Set<Integer> aAckSet      = aClientState.getAckSet();

					if (!aAckSet.isEmpty()) {
						clientsAckList.put(sClientUID, aAckSet);
					}
					final ClientAckState clientAckState = 
							new ClientAckState(sClientUID, aClientState.getLastMsgReceived(), aClientState.getBackupMsgs());
					clientsAckStateMap.put(sClientUID, clientAckState);
				}
			}
		}
	}

	//-------------------------------------------------------------------------
	private boolean isBadClientACKBehaviour(ClientAckState clientAckState) {
		final long now = System.currentTimeMillis();
		final long lastSeqNrMsg = clientAckState.getTimeStampLastMsgReceived();
		final int numOfMsgsForResend = clientAckState.getMsgsStoredForResend().size();
		long timeDiff = TimeStampUtils.timeDiff(lastSeqNrMsg, now); 
		
		boolean timeoutBadClient = timeDiff > timeoutValueBadClient;
		boolean threshholdMaxMsgs = numOfMsgsForResend > threshholdValueMaxMsgs;
		boolean timeoutForReconnectedClient = timeDiff > timeoutValueForReconnectClient;
		log.debug("timeoutBadClient: {}, thresholdMaxMsgs: {}, timeoutForReconnectClient: {}", timeoutBadClient, threshholdMaxMsgs, timeoutForReconnectedClient);
		boolean res =  isClientOnline(clientAckState.getClientUid()) &&
		  	   		   ((timeoutBadClient && (numOfMsgsForResend > 0)) ||
		  	   		   (threshholdMaxMsgs && timeoutForReconnectedClient));
		if (res) {
			Set<String> newMessages = clientAckState.getMsgsStoredForResend().stream().map(c -> c.getMessageID().getValue()).collect(Collectors.toSet());
			if (lastStoredMsgsForResend.containsKey(clientAckState.getClientUid())) {
				Set<String> oldMessages = lastStoredMsgsForResend.get(clientAckState.getClientUid());
				int sizeOldMessages = oldMessages.size();
				oldMessages.retainAll(newMessages);
				if (sizeOldMessages != oldMessages.size()) {
					lastStoredMsgsForResend.put(clientAckState.getClientUid(), clientAckState.getMsgsStoredForResend().stream().map(c -> c.getMessageID().getValue()).collect(Collectors.toSet()));
					res = false;					
				}
			} else {
				lastStoredMsgsForResend.put(clientAckState.getClientUid(), clientAckState.getMsgsStoredForResend().stream().map(c -> c.getMessageID().getValue()).collect(Collectors.toSet()));
				res = false;
			}
		} else {
			lastStoredMsgsForResend.remove(clientAckState.getClientUid());
		}
		if (res) {
			lastStoredMsgsForResend.remove(clientAckState.getClientUid());
			log.warn("Bad client ack behaviour found! TimeDiff: {}, messages to resend: {}", timeDiff, numOfMsgsForResend);
		}
		return res;
	}
	
	
}
