package com.openexchange.office.rt2.core.sequence;

import java.lang.ref.WeakReference;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.stream.Collectors;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.openexchange.log.LogProperties;
import com.openexchange.office.rt2.core.RT2NodeInfoService;
import com.openexchange.office.rt2.protocol.value.RT2CliendUidType;
import com.openexchange.office.tools.osgi.ServiceLookupRegistry;
import com.openexchange.office.tools.osgi.ServiceLookupRegistryService;
import com.openexchange.timer.ScheduledTimerTask;
import com.openexchange.timer.TimerService;

public class ClientLeftRemoveThread implements Runnable {
	
	private static final Logger log = LoggerFactory.getLogger(ClientLeftRemoveThread.class);
	
	//-------------------------------------------------------------------------	
	private static final long TIME_CLEANUP_CLIENTS_LEFT = 300000;
	
	//-------------------------------------------------------------------------
	private static final long FREQ_CLEANUP_CLIENTS_LEFT = 60000;

	//-------------------------------------------------------------------------	
	final private WeakReference<SequenceDocProcessor> docProcessor;

	//-------------------------------------------------------------------------
	private final Map<RT2CliendUidType, Long> clientsLeft = new HashMap<>();

	//-------------------------------------------------------------------------
	private final ScheduledTimerTask timerScheduleClientsLeft;
	
	//-------------------------------------------------------------------------
	private final AtomicBoolean shutdown = new AtomicBoolean(false);
	
	public ClientLeftRemoveThread(SequenceDocProcessor docProcessor) {
		this.docProcessor = new WeakReference<SequenceDocProcessor>(docProcessor);
		final TimerService aTimerService = ServiceLookupRegistryService.getInstance().getService(TimerService.class);
		timerScheduleClientsLeft = aTimerService.scheduleAtFixedRate(this, FREQ_CLEANUP_CLIENTS_LEFT, FREQ_CLEANUP_CLIENTS_LEFT);		
	}
	
	//-------------------------------------------------------------------------	
	public void addClientLeft(RT2CliendUidType clientUid) {
		synchronized (clientsLeft) {
			clientsLeft.put(clientUid, System.currentTimeMillis());
		}
	}
	
	//-------------------------------------------------------------------------
	public long getClientLeft(RT2CliendUidType clientUid) {
		synchronized (clientsLeft) {
			if (clientsLeft.containsKey(clientUid)) {
				return clientsLeft.get(clientUid);
			}
			return -1l;
		}
	}
	
	//-------------------------------------------------------------------------	
	public void shutdown() {
		if (shutdown.compareAndSet(false, true)) {
			timerScheduleClientsLeft.cancel();
		}
	}
	
	//-------------------------------------------------------------------------	
	@Override
	public void run() {
		checkIfClientHasToBeRemoved();
		checkForGapsInMessages();
	}

	private void checkForGapsInMessages() {
		synchronized (clientsLeft) {			
			SequenceDocProcessor seqDocProc = docProcessor.get();
			if (seqDocProc != null) {
				for (RT2CliendUidType clientUid : seqDocProc.getClientsSeqQueues().keySet()) {
					try {					
						ClientSequenceQueue clientSequenceQueue = docProcessor.get().getClientSequenceQueueForClient(clientUid, false);
						if (clientSequenceQueue != null) {
							if (clientSequenceQueue.hasQueuedInMessages()) {
								seqDocProc.checkForAndRequestMissingMessagesViaNack(clientUid);
							}
						}
					} catch (Exception ex) {
						log.warn(ex.getMessage());
					}
				}
			}
		}
	}
	
	private void checkIfClientHasToBeRemoved() {
		try {
			final RT2NodeInfoService rt2NodeInfo = ServiceLookupRegistry.get().getService(RT2NodeInfoService.class);
	        LogProperties.putProperty(LogProperties.Name.RT2_BACKEND_PART, "ClientLeftRemoveThread");
	        LogProperties.putProperty(LogProperties.Name.RT2_BACKEND_UID, rt2NodeInfo.getNodeUUID());        												
	        
			if (docProcessor.get() == null) {
				shutdown();
				clientsLeft.clear();
				return;
			}
			final long now = System.currentTimeMillis();

			Set<RT2CliendUidType> clientsToCleanup = null;
			synchronized (clientsLeft) {
				if (clientsLeft.size() > 0) {
					clientsToCleanup = clientsLeft.keySet()
							.stream()
							.filter(k -> { return needToCleanupClientUID(clientsLeft.get(k), now); })
							.collect(Collectors.toSet());
					clientsToCleanup.forEach(k -> { clientsLeft.remove(k); });
				}
        	}

			if ((clientsToCleanup != null) && (!clientsToCleanup.isEmpty())) {
				log.debug("RT2: removing {} left client uids after timeout", clientsToCleanup.size());
			}
		} catch (Exception e) {
            log.warn("RT2: Exception caught while trying to lazy remove stored left client uids", e);
		}
	}

	//-------------------------------------------------------------------------
	private static boolean needToCleanupClientUID(final Long time, long now) {
		return (time != null) ? ((now - time) >= TIME_CLEANUP_CLIENTS_LEFT) : false;
	}
}
