/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.rt2.core.doc;

import java.util.HashSet;
import java.util.Iterator;
import java.util.Set;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.openexchange.exception.ExceptionUtils;

/**
 * Implementation class of the global RT2 service to register/unregister for
 * document events like created/disposed.
 * Don't create instances of this class by yourself. The functions are provided
 * via services and must be retrieved from your service manager.
 *
 * @author Carsten Driesner
 * @since 7.10.0
 *
 */
public class RT2DocStateNotifier 
{
    //-------------------------------------------------------------------------
    public static final String DOCSTATE_EVENT_CREATED  = "created";
    public static final String DOCSTATE_EVENT_DISPOSED = "disposed";

    private static final Logger log = LoggerFactory.getLogger(RT2DocStateNotifier.class);

    //-------------------------------------------------------------------------
    private static class Event
    {
        public final String m_sEventName;
        public final RT2DocProcessor m_aDocProcInstance;

        public Event(String sEventName, final RT2DocProcessor aDocProc)
        {
            m_sEventName       = sEventName;
            m_aDocProcInstance = aDocProc;
        }
    }

    //-------------------------------------------------------------------------
    private final Set<IDocNotificationHandler> m_aHandlerSet = new HashSet<>();

    //-------------------------------------------------------------------------
    public RT2DocStateNotifier(final IDocNotificationHandler ... aHandlers)
    {
    	for (IDocNotificationHandler handler : aHandlers) {
    		m_aHandlerSet.add(handler);
    	}
        log.debug("RT2DocStateNotifier: RT2DocStateNotifier instance created");
    }

    //-------------------------------------------------------------------------
    public void notifyEvent(final String sEventName, final RT2DocProcessor aDocProc)
    {
        final Event aEvent = new Event(sEventName, aDocProc);
        notifyRegisteredHandlers(aEvent);
    }

    //-------------------------------------------------------------------------
    private void notifyRegisteredHandlers(final Event aEvent)
    {
        if (aEvent != null)
        {
        	if (m_aHandlerSet.isEmpty()) {
        		return;
        	}
            boolean bCancelNotification = false;
            final Iterator<IDocNotificationHandler> aIter = m_aHandlerSet.iterator();
            while (!bCancelNotification && aIter.hasNext())
            {
                try
                {
                    final IDocNotificationHandler aHandler = aIter.next();
                    switch (aEvent.m_sEventName)
                    {
                        case DOCSTATE_EVENT_CREATED: aHandler.docProcessorCreated(aEvent.m_aDocProcInstance); break;
                        case DOCSTATE_EVENT_DISPOSED: aHandler.docProcessorDisposed(aEvent.m_aDocProcInstance); break;
                        default:
                            log.warn("RT2DocNotifier detected a unknown notification: " + aEvent.m_sEventName + ". Event will be discarded!");
                            bCancelNotification = true;
                            break;
                    }
                }
                catch (final Throwable t)
                {
                    ExceptionUtils.handleThrowable(t);
                    log.error("RT2DocStateNotifier: received exception while notification handler has been called!", t);
                }
            }
        }
    }
}
