/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.rt2.core.doc;

import java.lang.ref.WeakReference;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.apache.commons.lang3.Validate;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.openexchange.config.ConfigurationService;
import com.openexchange.exception.ExceptionUtils;
import com.openexchange.office.tools.common.EasyStopWatch;
import com.openexchange.office.tools.doc.ApplicationType;
import com.openexchange.office.tools.error.ErrorCode;
import com.openexchange.office.tools.monitoring.BackgroundSaveEvent;
import com.openexchange.office.tools.monitoring.BackgroundSaveEventType;
import com.openexchange.office.tools.monitoring.Statistics;
import com.openexchange.office.tools.osgi.ServiceLookupRegistry;

/**
 * A class which controls the background save of modified documents according to
 * the pre-defined rule set. The background processing is done using the OX
 * TimerService and tries to have a minimal impact of overall system performance.
 * It checks the runtime of save processing and delays it if necessary.
 *
 * @author <a href="mailto:carsten.driesner@open-xchange.com">Carsten Driesner</a>
 * @since v7.10.0
 *
 */
public class RT2BackgroundDocSaveManager implements IDocNotificationHandler, IProcessDocCollection {
    /**
     * Internal class to encapsulate the timer based background processing of the
     * document processors.
     *
     * {@link ProcessingRunnable}
     *
     * @author <a href="mailto:carsten.driesner@open-xchange.com">Carsten Driesner</a>
     * @since v7.8.4
     */
    private static class ProcessingRunnable implements Runnable {
        private final WeakReference<IProcessDocCollection> m_handler;

        public ProcessingRunnable(final IProcessDocCollection handler) {
            super();

            m_handler = new WeakReference<>(handler);
        }

        @Override
        public void run() {
            IProcessDocCollection handler = m_handler.get();
            if (null != handler) {
                handler.processDocCollection();
            }
        }
    }

    //-------------------------------------------------------------------------

    private static final Logger log = LoggerFactory.getLogger(RT2BackgroundDocSaveManager.class);
    private static final String BACKGROUND_SAVE_THREAD_NAME = "com.openexchange.office.rt2.core.doc.RT2BackgroundDocSaveManager";

    private static final int MIN_OPERATIONS_FOR_FAILSAFESAVE = 50;
    private static final int MIN_OPERATIONS_FOR_FAILSAFESAVE_FOR_OT = 200;
    private static final int MIN_TIME_FOR_MANY_OPS_FAILSAFESAVE = 60 * 1000; // time in milliseconds
    private static final int MAX_TIME_FOR_MODIFIED_FAILSAFESAVE = 15 * 60 * 1000; // time in milliseconds

    //-------------------------------------------------------------------------
    private Thread                                         m_backgroundThread     = null;
    private Map<String, WeakReference<IBackgroundSavable>> m_savableDocCollection = new HashMap<>();
    private boolean                                        m_clearInsteadOfSave   = false;

    //-------------------------------------------------------------------------
    public RT2BackgroundDocSaveManager() {
        final ConfigurationService cfgService = ServiceLookupRegistry.get().getService(ConfigurationService.class);

        int minOperationsForFailSafeSaveText = MIN_OPERATIONS_FOR_FAILSAFESAVE;
        int minOperationsForFailSafeSaveSpreadsheet = MIN_OPERATIONS_FOR_FAILSAFESAVE;
        int minOperationsForFailSafeSavePresentation = MIN_OPERATIONS_FOR_FAILSAFESAVE;

        if (null != cfgService) {
            // Retrieve configuration to support reset operation queue instead of saving the document
            // ATTENTION: This is ONLY for sizing tests, otherwise data can/will be lost
            m_clearInsteadOfSave = cfgService.getBoolProperty("io.ox/office//module/fssResetOperationQueue", false);
            if (m_clearInsteadOfSave) {
                log.warn("RT2: Warning - fail safe save is configured to clear the operation queue of the documents - this is for sizing tests only!");
            }
            minOperationsForFailSafeSaveText = ConfigOT.isOTEnabled(cfgService, ApplicationType.APP_TEXT_STRING) ? MIN_OPERATIONS_FOR_FAILSAFESAVE_FOR_OT : MIN_OPERATIONS_FOR_FAILSAFESAVE;
            minOperationsForFailSafeSaveSpreadsheet = ConfigOT.isOTEnabled(cfgService, ApplicationType.APP_SPREADSHEET_STRING) ? MIN_OPERATIONS_FOR_FAILSAFESAVE_FOR_OT : MIN_OPERATIONS_FOR_FAILSAFESAVE;
            minOperationsForFailSafeSavePresentation = ConfigOT.isOTEnabled(cfgService, ApplicationType.APP_PRESENTATION_STRING) ? MIN_OPERATIONS_FOR_FAILSAFESAVE_FOR_OT : MIN_OPERATIONS_FOR_FAILSAFESAVE;
        }

        final Statistics statistics = Statistics.get();
        if (null != statistics) {
            statistics.setBackgroundSaveMinNumOfMsgForFasterSaveText(minOperationsForFailSafeSaveText);
            statistics.setBackgroundSaveMinNumOfMsgForFasterSaveSpreadsheet(minOperationsForFailSafeSaveSpreadsheet);
            statistics.setBackgroundSaveMinNumOfMsgForFasterSavePresentation(minOperationsForFailSafeSavePresentation);
            statistics.setBackgroundSaveMinTimeFasterSave(MIN_TIME_FOR_MANY_OPS_FAILSAFESAVE);
            statistics.setBackgroundSaveMaxTimeForSave(MAX_TIME_FOR_MODIFIED_FAILSAFESAVE);
        }
    }

    //-------------------------------------------------------------------------
    /**
     * Starts the document processing on a timer-based scheme.
     */
    public synchronized void start() {
        if (m_backgroundThread == null) {
            m_backgroundThread = new Thread(new ProcessingRunnable(this));
            m_backgroundThread.setPriority(Thread.NORM_PRIORITY - 1);
            m_backgroundThread.setName(BACKGROUND_SAVE_THREAD_NAME);
            m_backgroundThread.start();
        }
    }

    //-------------------------------------------------------------------------
    /**
     * Stops a running document processing timer.
     */
    public synchronized void stop() {
        if (m_backgroundThread != null) {
            try {
                m_backgroundThread.interrupt();
                m_backgroundThread.join();
                m_backgroundThread = null;
            }
            catch (final InterruptedException e) {
                log.debug("RT2: RT2BackgroundDocSaveManager InterruptedException caught preparing shutdown of the background saving thread", e);
                Thread.currentThread().interrupt();
            }
        }
    }

    //-------------------------------------------------------------------------
	@Override
	public void docProcessorCreated(final RT2DocProcessor aCreatedInstance) {
		Validate.notNull(aCreatedInstance);

		if (aCreatedInstance instanceof IBackgroundSavable) {
			synchronized(m_savableDocCollection) {
				final IBackgroundSavable iSavableDoc             = (IBackgroundSavable)aCreatedInstance;
				final WeakReference<IBackgroundSavable> aWeakRef = new WeakReference<>(iSavableDoc);

				final WeakReference<IBackgroundSavable> aOldWeakRef = m_savableDocCollection.put(aCreatedInstance.getDocUID().getValue(), aWeakRef);
				if (aOldWeakRef != null) {
					log.debug("RT2: RT2BackgroundDocSaveManager.docProcessorCreated: There is a previous reference to the DocUID {} stored", aCreatedInstance.getDocUID());
				}
			}
		}
	}

    //-------------------------------------------------------------------------
	@Override
	public void docProcessorDisposed(final RT2DocProcessor aDisposedInstance) {
		Validate.notNull(aDisposedInstance);

		if (aDisposedInstance instanceof IBackgroundSavable) {
			synchronized(m_savableDocCollection) {
				final WeakReference<IBackgroundSavable> aOldWeakRef = m_savableDocCollection.remove(aDisposedInstance.getDocUID().getValue());
				if (aOldWeakRef == null) {
					log.debug("RT2: RT2BackgroundDocSaveManager.docProcessorDisposed: Unknown DocUID {}!", aDisposedInstance.getDocUID());
				}
			}
		}
	}

    //-------------------------------------------------------------------------
	@Override
	public void processDocCollection() {
		boolean interrupted = false;

		while (!interrupted) {
	        try {
	            final EasyStopWatch stopWatch = new EasyStopWatch();
	            final List<WeakReference<IBackgroundSavable>> aAllSavableDocList = new ArrayList<>();

	            synchronized (m_savableDocCollection) {
	                if (!m_savableDocCollection.isEmpty())
	                    aAllSavableDocList.addAll(m_savableDocCollection.values());
	            }

	            // filter document list to find only documents that must be saved now
	            final List<WeakReference<IBackgroundSavable>>     aSavableDocList = getDocumentsToBeSaved(aAllSavableDocList);
	            final Iterator<WeakReference<IBackgroundSavable>> iter            = aSavableDocList.iterator();
	            final Date                                        aNow            = new Date();

	            if (iter.hasNext()) {
	            	try {
		                int numOfDocumentsSaved = 0;

		                // process documents
		                stopWatch.start();
		                while (iter.hasNext()) {
		                    final IBackgroundSavable aSavableDoc = iter.next().get();

		                    BackgroundSaveReason eReason = BackgroundSaveReason.NO_REASON;
		                    if ((null != aSavableDoc) && ((eReason = mustBeProcessed(aSavableDoc, aNow)) != BackgroundSaveReason.NO_REASON)) {
		                        if (m_clearInsteadOfSave)
		                            eReason = BackgroundSaveReason.DELETE_PENDING_OPERATIONS;

		                        // save document and suppress any exception
		                        boolean bSuccess = backgroundSaveDocument(aSavableDoc, eReason);
		                        if (bSuccess) {
		                            // document is processed, if contEnumeration is true
		                            aSavableDoc.setProcessedState(true);
		                            numOfDocumentsSaved++;
		                        }
		                    }

		                    // detect interrupted state to bail out early
		                    if (Thread.interrupted())
		                    	throw new InterruptedException();
		                }

		                final long elapsedTime =  stopWatch.stop();

		                if (numOfDocumentsSaved > 0) {
			                final long timePerDoc  = (elapsedTime / Math.max(numOfDocumentsSaved, 1));

			                Statistics.handleBackgroundSaveDocEvent(new BackgroundSaveEvent(BackgroundSaveEventType.FSS_LAST_CYCLE_DOC_COUNT, numOfDocumentsSaved));
			                Statistics.handleBackgroundSaveDocEvent(new BackgroundSaveEvent(BackgroundSaveEventType.FSS_LAST_CYCLE_TIME, elapsedTime));
			                Statistics.handleBackgroundSaveDocEvent(new BackgroundSaveEvent(BackgroundSaveEventType.FSS_LAST_CYCLE_TIME_PER_DOC, timePerDoc));

			                log.debug("RT2: FSS Pool contains {} document(s) to be saved - {} document(s) were saved in {} ms, time per document = {}", aSavableDocList.size(), numOfDocumentsSaved, elapsedTime, timePerDoc);
		                } else {
			            	resetBackgroundSaveMetrics();
		                }

		                if (!iter.hasNext()) {
		                    // reset processed state to enable processing again
		                    final Iterator<WeakReference<IBackgroundSavable>> resetIter = aAllSavableDocList.iterator();
		                    while (resetIter.hasNext()) {
		                        final IBackgroundSavable aSavableDoc = resetIter.next().get();
		                        if (null != aSavableDoc) {
		                            aSavableDoc.setProcessedState(false);
		                        }
		                    }
		                }
	            	} catch (InterruptedException e) {
	            		Thread.currentThread().interrupt();
	            		interrupted = true;
	            		throw new InterruptedException();
	            	} catch (final Exception e) {
	    	            log.error("RT2: Exception caught while FSS processes connections", e);
	    	        }
	            }
	            else {
	            	resetBackgroundSaveMetrics();
	            }

	            Thread.sleep(1000);
	        } catch (final InterruptedException e) {
        		Thread.currentThread().interrupt();
        		interrupted = true;
	        }
		}
	}

    //-------------------------------------------------------------------------
    private void resetBackgroundSaveMetrics() {
        Statistics.handleBackgroundSaveDocEvent(new BackgroundSaveEvent(BackgroundSaveEventType.FSS_LAST_CYCLE_DOC_COUNT, 0));
        Statistics.handleBackgroundSaveDocEvent(new BackgroundSaveEvent(BackgroundSaveEventType.FSS_LAST_CYCLE_TIME, 0));
        Statistics.handleBackgroundSaveDocEvent(new BackgroundSaveEvent(BackgroundSaveEventType.FSS_LAST_CYCLE_TIME_PER_DOC_RESET));
    }

    //-------------------------------------------------------------------------
    private boolean backgroundSaveDocument(final IBackgroundSavable aSavableDoc, BackgroundSaveReason eReason) {
        try {
            aSavableDoc.save(eReason, false);
            return true;
        }
        catch (final Throwable t) {
            ExceptionUtils.handleThrowable(t);
            log.error("RT2: Exception caught while trying to background save document with DocUID: " + aSavableDoc.getUniqueID(), t);
            return false;
        }
    }

    //-------------------------------------------------------------------------
    private List<WeakReference<IBackgroundSavable>> getDocumentsToBeSaved(final List<WeakReference<IBackgroundSavable>> aSavableDocList) {
        List<WeakReference<IBackgroundSavable>> aFilteredList = aSavableDocList;

        if ((null != aSavableDocList) && (!aSavableDocList.isEmpty())) {
            final ArrayList<WeakReference<IBackgroundSavable>> aFilteredCollection = new ArrayList<>();
            final Iterator<WeakReference<IBackgroundSavable>> iter = aSavableDocList.iterator();

            while (iter.hasNext()) {
                final WeakReference<IBackgroundSavable> aSavableDocRef = iter.next();
                final IBackgroundSavable aSavableDoc = aSavableDocRef.get();

                if ((aSavableDoc != null) && (aSavableDoc.isModified()))
                    aFilteredCollection.add(aSavableDocRef);
            }

            aFilteredList = aFilteredCollection;
        }

        return aFilteredList;
    }

    //-------------------------------------------------------------------------
    /**
     * Determines, if a connection must be processed or not by fail safe save.
     *
     * @param aSavableDoc A document to be checked for background save processing.
     * @param now The current time
     * return the reason for the save that must be processed or null is running up?
     *           when there's no need to process saving. This parameter may be null
     */
    private BackgroundSaveReason mustBeProcessed(final IBackgroundSavable aSavableDoc, final Date now) {
        BackgroundSaveReason eReason = BackgroundSaveReason.NO_REASON;

        if  (shouldBeSaved(aSavableDoc))
        {
            final long diff = getTimeDifference(aSavableDoc.getLastSaveTimeStamp(), now.getTime());
            final long nOps = aSavableDoc.getPendingOperationsCount();

            BackgroundSaveReason saveReason = BackgroundSaveReason.NO_REASON;
            final int minOpsForFasterFailSafeSave = aSavableDoc.isOTEnabled() ? MIN_OPERATIONS_FOR_FAILSAFESAVE_FOR_OT : MIN_OPERATIONS_FOR_FAILSAFESAVE;

            if ((diff >= MIN_TIME_FOR_MANY_OPS_FAILSAFESAVE) && (nOps >= minOpsForFasterFailSafeSave))
                saveReason = BackgroundSaveReason.TOO_MANY_OPS;
            else if (((diff >= MAX_TIME_FOR_MODIFIED_FAILSAFESAVE) && (nOps > 0)))
                saveReason = BackgroundSaveReason.MODIFICATION_TIMEOUT;

            eReason = saveReason;
        }

        return eReason;
    }

    //-------------------------------------------------------------------------
    /**
     * Determines if a document should be saved or not.
     *
     * @param aSavableDoc a document which can be saved in the background
     * @return TRUE if the document should be saved, otherwise FALSE
     * @throws Exception
     */
    private boolean shouldBeSaved(final IBackgroundSavable aSavableDoc) {
        return (isInValidState(aSavableDoc) &&
                aSavableDoc.isModified() &&
                !aSavableDoc.isSaveInProgress() &&
                (aSavableDoc.getLastSaveErrorCode().getCode() == ErrorCode.NO_ERROR.getCode()));
    }

    //-------------------------------------------------------------------------
    /**
     * Determines if a document instance is valid to be processed.
     *
     * @param aSavableDoc a document which should be checked
     * @return TRUE if the document is in a valid state otherwise FALSE
     * @throws Exception
     */
    private boolean isInValidState(final IBackgroundSavable aSavableDoc) {
        return ((aSavableDoc != null) && !aSavableDoc.isDisposed());
    }

    //-------------------------------------------------------------------------
    /**
     * Calculates the absolute difference between two Dates
     *
     * @param t1 time stamp start in milliseconds
     * @param t2 time stamp end in milliseconds
     * @return The difference in milliseconds.
     */
    private long getTimeDifference(final long t1, final long t2) {
        return Math.abs(t2 - t1);
    }

    //-------------------------------------------------------------------------
    public void shutdown() {
        stop();

        // Let the emergency saver decide which document must be
        // handle via emergencySave. Due to the advisory lock feature
        // this depends on several more conditions.
        EmergencySaver.emergencySave();
    }

}
