/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.rt2.core.doc;

import java.lang.ref.WeakReference;
import java.util.Iterator;
import java.util.List;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.openexchange.exception.ExceptionUtils;
import com.openexchange.office.document.AdvisoryLockInfoService;
import com.openexchange.office.tools.common.weakref.WeakRefUtils;
import com.openexchange.office.tools.osgi.ServiceLookupRegistry;
import com.openexchange.office.tools.osgi.ServiceLookupRegistryService;

public class EmergencySaver
{
    private static final Logger log = LoggerFactory.getLogger(EmergencySaver.class);

    //-------------------------------------------------------------------------
    private EmergencySaver() {
        super();
    }

    //-------------------------------------------------------------------------
    public static void emergencySave() {
        final IDocProcessorContainer docProcessorManager = ServiceLookupRegistryService.getInstance().getService(IRT2DocProcessorManager.class);
        final Set<WeakReference<RT2DocProcessor>> aOpenDocs = docProcessorManager.getWeakReferenceToDocProcessors();
        List<WeakReference<IBackgroundSavable>> savableDocs =
            aOpenDocs.stream()
                     .map(w -> WeakRefUtils.getHardRef(w))
                     .filter(Objects::nonNull)
                     .filter(r -> (r instanceof IBackgroundSavable))
                     .map(r -> new WeakReference<IBackgroundSavable>((IBackgroundSavable)r))
                     .collect(Collectors.toList());
        emergencySave(savableDocs);
    }

    //-------------------------------------------------------------------------
    private static void emergencySave(final List<WeakReference<IBackgroundSavable>> docsForEmergencySave) {
        for (WeakReference<IBackgroundSavable> ref : docsForEmergencySave) {
            final IBackgroundSavable savableDoc = ref.get();
            if (null != savableDoc) {
                // filter document list to find only documents that must be saved or
                // reset their advisory lock info
                if (savableDoc.isModified() || savableDoc.hasSetAdvisoryLock()) {
                    saveDocument(savableDoc);
                }
            }
        }
    }

    //-------------------------------------------------------------------------
    private static boolean saveDocument(final IBackgroundSavable aSavableDoc) {
        try {
            aSavableDoc.save(BackgroundSaveReason.SHUTDOWN, false);
            return true;
        } catch (final Throwable t) {
            ExceptionUtils.handleThrowable(t);
            log.error("RT2: Exception caught while trying to save document with doc-uid " + aSavableDoc.getUniqueID(), t);
            return false;
        }
    }

}
