/*
/*
*
*    OPEN-XCHANGE legal information
*
*    All intellectual property rights in the Software are protected by
*    international copyright laws.
*
*
*    In some countries OX, OX Open-Xchange, open xchange and OXtender
*    as well as the corresponding Logos OX Open-Xchange and OX are registered
*    trademarks of the OX Software GmbH group of companies.
*    The use of the Logos is not covered by the GNU General Public License.
*    Instead, you are allowed to use these Logos according to the terms and
*    conditions of the Creative Commons License, Version 2.5, Attribution,
*    Non-commercial, ShareAlike, and the interpretation of the term
*    Non-commercial applicable to the aforementioned license is published
*    on the web site http://www.open-xchange.com/EN/legal/index.html.
*
*    Please make sure that third-party modules and libraries are used
*    according to their respective licenses.
*
*    Any modifications to this package must retain all copyright notices
*    of the original copyright holder(s) for the original code used.
*
*    After any such modifications, the original and derivative code shall remain
*    under the copyright of the copyright holder(s) and/or original author(s)per
*    the Attribution and Assignment Agreement that can be located at
*    http://www.open-xchange.com/EN/developer/. The contributing author shall be
*    given Attribution for the derivative code and a license granting use.
*
*     Copyright (C) 2016-2020 OX Software GmbH
*     Mail: info@open-xchange.com
*
*
*     This program is free software; you can redistribute it and/or modify it
*     under the terms of the GNU General Public License, Version 2 as published
*     by the Free Software Foundation.
*
*     This program is distributed in the hope that it will be useful, but
*     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
*     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
*     for more details.
*
*     You should have received a copy of the GNU General Public License along
*     with this program; if not, write to the Free Software Foundation, Inc., 59
*     Temple Place, Suite 330, Boston, MA 02111-1307 USA
*
*/

package com.openexchange.office.rt2.core.control.impl;

import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.openexchange.office.rt2.core.control.Task;
import com.openexchange.office.rt2.jms.RT2AdminJmsConsumer;
import com.openexchange.office.rt2.protocol.RT2Message;
import com.openexchange.office.rt2.protocol.RT2MessageFactory;
import com.openexchange.office.rt2.protocol.RT2MessageGetSet;
import com.openexchange.office.rt2.protocol.value.RT2MessageIdType;
import com.openexchange.office.rt2.protocol.value.RT2MessageType;
import com.openexchange.office.tools.osgi.ServiceLookupRegistry;

public class MasterCleanupTask extends Task
{
	private static final Logger log = LoggerFactory.getLogger(MasterCleanupTask.class);
	
    //-------------------------------------------------------------------------
    private final long          m_nCreationTime;

    //-------------------------------------------------------------------------
    private final String        m_sNodeUUID;

    //-------------------------------------------------------------------------
	private final String        m_sNodeUUIDToCleanup;

    //-------------------------------------------------------------------------
    final Map<String, Boolean>  m_aHealthyMembers;

    //-------------------------------------------------------------------------
    public MasterCleanupTask(final String sTaskID, final String sNodeUUID, final String sNodeUUIDToCleanup, final Set<String> aHealthyMembers)
    {
        super(sTaskID);

        m_sNodeUUID          = sNodeUUID;
        m_sNodeUUIDToCleanup = sNodeUUIDToCleanup;
        m_nCreationTime      = System.currentTimeMillis();
        m_aHealthyMembers    = new HashMap<>();

        for (String s : aHealthyMembers)
            m_aHealthyMembers.put(s, new Boolean(false));
    }

    //-------------------------------------------------------------------------
    public String getMemberUUID()
    {
        return m_sNodeUUID;
    }

    //-------------------------------------------------------------------------
    public String getMemberUUIDToCleanup()
    {
        return m_sNodeUUIDToCleanup;
    }

    //-------------------------------------------------------------------------
    public long getCreationTime()
    {
        return m_nCreationTime;
    }

    //-------------------------------------------------------------------------
    public synchronized Set<String> getHealthMembersUUID()
    {
        final Set<String> aSet = new HashSet<String>();
        final Set<String> aKeySet = m_aHealthyMembers.keySet();
        for (String s : aKeySet)
            aSet.add(s);

        return aSet;
    }

    //-------------------------------------------------------------------------
    public boolean setMemberToCompleted(String sUUID)
    {
        boolean bCompleted = false;

        synchronized (this)
        {
            if (m_aHealthyMembers.containsKey(sUUID))
            {
                m_aHealthyMembers.put(sUUID, true);

                bCompleted = true;
                final Set<String> aKeySet = m_aHealthyMembers.keySet();
                for (String id : aKeySet)
                    bCompleted &= m_aHealthyMembers.get(id);
            }
        }

        if (bCompleted)
            setCompleted(true);

        return bCompleted;
    }

    //-------------------------------------------------------------------------
    @Override
    public boolean process() throws Exception
    {
        final RT2AdminJmsConsumer   aRT2Admin = ServiceLookupRegistry.get().getService(RT2AdminJmsConsumer.class);
        final RT2Message aCleanupForNodeTask = RT2MessageFactory.newAdminMessage(RT2MessageType.ADMIN_TASK_CLEANUP_FOR_CRASHED_NODE);

        // ATTENTION: The message ID MUST BE the taken from the clean-up task, otherwise
        // the master node cannot detect which sub-cleanup task belongs to which master task.
        // There can be more than one clean-up at a time!
        aCleanupForNodeTask.setMessageID(new RT2MessageIdType(getTaskID()));
        RT2MessageGetSet.setAdminHZMemberUUID(aCleanupForNodeTask, getMemberUUIDToCleanup());
        RT2MessageGetSet.setAdminHZMasterUUID(aCleanupForNodeTask, getMemberUUID());

        log.debug("RT2 MasterCleanupTask broadcast admin cleanup task to all nodes for crashed node with uid {}", getMemberUUIDToCleanup());

        aRT2Admin.send(aCleanupForNodeTask);

        return true;
    }

    //-------------------------------------------------------------------------
    @Override
    public String toString()
    {
        final StringBuffer aTmp = new StringBuffer(super.toString());
        aTmp.append(",member uuid=");
        aTmp.append(this.getMemberUUID());
        aTmp.append(",master uuid=");
        aTmp.append(this.getMemberUUIDToCleanup());
        return aTmp.toString();
    }
}
