/*
*
*    OPEN-XCHANGE legal information
*
*    All intellectual property rights in the Software are protected by
*    international copyright laws.
*
*
*    In some countries OX, OX Open-Xchange, open xchange and OXtender
*    as well as the corresponding Logos OX Open-Xchange and OX are registered
*    trademarks of the OX Software GmbH group of companies.
*    The use of the Logos is not covered by the GNU General Public License.
*    Instead, you are allowed to use these Logos according to the terms and
*    conditions of the Creative Commons License, Version 2.5, Attribution,
*    Non-commercial, ShareAlike, and the interpretation of the term
*    Non-commercial applicable to the aforementioned license is published
*    on the web site http://www.open-xchange.com/EN/legal/index.html.
*
*    Please make sure that third-party modules and libraries are used
*    according to their respective licenses.
*
*    Any modifications to this package must retain all copyright notices
*    of the original copyright holder(s) for the original code used.
*
*    After any such modifications, the original and derivative code shall remain
*    under the copyright of the copyright holder(s) and/or original author(s)per
*    the Attribution and Assignment Agreement that can be located at
*    http://www.open-xchange.com/EN/developer/. The contributing author shall be
*    given Attribution for the derivative code and a license granting use.
*
*     Copyright (C) 2016-2020 OX Software GmbH
*     Mail: info@open-xchange.com
*
*
*     This program is free software; you can redistribute it and/or modify it
*     under the terms of the GNU General Public License, Version 2 as published
*     by the Free Software Foundation.
*
*     This program is distributed in the hope that it will be useful, but
*     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
*     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
*     for more details.
*
*     You should have received a copy of the GNU General Public License along
*     with this program; if not, write to the Free Software Foundation, Inc., 59
*     Temple Place, Suite 330, Boston, MA 02111-1307 USA
*
*/

package com.openexchange.office.rt2.core.control.impl;

import java.lang.ref.WeakReference;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.atomic.AtomicBoolean;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.openexchange.office.rt2.cache.ClusterLockService;
import com.openexchange.office.rt2.cache.ClusterLockService.ClusterLock;
import com.openexchange.office.rt2.cache.RT2DocInfo;
import com.openexchange.office.rt2.core.RT2Constants;
import com.openexchange.office.rt2.core.control.DelayedTask;
import com.openexchange.office.rt2.core.doc.BackgroundSaveReason;
import com.openexchange.office.rt2.core.doc.DocProcessorClientInfo;
import com.openexchange.office.rt2.core.doc.IBackgroundSavable;
import com.openexchange.office.rt2.core.doc.IDocProcessorContainer;
import com.openexchange.office.rt2.core.doc.RT2DocProcessor;
import com.openexchange.office.rt2.protocol.value.RT2CliendUidType;
import com.openexchange.office.rt2.protocol.value.RT2DocUidType;
import com.openexchange.office.rt2.proxy.RT2DocInfoRegistry;
import com.openexchange.office.tools.logging.annotation.LogMethodCallHelper;
import com.openexchange.office.tools.logging.annotation.Loggable;
import com.openexchange.office.tools.osgi.ServiceLookupRegistry;

public class GCLostClientsTask extends DelayedTask implements Loggable
{
	private static final Logger log = LoggerFactory.getLogger(GCLostClientsTask.class);

    //-------------------------------------------------------------------------
    private final String  m_sNodeUUID;

   //-------------------------------------------------------------------------
    private final String  m_sNodeUUIDToCleanup;

    //-------------------------------------------------------------------------
    private AtomicBoolean m_bSuccessful = new AtomicBoolean(false);

    private final IDocProcessorContainer docProcContainer;
    
    private final ClusterLockService clusterLockService;
    
    //-------------------------------------------------------------------------
	public GCLostClientsTask(String sTaskID, String sNodeUUID, String sNodeUUIDToCleanup, ClusterLockService clusterLockService,
			                 IDocProcessorContainer docProcessorContainer)
	{
		super(sTaskID, RT2Constants.DELAY_GC_LOST_CLIENTS);
		m_sNodeUUID          = sNodeUUID;
		m_sNodeUUIDToCleanup = sNodeUUIDToCleanup;
		this.clusterLockService = clusterLockService;
		this.docProcContainer = docProcessorContainer;
	}

    //-------------------------------------------------------------------------
    public String getMemberUUID()
    {
        return m_sNodeUUID;
    }

    //-------------------------------------------------------------------------
    public String getMemberUUIDToCleanup()
    {
        return m_sNodeUUIDToCleanup;
    }

    //-------------------------------------------------------------------------
    public boolean successful()
    {
        return m_bSuccessful.get();
    }

    //-------------------------------------------------------------------------
    private void setSuccess()
    {
        m_bSuccessful.set(true);
    }

    //-------------------------------------------------------------------------
	@Override
    public boolean process() throws Exception
    {
        LogMethodCallHelper.logMethodCall(this, "process");

        try {
            updateStateOfLocalDocuments(m_sNodeUUIDToCleanup);
            setSuccess();
        }
        finally
        {
            setCompleted(true);
        }

         return true;
    }

    //-------------------------------------------------------------------------
	@Override
	public String toString()
	{
		final StringBuilder aTmp = new StringBuilder(super.toString());
		aTmp.append(", node uuid: ");
		aTmp.append(this.m_sNodeUUID);
		aTmp.append(", clean-up node uuid: ");
		aTmp.append(this.m_sNodeUUIDToCleanup);
		return aTmp.toString();
	}

    //-------------------------------------------------------------------------
    private void updateStateOfLocalDocuments(final String sNodeToCleanup) throws Exception
    {
        Set<WeakReference<RT2DocProcessor>> aLocalDocProcessors = docProcContainer.getWeakReferenceToDocProcessors();
        LogMethodCallHelper.logMethodCall(this, "updateStateOfLocalDocuments", sNodeToCleanup, aLocalDocProcessors.size());

        for (final WeakReference<RT2DocProcessor> aWeakRefDocProc : aLocalDocProcessors)
        {
            final RT2DocProcessor aDocProcessor = aWeakRefDocProc.get();
            if (null != aDocProcessor)
            {
                try
                {
                    updateDocumentState(aDocProcessor, sNodeToCleanup);
                }
                catch (final Exception e)
                {
                    log.error("GCLostClientsTask: Exception caught while trying to save modified document with id " +  aDocProcessor.getDocUID() + " held alive by remote clients on crashed node");
                }
            }
        }
        LogMethodCallHelper.logMethodCallRes(this, "updateStateOfLocalDocuments", Void.class, sNodeToCleanup, docProcContainer.getWeakReferenceToDocProcessors().size());
    }

    //-------------------------------------------------------------------------
    private void updateDocumentState(final RT2DocProcessor aDocProcessor, final String sNodeToCleanup) throws Exception
    {
        final RT2DocInfoRegistry aDocInfoRegistry = ServiceLookupRegistry.get().getService(RT2DocInfoRegistry.class);
        final RT2DocUidType    sDocUID                   = aDocProcessor.getDocUID();
        final RT2DocInfo       aDocInfo                  = aDocInfoRegistry.getDocInfo(sDocUID);
        final List<DocProcessorClientInfo> aClientInfos              = aDocProcessor.getClientsInfo();
        final Set<RT2CliendUidType>      aClientsOnCleanupNode     = determineClientsOfDocOnNode(aClientInfos, sNodeToCleanup);
        final long             nClientCountOnCleanupNode = aClientsOnCleanupNode.size();

        boolean bDeregisterClients = true;
        long    nRefCount;

        ClusterLock clusterLock = clusterLockService.getLock(sDocUID);
        try
        {
            // ATTENTION: We MUST use a document lock and atomic long for the ref-count!
            // In case the ref-count is zero, it's possible that a client wants to dispose
            // document resources while another client wants to create document resources.
            // To have a cluster-wide synchronization we need this lock!
        	clusterLock.lock();

            nRefCount = aDocInfo.decRefCount4Clients(nClientCountOnCleanupNode);

            if (nRefCount <= 0)
            {
                if (nRefCount < 0)
                    aDocInfo.setRefCount4Clients(0);

                bDeregisterClients = false;
                disposeDocumentResources(aDocProcessor, aClientsOnCleanupNode);
            }
        }
        finally
        {
        	clusterLock.unlock();
        }

        if (bDeregisterClients)
            deregisterClientsFromDocument(aDocProcessor, aClientsOnCleanupNode);
    }

    //-------------------------------------------------------------------------
    private void deregisterClientsFromDocument(final RT2DocProcessor aDocProcessor, final Set<RT2CliendUidType> aClientsToDeregister) throws Exception
    {
        if ((null == aDocProcessor) || (null == aClientsToDeregister))
           return;

        for (final RT2CliendUidType sClientUID : aClientsToDeregister)
        {
            try
            {
                aDocProcessor.deregisterClient(sClientUID);
            }
            catch (final Exception e)
            {
                log.error("CleanupTask: Exception caught while trying deregister client with uid " + sClientUID);
            }
        }
    }

    //-------------------------------------------------------------------------
    private void disposeDocumentResources(final RT2DocProcessor aDocProcessor, final Set<RT2CliendUidType> aClientsOfDoc) throws Exception
    {
        if (null == aDocProcessor)
            return;

        tryToDoEmergencySaveForDocument(aDocProcessor);
        deregisterClientsFromDocument(aDocProcessor, aClientsOfDoc);
        aDocProcessor.dispose();
    }

    //-------------------------------------------------------------------------
    private Set<RT2CliendUidType> determineClientsOfDocOnNode(final List<DocProcessorClientInfo> aClientInfos, final String sNodeToCleanup) throws Exception
    {
        final Set<RT2CliendUidType> aClientSet = new HashSet<>();

        for (final DocProcessorClientInfo aClientInfo : aClientInfos)
        {
            if (sNodeToCleanup.equals(aClientInfo.getNodeUUID()))
                aClientSet.add(aClientInfo.getClientUID());
        }

        return aClientSet;
    }

    //-------------------------------------------------------------------------
    private void tryToDoEmergencySaveForDocument(final RT2DocProcessor aDocProcessor) throws Exception
    {
        final IBackgroundSavable aBackgroundSavable = (IBackgroundSavable)aDocProcessor;
        if ((null != aBackgroundSavable) && (aBackgroundSavable.isModified()))
        {
            try
            {
                aBackgroundSavable.save(BackgroundSaveReason.SHUTDOWN, true);
            }
            catch (Exception e)
            {
                log.error("CleanupTask: Exception caught while trying to save modified document with docUid " + aDocProcessor.getDocUID() + " held alive by remote clients on crashed node", e);
            }
        }
    }

	@Override
	public Logger getLogger() {
		return log;
	}

	@Override
	public Map<String, Object> getAdditionalLogInfo() {
		Map<String, Object> res = new HashMap<>();
		res.put("nodeUuidToCleanup", m_sNodeUUIDToCleanup);
		
		return res;
	}
}
