/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.rt2.config;

import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import org.apache.commons.lang3.StringUtils;
import com.openexchange.config.ConfigurationService;
import com.openexchange.office.tools.osgi.ServiceLookupRegistryService;

//=============================================================================
public class RT2ConfigItem {

    //-------------------------------------------------------------------------
    public static final String KEY_MONITOR_SERVER_HOST = "rt2.monitoring.server.host";
    public static final String KEY_MONITOR_SERVER_PORT = "rt2.monitoring.server.port";

    public static final String KEY_OX_NODE_ID = "com.openexchange.server.backendRoute";

    public static final String KEY_GC_FREQUENCY = "rt2.gc.frequency";
    public static final String KEY_GC_OFFLINE_TRESHOLD = "rt2.gc.offline.treshold";

    public static final String KEY_TIMEOUT_4_SEND = "rt2.timeout.send";

    public static final String KEY_KEEPALIVE_FREQUENCY = "rt2.keepalive.frequency";

    public static final String KEY_MAX_SESSIONS_PER_NODE = "com.openexchange.office.maxOpenDocumentsPerUser";

    public static final String KEY_ACK_FLUSH = "rt2.ack.flush";

    public static final String KEY_TIMEOUT_FOR_RECONNECTED_CLIENT = "rt2.client.timeout.reconnect.before.hangup";
    public static final String KEY_TIMEOUT_FOR_BAD_CLIENT = "rt2.client.timeout.bad.client";
    public static final String KEY_MAX_MESSAGES_BEFORE_HANGUP = "rt2.client.max.messages.before.hangup";
    public static final String KEY_NACK_FREQUENCE_OF_SERVER = "internalNackFrequenceOfServer";

    public static final String JMS_KEY_SSL_VERIFY_HOSTNAME = "com.openexchange.dcs.client.ssl.verifyHostname";

    public static final String KEY_JMS_SSL_KEYSTORE_PATH = "com.openexchange.dcs.client.ssl.keystore.path";
    public static final String JMS_SSL_KEYSTORE_PATH_DEFAULT = "/opt/open-xchange/etc/security/documents-collaboration-client.ks";

    public static final String KEY_JMS_SSL_KEYSTORE_PASSWORD = "com.openexchange.dcs.client.ssl.keystore.password";
    public static final String JMS_SSL_KEYSTORE_PASSWORD_DEFAULT = "password";

    public static final String KEY_JMS_SSL_TRUSTSTORE_PATH = "com.openexchange.dcs.client.ssl.truststore.path";
    public static final String JMS_SSL_TRUSTSTORE_PATH_DEFAULT = "/opt/open-xchange/etc/security/documents-collaboration-client.ts";

    public static final String KEY_JMS_SSL_TRUSTSTORE_PASSWORD = "com.openexchange.dcs.client.ssl.truststore.password";
    public static final String JMS_SSL_TRUSTSTORE_PASSWORD_DEFAULT = "password";

    public static final String KEY_BACKGROUND_SAVE_DOC = "io.ox/office//module/backgroundSaveDoc";

    //-------------------------------------------------------------------------
    private static RT2ConfigItem m_gDefaultInst = null;

    //-------------------------------------------------------------------------
    private Map<String, Object> m_overrides = new ConcurrentHashMap<>();

    //-------------------------------------------------------------------------
    public static synchronized RT2ConfigItem get() {
        if (m_gDefaultInst == null)
            m_gDefaultInst = new RT2ConfigItem();
        return m_gDefaultInst;
    }

    //-------------------------------------------------------------------------
    public boolean isRT2MonitoringOn() {
        final boolean bIsOn = impl_existsKey(KEY_MONITOR_SERVER_HOST);
        return bIsOn;
    }

    //-------------------------------------------------------------------------
    public String getRT2MonitoringServerHost() {
        final String sHost = impl_readString(KEY_MONITOR_SERVER_HOST, RT2Const.DEFAULT_MONITOR_HOST);
        return sHost;
    }

    //-------------------------------------------------------------------------
    public int getRT2MonitoringServerPort() {
        final int nPort = impl_readInt(KEY_MONITOR_SERVER_PORT, RT2Const.DEFAULT_MONITOR_PORT);
        return nPort;
    }

    //-------------------------------------------------------------------------
    public String getOXNodeID() {
        final String sID = mem_Config().getProperty(KEY_OX_NODE_ID);
        return sID;
    }

    //-------------------------------------------------------------------------
    public long getRT2GCFrequencyInMS() {
        long nFrequencyInMS = RT2Const.DEFAULT_GC_FREQUENCY_IN_MS;

        final Long nValue = getValueFromOverride(KEY_GC_FREQUENCY, Long.class);
        if (null == nValue)
            nFrequencyInMS = impl_readLong(KEY_GC_FREQUENCY, RT2Const.DEFAULT_GC_FREQUENCY_IN_MS);
        else
            nFrequencyInMS = nValue;
        return nFrequencyInMS;
    }

    //-------------------------------------------------------------------------
    public long getRT2GCOfflineTresholdInMS() {
        long nTresholdInMS = RT2Const.DEFAULT_GC_OFFLINE_TRESHOLD_IN_MS;

        final Long nValue = getValueFromOverride(KEY_GC_OFFLINE_TRESHOLD, Long.class);
        if (null == nValue)
            nTresholdInMS = impl_readLong(KEY_GC_OFFLINE_TRESHOLD, RT2Const.DEFAULT_GC_OFFLINE_TRESHOLD_IN_MS);
        else
            nTresholdInMS = nValue;
        return nTresholdInMS;
    }

    //-------------------------------------------------------------------------
    public long getRT2KeepAliveFrequencyInMS() {
        final long nFrequencyInMS = impl_readLong(KEY_KEEPALIVE_FREQUENCY, RT2Const.DEFAULT_KEEPALIVE_FREQUENCY_IN_MS);
        return nFrequencyInMS;
    }

    //-------------------------------------------------------------------------
    public boolean isBackgroundSaveOn() {
        final boolean bIsOn = impl_readBoolean(KEY_BACKGROUND_SAVE_DOC, RT2Const.DEFAULT_BACKGROUND_SAVE_DOC);
        return bIsOn;
    }

    //-------------------------------------------------------------------------
    public long getRT2TimeoutSendInMS() {
        final long nTimeoutInMS = impl_readLong(KEY_TIMEOUT_4_SEND, RT2Const.DEFAULT_TIMEOUT_4_SEND_IN_MS);
        return nTimeoutInMS;
    }

    //-------------------------------------------------------------------------
    public void setRT2GCFrequencyInMS(long frequency) {
        m_overrides.put(KEY_GC_FREQUENCY, frequency);
    }

    //-------------------------------------------------------------------------
    public void setRT2GCOfflineTresholdInMS(long timeout) {
        m_overrides.put(KEY_GC_OFFLINE_TRESHOLD, timeout);
    }

    //-------------------------------------------------------------------------
    public int getRT2MaxSessionsPerNode() {
        int maxSessions = impl_readInt(KEY_MAX_SESSIONS_PER_NODE, RT2Const.DEFAULT_MAX_SESSIONS_PER_NODE);
        return maxSessions;
    }

    //-------------------------------------------------------------------------
    public int getAckResponsePeriod(int defValue) {
        int res = impl_readInt(KEY_ACK_FLUSH, defValue);
        return res;
    }

    //-------------------------------------------------------------------------
    public int getTimeoutBeforeHangupReconnectedClient(int defValue) {
        int res = impl_readInt(KEY_TIMEOUT_FOR_RECONNECTED_CLIENT, defValue);
        return res;
    }

    //-------------------------------------------------------------------------
    public int getTimeoutForBadClient(int defValue) {
        int res = impl_readInt(KEY_TIMEOUT_FOR_BAD_CLIENT, defValue);
        return res;
    }

    //-------------------------------------------------------------------------
    public int getMaxMessagesCountWithoutAck(int defValue) {
        int res = impl_readInt(KEY_MAX_MESSAGES_BEFORE_HANGUP, defValue);
        return res;
    }

    //-------------------------------------------------------------------------
    public Integer getNackFrequenceOfServer() {
        if (m_overrides.containsKey(KEY_NACK_FREQUENCE_OF_SERVER)) {
            return (Integer) m_overrides.get(KEY_NACK_FREQUENCE_OF_SERVER);
        }
        return 60;
    }

    //-------------------------------------------------------------------------
    public void updateNackFrequenceOfServer(Integer value) {
        m_overrides.put(KEY_NACK_FREQUENCE_OF_SERVER, value);
    }

    //-------------------------------------------------------------------------
    public boolean isRT2SSLVerifyHostname() {
        return mem_Config().getBoolProperty(JMS_KEY_SSL_VERIFY_HOSTNAME, false);
    }

    //-------------------------------------------------------------------------
    public String getRT2SSLKeystorePath() {
        final String sID = mem_Config().getProperty(KEY_JMS_SSL_KEYSTORE_PATH);
        return StringUtils.isBlank(sID) ? JMS_SSL_KEYSTORE_PATH_DEFAULT : sID;
    }

    //-------------------------------------------------------------------------
    public String getRT2SSLKeystorePassword() {
        final String sID = mem_Config().getProperty(KEY_JMS_SSL_KEYSTORE_PASSWORD);
        return StringUtils.isBlank(sID) ? JMS_SSL_KEYSTORE_PASSWORD_DEFAULT : sID;
    }

    //-------------------------------------------------------------------------
    public String getRT2SSLTrustStorePath() {
        final String sID = mem_Config().getProperty(KEY_JMS_SSL_TRUSTSTORE_PATH);
        return StringUtils.isBlank(sID) ? JMS_SSL_TRUSTSTORE_PATH_DEFAULT : sID;
    }

    //-------------------------------------------------------------------------
    public String getRT2SSLTrustStorePassword() {
        final String sID = mem_Config().getProperty(KEY_JMS_SSL_TRUSTSTORE_PASSWORD);
        return StringUtils.isBlank(sID) ? JMS_SSL_TRUSTSTORE_PASSWORD_DEFAULT : sID;
    }

    //-------------------------------------------------------------------------
    private boolean impl_existsKey(final String sKey) {
        final String sValue = mem_Config().getProperty(sKey);
        final boolean bIs = !StringUtils.isEmpty(sValue);
        return bIs;
    }

    //-------------------------------------------------------------------------
    private String impl_readString(final String sKey, final String sDefault) {
        final String sValue = mem_Config().getProperty(sKey, sDefault);
        if (sValue == null) {
            return sDefault;
        }
        return sValue;
    }

    //-------------------------------------------------------------------------
    private int impl_readInt(final String sKey, final int nDefault) {
        final String sValue = mem_Config().getProperty(sKey, Integer.toString(nDefault));
        if (sValue == null) {
            return nDefault;
        }
        final int nValue = Integer.parseInt(sValue);
        return nValue;
    }

    //-------------------------------------------------------------------------
    private long impl_readLong(final String sKey, final long nDefault) {
        final String sValue = mem_Config().getProperty(sKey, Long.toString(nDefault));
        if (sValue == null) {
            return nDefault;
        }
        final long nValue = Long.parseLong(sValue);
        return nValue;
    }

    //-------------------------------------------------------------------------
    private boolean impl_readBoolean(final String sKey, final boolean bDefault) {
        final String sValue = mem_Config().getProperty(sKey, Boolean.toString(bDefault));
        if (sValue == null) {
            return bDefault;
        }
        final boolean bValue = Boolean.parseBoolean(sValue);
        return bValue;
    }

    //-------------------------------------------------------------------------
    private ConfigurationService mem_Config() {
        final ConfigurationService aConfig = ServiceLookupRegistryService.getInstance().getService(ConfigurationService.class);
        return aConfig;
    }

    //-------------------------------------------------------------------------
    @SuppressWarnings("unchecked")
    private <T> T getValueFromOverride(String key, Class<T> clazz) {
        final Object objValue = m_overrides.get(key);
        if ((objValue != null) && (clazz.isAssignableFrom(objValue.getClass()))) {
            return (T) objValue;
        }

        return null;
    }

}
