/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.rest.job;

import java.util.Queue;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicReference;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class AsyncJobWorker<T extends Job> implements Runnable {

    // ---------------------------------------------------------------
    protected static final Logger log = LoggerFactory.getLogger(AsyncJobWorker.class);

    // ---------------------------------------------------------------
    private ExecutorService pool;

    // ---------------------------------------------------------------
    private Queue<T> aJobQueue = new ConcurrentLinkedQueue<>();

    // ---------------------------------------------------------------
    private AtomicBoolean shutdown = new AtomicBoolean(false);

    // ---------------------------------------------------------------
    private AtomicReference<Thread> started = new AtomicReference<>(null);

    // ---------------------------------------------------------------
    public AsyncJobWorker(int threadCount) {
        pool = getThreadPool(threadCount);
    }

    // ---------------------------------------------------------------
    public boolean hasJobs() { return (aJobQueue.size() > 0); }

    // ---------------------------------------------------------------
    public void addJob(final T job) {
        aJobQueue.offer(job);

        if (started.compareAndSet(null, new Thread(this, "com.openexchange.office.rest.AsyncOperationsPrefetcher"))) {
            started.get().start();
        }

        synchronized (aJobQueue) {
            aJobQueue.notifyAll();
        }
    }

    // ---------------------------------------------------------------
    public void shutdown() {
        if (shutdown.compareAndSet(false, true)) {
            pool.shutdown(); // Disable new tasks from being submitted

            try {
                if (!pool.awaitTermination(5, TimeUnit.SECONDS)) {
                  pool.shutdownNow();

                  if (!pool.awaitTermination(5, TimeUnit.SECONDS))
                      log.warn("AsyncJobWorker pool did not terminate");
                }
            } catch (InterruptedException e) {
                pool.shutdownNow();
                Thread.currentThread().interrupt();
            }
        }
    }

    // ---------------------------------------------------------------
    @Override
    public void run() {
        while (!shutdown.get()) {
            final T job = aJobQueue.poll();

            if (null != job) {
                pool.submit(new JobRunner<>(job));
            } else {
                try {
                    synchronized (aJobQueue) {
                        aJobQueue.wait(15);
                    }
                } catch (InterruptedException e) {
                    Thread.currentThread().interrupt();
                }
            }
        }

        log.debug("AsyncOperationsPrefetcher stopped");
    }

    // ---------------------------------------------------------------
    private static ExecutorService getThreadPool(int threadCount)
    {
        return Executors.newFixedThreadPool(threadCount);
    }

}
