/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.rest;

import java.io.InputStream;
import java.util.HashMap;
import java.util.Map;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringUtils;
import org.json.JSONException;
import org.json.JSONObject;
import com.openexchange.ajax.container.ByteArrayFileHolder;
import com.openexchange.ajax.requesthandler.AJAXRequestData;
import com.openexchange.ajax.requesthandler.AJAXRequestResult;
import com.openexchange.ajax.requesthandler.DispatcherNotes;
import com.openexchange.documentconverter.Properties;
import com.openexchange.exception.OXException;
import com.openexchange.filemanagement.ManagedFile;
import com.openexchange.filemanagement.ManagedFileManagement;
import com.openexchange.java.Streams;
import com.openexchange.office.documents.DocRequest;
import com.openexchange.office.documents.DocRequestFactory;
import com.openexchange.office.documents.DocRequester;
import com.openexchange.office.documents.DocRequests;
import com.openexchange.office.message.MessagePropertyKey;
import com.openexchange.office.rest.tools.ConversionHelper;
import com.openexchange.office.rest.tools.ConversionHelper.ConversionData;
import com.openexchange.office.rest.tools.ParamValidator;
import com.openexchange.office.tools.error.ErrorCode;
import com.openexchange.office.tools.error.HttpStatusCode;
import com.openexchange.office.tools.files.FileHelper;
import com.openexchange.office.tools.logging.v2.ELogLevel;
import com.openexchange.office.tools.logging.v2.impl.Logger;
import com.openexchange.office.tools.logging.v2.impl.Slf4JLogger;
import com.openexchange.office.tools.osgi.ServiceLookupRegistry;
import com.openexchange.tools.session.ServerSession;

@DispatcherNotes(defaultFormat = "file", allowPublicSession = true)
public class GetDocumentAction extends DocumentRESTAction {

    // ---------------------------------------------------------------
    private static final Logger   LOG                    = Slf4JLogger.create(GetDocumentAction.class);

    // ---------------------------------------------------------------
    private static final String   PDF_MIMETYPE           = "application/pdf";
    private static final String   PDF_EXTENSION          = "pdf";
    private static final String[] aMandatoryParams       = { ParameterDefinitions.PARAM_DOCUID,
                                                             ParameterDefinitions.PARAM_FILE_ID,
                                                             ParameterDefinitions.PARAM_FOLDER_ID,
                                                             ParameterDefinitions.PARAM_FILENAME,
                                                             ParameterDefinitions.PARAM_MIMETYPE,
                                                             ParameterDefinitions.PARAM_CLIENTUID };

    /**
     * Initializes a new {@link GetDocumentAction}.
     *
     * @param services
     * @param oqm
     */
    // ---------------------------------------------------------------
    public GetDocumentAction() {
        super();
    }

    // ---------------------------------------------------------------
	@Override
	public AJAXRequestResult perform(AJAXRequestData requestData, ServerSession session) throws OXException {
        AJAXRequestResult ajaxResult = null;

        if (!ParamValidator.areAllParamsNonEmpty(requestData, aMandatoryParams))
            return ParamValidator.getResultFor(HttpStatusCode.BAD_REQUEST.getStatusCode());

        final String folderId = requestData.getParameter(ParameterDefinitions.PARAM_FOLDER_ID);
        final String fileId   = requestData.getParameter(ParameterDefinitions.PARAM_FILE_ID);

        if (!ParamValidator.canReadFile(session, folderId, fileId))
            return ParamValidator.getResultFor(HttpStatusCode.FORBIDDEN.getStatusCode());

        final String clientUID    = requestData.getParameter(ParameterDefinitions.PARAM_CLIENTUID);
        final String docUID       = requestData.getParameter(ParameterDefinitions.PARAM_DOCUID);
        final String fileName     = requestData.getParameter(ParameterDefinitions.PARAM_FILENAME);
        final String authCode     = requestData.getParameter(ParameterDefinitions.PARAM_AUTH_CODE);
        final boolean encrypted   = StringUtils.isNoneEmpty(authCode);

        ErrorCode  errorCode      = ErrorCode.LOADDOCUMENT_CANNOT_RETRIEVE_OPERATIONS_ERROR;

        final DocRequester aDocRequester = ServiceLookupRegistry.get().getService(DocRequester.class);
        if (null == aDocRequester)
            return ParamValidator.getResultFor(HttpStatusCode.INTERNAL_SERVER_ERROR.getStatusCode());

        final ManagedFileManagement fileManager = ServiceLookupRegistry.get().getService(ManagedFileManagement.class);

        if (null != fileManager) {
            ManagedFile aManagedFile = null;
            try {
                final DocRequest aFlushDocToManagedFileReq = DocRequestFactory.createDocRequest(DocRequests.DOCREQUEST_FLUSHTOMANAGEDFILE, docUID, clientUID);
                final String sManagedFileID = aDocRequester.sendSyncRequest(aFlushDocToManagedFileReq);

                if (StringUtils.isNotEmpty(sManagedFileID)) {

                    aManagedFile = fileManager.getByID(sManagedFileID);

                    ajaxResult = convertManagedFile(session, aManagedFile, fileName, encrypted);

                    // in case of a json result make sure that no conversion error occurs
                    // and set format of the request
                    if ("json".equals(ajaxResult.getFormat()))
                        requestData.setFormat("json");
                }
            } catch (Exception e) {
                LOG.forLevel(ELogLevel.E_ERROR).withError(e).log();
            } finally {
                if (null != aManagedFile)
                    fileManager.removeByID(aManagedFile.getID());
            }
        } else {
            errorCode = ErrorCode.GENERAL_MISSING_SERVICE_ERROR;
        }

        if (null == ajaxResult) {
            ajaxResult = createErrorResult(errorCode);
            requestData.setFormat("json");
        }

        return ajaxResult;
	}

    // ---------------------------------------------------------------
	private AJAXRequestResult convertManagedFile(final ServerSession session, final ManagedFile aManagedFile, String fileName, boolean encrypted) throws Exception {
        AJAXRequestResult ajaxResult = null;

        final Map<String, String> conversionFormat = new HashMap<>(2);
        conversionFormat.put(Properties.PROP_MIME_TYPE, PDF_MIMETYPE);
        conversionFormat.put(Properties.PROP_INPUT_TYPE, PDF_EXTENSION);
        conversionFormat.put(Properties.PROP_FILTER_SHORT_NAME, PDF_EXTENSION);

        final InputStream origDocumentStm = aManagedFile.getInputStream();
        final ConversionData aConversionData = ConversionHelper.convertDocument(session, origDocumentStm, fileName, conversionFormat, encrypted);
        final InputStream outDocumentStm = aConversionData.getInputStream();

        ErrorCode errorCode = aConversionData.getErrorCode();

        if (null != outDocumentStm) {
            try {
                final byte[] dataBuffer = IOUtils.toByteArray(outDocumentStm);

                if (null != dataBuffer) {
                    try (final ByteArrayFileHolder fileHolder = new ByteArrayFileHolder(dataBuffer)) {
                        final String resourceName = FileHelper.getFileNameWithReplacedExtension(fileName, PDF_EXTENSION, encrypted);

                        fileHolder.setName(resourceName);
                        fileHolder.setContentType(PDF_MIMETYPE);
                        ajaxResult = new AJAXRequestResult(fileHolder, "file");
                    } catch (Exception e) {
                        LOG.forLevel(ELogLevel.E_ERROR).withError(e).log();
                        errorCode = ErrorCode.GENERAL_UNKNOWN_ERROR;
                    }
                }
            } finally {
                Streams.close(outDocumentStm);
            }
        }

        if (null == ajaxResult) {
            ajaxResult = createErrorResult(errorCode);
        }

        return ajaxResult;
	}

    // ---------------------------------------------------------------
	private AJAXRequestResult createErrorResult(ErrorCode errorCode) {
        final JSONObject jsonResult = new JSONObject();

        try {
            jsonResult.put(MessagePropertyKey.KEY_ERROR_DATA, errorCode.getAsJSON());
        } catch (final JSONException je) {
            /* ignore it */
        }

        return new AJAXRequestResult(jsonResult, "json");
	}
}
