/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.rest;

import java.util.LinkedList;
import java.util.Queue;
import java.util.concurrent.atomic.AtomicBoolean;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;

public abstract class AsyncFileWriter<T> {

    private String filePath = null;
    private Queue<T> dataQueue = new LinkedList<T>();
    private Runnable asyncWriterImpl = null;
    private Thread asyncWriterThread = null;
    private AtomicBoolean shutdown = new AtomicBoolean(false);
    private Object queueNotEmpty = new Object();
    private final String threadName;

    public AsyncFileWriter(String threadName) {
        this.threadName = threadName;
    }

    public void init(String filePathToWrite) throws Exception {
        Validate.notEmpty(filePathToWrite);

        synchronized (dataQueue) {
            if (null == filePath)
                filePath = filePathToWrite;
        }
    }

    protected String getFilePath() {
        return filePath;
    }

    public boolean add(final T data) {
        boolean added = false;

        synchronized (dataQueue) {
            if (!shutdown.get() && StringUtils.isNotEmpty(filePath)) {
                dataQueue.add(data);
                added = true;
            }

            if (added && (null == asyncWriterImpl)) {
                asyncWriterImpl = createRunnable();
                asyncWriterThread = new Thread(asyncWriterImpl, threadName);
                asyncWriterThread.start();
            }
        }

        if (added) {
            synchronized (queueNotEmpty) {
                queueNotEmpty.notifyAll();
            }
        }

        return added;
    }

    public void reset() {
        synchronized (dataQueue) {
            dataQueue.clear();
        }
    }

    public boolean hasData() {
        synchronized (dataQueue) {
            return !dataQueue.isEmpty();
        }
    }

    public void shutdown() {
        Thread aAsyncWriterThread = null;

        if (shutdown.compareAndSet(false, true)) {
            synchronized (dataQueue) {
                dataQueue.clear();
                asyncWriterImpl = null;
                aAsyncWriterThread = asyncWriterThread;
                asyncWriterThread = null;
            }
        }

        if (null != aAsyncWriterThread) {
            synchronized (queueNotEmpty) {
                queueNotEmpty.notifyAll();
            }

            try {
                aAsyncWriterThread.join(500);
            } catch (InterruptedException e) {
                Thread.currentThread().interrupt();
            }
        }
    }

    protected abstract void writeData(final T dataToWrite);

    protected abstract void logErrorMessage(final String sMsg, final Exception e);

    private Runnable createRunnable() {
        return new Runnable() {

            @SuppressWarnings("synthetic-access")
            @Override
            public void run() {
                while (!shutdown.get()) {
                    try {
                        synchronized (queueNotEmpty) {
                            queueNotEmpty.wait(50);
                        }

                        T nextDataToWrite = null;

                        synchronized (dataQueue) {
                            nextDataToWrite = dataQueue.poll();
                        }

                        if (null != nextDataToWrite) {
                            writeData(nextDataToWrite);
                        }

                    } catch (final InterruptedException e) {
                        Thread.currentThread().interrupt();
                    } catch (final Exception e) {
                        logErrorMessage("Exception caught trying to determine and write performance data asynchronously", e);
                    }
                }
            }
        };
    }
}
