/*
 *
 *  Copyright (C) 2016-2020 OX Software GmbH All rights reserved.
 *
 *  This program is proprietary software and licensed to you under
 *  OX Software GmbH's Software License Agreement.
 *
 *  In some countries OX, OX Open-Xchange, open-xchange and OXtender as well as
 *  the corresponding Logos OX Open-Xchange and OX are registered trademarks.
 *
 */

package com.openexchange.office.monitoring;

import javax.management.NotCompliantMBeanException;
import javax.management.StandardMBean;
import com.openexchange.annotation.NonNull;
import com.openexchange.office.monitoring.api.IMonitoring;
import com.openexchange.office.tools.doc.DocumentType;
import com.openexchange.office.tools.monitoring.CloseType;
import com.openexchange.office.tools.monitoring.OpenType;
import com.openexchange.office.tools.monitoring.OperationsType;
import com.openexchange.office.tools.monitoring.SaveType;
import com.openexchange.office.tools.monitoring.Statistics;

/**
 * {@link OfficeMonitoring}
 *
 * @author <a href="mailto:kai.ahrens@open-xchange.com">Kai Ahrens</a>
 */
public class OfficeMonitoring extends StandardMBean implements IMonitoring {

    /**
     * Domain
     */
    public static final String DOMAIN = "com.openexchange.office";

    /**
     * Initializes a new {@link OfficeMonitoring}.
     * @param statistics
     * @throws NotCompliantMBeanException
     */
    public OfficeMonitoring(@NonNull Statistics statistics) throws NotCompliantMBeanException {
        super(IMonitoring.class);

        m_statistics = statistics;
    }

    // - IMonitoringCommon

    @Override
    public String getAdvisoryLockInfo_Mode() {
        return m_statistics.getAdvisoryLockInfoMode();
    }

    /* (non-Javadoc)
     * @see com.openexchange.office.IInformation#getDocumentsCreated_Total()
     */
    @Override
    public long getDocumentsCreated_Total() {
        return m_statistics.getDocumentsCreated(null);
    }

    /* (non-Javadoc)
     * @see com.openexchange.office.getDocumentsOperations_Text_Outgoing_Distributeted#getDocumentsCreated_Text()
     */
    @Override
    public long getDocumentsCreated_Text() {
        return m_statistics.getDocumentsCreated(DocumentType.TEXT);
    }

    /* (non-Javadoc)
     * @see com.openexchange.office.IInformation#getDocumentsCreated_Spreadsheet()
     */
    @Override
    public long getDocumentsCreated_Spreadsheet() {
        return m_statistics.getDocumentsCreated(DocumentType.SPREADSHEET);
    }

    /* (non-Javadoc)
     * @see com.openexchange.office.IInformation#getDocumentsCreated_Presentation()
     */
    @Override
    public long getDocumentsCreated_Presentation() {
        return m_statistics.getDocumentsCreated(DocumentType.PRESENTATION);
    }

    /* (non-Javadoc)
     * @see com.openexchange.office.IMonitoring#getMedianDocumentsSize_Total()
     */
    @Override
    public long getDocumentsSizeMedian_Total() {
        return m_statistics.getDocumentsSizeMedian(null);
    }

    /* (non-Javadoc)
     * @see com.openexchange.office.getDocumentsOperations_Text_Outgoing_Distributeted#getMedianDocumentsSize_Text_KB()
     */
    @Override
    public long getDocumentsSizeMedian_Text() {
        return m_statistics.getDocumentsSizeMedian(DocumentType.TEXT);
    }

    /* (non-Javadoc)
     * @see com.openexchange.office.IInformation#getMedianDocumentsSize_Spreadsheet()
     */
    @Override
    public long getDocumentsSizeMedian_Spreadsheet() {
        return m_statistics.getDocumentsSizeMedian(DocumentType.SPREADSHEET);
    }

    /* (non-Javadoc)
     * @see com.openexchange.office.IInformation#getMedianDocumentsSize_Presentation()
     */
    @Override
    public long getDocumentsSizeMedian_Presentation() {
        return m_statistics.getDocumentsSizeMedian(DocumentType.PRESENTATION);
    }

    @Override
    public long getRestoreDocsCreated_Total() {
        return m_statistics.getRestoreDocCreated();
    }

    @Override
    public long getRestoreDocsRemoved_Total() {
        return m_statistics.getRestoreDocRemoved();
    }

    @Override
    public long getRestoreDocs_Current() {
        return m_statistics.getRestoreDocCurrent();
    }

    @Override
    public long getRestoreDocsRestored_Success() {
        return m_statistics.getRestoreDocRestoredSuccess();
    }

    @Override
    public long getRestoreDocsRestored_Failure() {
        return m_statistics.getRestoreDocRestoredFailure();
    }

    // - IMonitoringText

    /* (non-Javadoc)
     * @see com.openexchange.office.getDocumentsOperations_Text_Outgoing_Distributeted#getDocumentsOpened_Text_Total()
     */
    @Override
    public long getDocumentsOpened_Text_Total() {
        return m_statistics.getDocumentsOpened(DocumentType.TEXT, null);
    }

    /* (non-Javadoc)
     * @see com.openexchange.office.getDocumentsOperations_Text_Outgoing_Distributeted#getDocumentsOpened_Text_Msx()
     */
    @Override
    public long getDocumentsOpened_Text_OOXML() {
        return m_statistics.getDocumentsOpened(DocumentType.TEXT, OpenType.MSX);
    }

    /* (non-Javadoc)
     * @see com.openexchange.office.getDocumentsOperations_Text_Outgoing_Distributeted#getDocumentsOpened_Text_Binary()
     */
    @Override
    public long getDocumentsOpened_Text_Binary() {
        return m_statistics.getDocumentsOpened(DocumentType.TEXT, OpenType.BINARY);
    }

    /* (non-Javadoc)
     * @see com.openexchange.office.getDocumentsOperations_Text_Outgoing_Distributeted#getDocumentsOpened_Text_Odf()
     */
    @Override
    public long getDocumentsOpened_Text_ODF() {
        return m_statistics.getDocumentsOpened(DocumentType.TEXT, OpenType.ODF);
    }

    /* (non-Javadoc)
     * @see com.openexchange.office.getDocumentsOperations_Text_Outgoing_Distributeted#getDocumentsClosed_Text_Total()
     */
    @Override
    public long getDocumentsClosed_Text_Total() {
        return m_statistics.getDocumentsClosed(DocumentType.TEXT, CloseType.CLOSE);
    }

    /* (non-Javadoc)
     * @see com.openexchange.office.getDocumentsOperations_Text_Outgoing_Distributeted#getDocumentsClosed_Text_Timeout()
     */
    @Override
    public long getDocumentsClosed_Text_Timeout() {
        return m_statistics.getDocumentsClosed(DocumentType.TEXT, CloseType.TIMEOUT);
    }

    /* (non-Javadoc)
     * @see com.openexchange.office.IInformation#getDocumentsSaved_Text_Total()
     */
    @Override
    public long getDocumentsSaved_Text_Total() {
        return m_statistics.getDocumentsSaved(DocumentType.TEXT, null);
    }

    /* (non-Javadoc)
     * @see com.openexchange.office.getDocumentsOperations_Text_Outgoing_Distributeted#getDocumentsSaved_Text_Close()
     */
    @Override
    public long getDocumentsSaved_Text_Close() {
        return m_statistics.getDocumentsSaved(DocumentType.TEXT, SaveType.CLOSE);
    }

    /* (non-Javadoc)
     * @see com.openexchange.office.getDocumentsOperations_Text_Outgoing_Distributeted#getDocumentsSaved_Text_100ops()
     */
    @Override
    public long getDocumentsSaved_Text_100ops() {
        return m_statistics.getDocumentsSaved(DocumentType.TEXT, SaveType.OPS_100);
    }

    /* (non-Javadoc)
     * @see com.openexchange.office.getDocumentsOperations_Text_Outgoing_Distributeted#getDocumentsSaved_Text_15min()
     */
    @Override
    public long getDocumentsSaved_Text_15mins() {
        return m_statistics.getDocumentsSaved(DocumentType.TEXT, SaveType.OPS_15MINS);
    }

    /* (non-Javadoc)
     * @see com.openexchange.office.getDocumentsOperations_Text_Outgoing_Distributeted#getDocumentOperations_Text_Incoming()
     */
    @Override
    public long getDocumentsOperations_Text_Incoming() {
        return m_statistics.getDocumentsOperations(DocumentType.TEXT, OperationsType.INCOMING);
    }

    /* (non-Javadoc)
     * @see com.openexchange.office.getDocumentsOperations_Text_Outgoing_Distributeted#getDocumentOperations_Text_Outgoing_Distributeted()
     */
    @Override
    public long getDocumentsOperations_Text_Distributed() {
        return m_statistics.getDocumentsOperations(DocumentType.TEXT, OperationsType.DISTRIBUTED);
    }

    // - IMonitoringSpreadsheet

    /* (non-Javadoc)
     * @see com.openexchange.office.getDocumentsOperations_Spreadsheet_Outgoing_Distributeted#getDocumentsOpened_Spreadsheet_Total()
     */
    @Override
    public long getDocumentsOpened_Spreadsheet_Total() {
        return m_statistics.getDocumentsOpened(DocumentType.SPREADSHEET, null);
    }

    /* (non-Javadoc)
     * @see com.openexchange.office.getDocumentsOperations_Spreadsheet_Outgoing_Distributeted#getDocumentsOpened_Spreadsheet_Msx()
     */
    @Override
    public long getDocumentsOpened_Spreadsheet_OOXML() {
        return m_statistics.getDocumentsOpened(DocumentType.SPREADSHEET, OpenType.MSX);
    }

    /* (non-Javadoc)
     * @see com.openexchange.office.getDocumentsOperations_Spreadsheet_Outgoing_Distributeted#getDocumentsOpened_Spreadsheet_Binary()
     */
    @Override
    public long getDocumentsOpened_Spreadsheet_Binary() {
        return m_statistics.getDocumentsOpened(DocumentType.SPREADSHEET, OpenType.BINARY);
    }

    /* (non-Javadoc)
     * @see com.openexchange.office.getDocumentsOperations_Spreadsheet_Outgoing_Distributeted#getDocumentsOpened_Spreadsheet_Odf()
     */
    @Override
    public long getDocumentsOpened_Spreadsheet_ODF() {
        return m_statistics.getDocumentsOpened(DocumentType.SPREADSHEET, OpenType.ODF);
    }

    /* (non-Javadoc)
     * @see com.openexchange.office.getDocumentsOperations_Spreadsheet_Outgoing_Distributeted#getDocumentsClosed_Spreadsheet_Total()
     */
    @Override
    public long getDocumentsClosed_Spreadsheet_Total() {
        return m_statistics.getDocumentsClosed(DocumentType.SPREADSHEET, CloseType.CLOSE);
    }

    /* (non-Javadoc)
     * @see com.openexchange.office.getDocumentsOperations_Spreadsheet_Outgoing_Distributeted#getDocumentsClosed_Spreadsheet_Timeout()
     */
    @Override
    public long getDocumentsClosed_Spreadsheet_Timeout() {
        return m_statistics.getDocumentsClosed(DocumentType.SPREADSHEET, CloseType.TIMEOUT);
    }

    /* (non-Javadoc)
     * @see com.openexchange.office.IInformation#getDocumentsSaved_Spreadsheet_Total()
     */
    @Override
    public long getDocumentsSaved_Spreadsheet_Total() {
        return m_statistics.getDocumentsSaved(DocumentType.SPREADSHEET, null);
    }

    /* (non-Javadoc)
     * @see com.openexchange.office.getDocumentsOperations_Spreadsheet_Outgoing_Distributeted#getDocumentsSaved_Spreadsheet_Close()
     */
    @Override
    public long getDocumentsSaved_Spreadsheet_Close() {
        return m_statistics.getDocumentsSaved(DocumentType.SPREADSHEET, SaveType.CLOSE);
    }

    /* (non-Javadoc)
     * @see com.openexchange.office.getDocumentsOperations_Spreadsheet_Outgoing_Distributeted#getDocumentsSaved_Spreadsheet_100ops()
     */
    @Override
    public long getDocumentsSaved_Spreadsheet_100ops() {
        return m_statistics.getDocumentsSaved(DocumentType.SPREADSHEET, SaveType.OPS_100);
    }

    /* (non-Javadoc)
     * @see com.openexchange.office.getDocumentsOperations_Spreadsheet_Outgoing_Distributeted#getDocumentsSaved_Spreadsheet_15min()
     */
    @Override
    public long getDocumentsSaved_Spreadsheet_15mins() {
        return m_statistics.getDocumentsSaved(DocumentType.SPREADSHEET, SaveType.OPS_15MINS);
    }

    /* (non-Javadoc)
     * @see com.openexchange.office.getDocumentsOperations_Spreadsheet_Outgoing_Distributeted#getDocumentOperations_Spreadsheet_Incoming()
     */
    @Override
    public long getDocumentsOperations_Spreadsheet_Incoming() {
        return m_statistics.getDocumentsOperations(DocumentType.SPREADSHEET, OperationsType.INCOMING);
    }

    /* (non-Javadoc)
     * @see com.openexchange.office.getDocumentsOperations_Spreadsheet_Outgoing_Distributeted#getDocumentOperations_Spreadsheet_Outgoing_Distributeted()
     */
    @Override
    public long getDocumentsOperations_Spreadsheet_Distributed() {
        return m_statistics.getDocumentsOperations(DocumentType.SPREADSHEET, OperationsType.DISTRIBUTED);
    }

    // - IMonitoring Presentation ----------------------------------------------

    /* (non-Javadoc)
     * @see com.openexchange.office.getDocumentsOperations_Presentation_Outgoing_Distributeted#getDocumentsOpened_Presentation_Total()
     */
    @Override
    public long getDocumentsOpened_Presentation_Total() {
        return m_statistics.getDocumentsOpened(DocumentType.PRESENTATION, null);
    }

    /* (non-Javadoc)
     * @see com.openexchange.office.getDocumentsOperations_Presentation_Outgoing_Distributeted#getDocumentsOpened_Presentation_Msx()
     */
    @Override
    public long getDocumentsOpened_Presentation_OOXML() {
        return m_statistics.getDocumentsOpened(DocumentType.PRESENTATION, OpenType.MSX);
    }

    /* (non-Javadoc)
     * @see com.openexchange.office.getDocumentsOperations_Presentation_Outgoing_Distributeted#getDocumentsOpened_Presentation_Binary()
     */
    @Override
    public long getDocumentsOpened_Presentation_Binary() {
        return m_statistics.getDocumentsOpened(DocumentType.PRESENTATION, OpenType.BINARY);
    }

    /* (non-Javadoc)
     * @see com.openexchange.office.getDocumentsOperations_Presentation_Outgoing_Distributeted#getDocumentsOpened_Presentation_Odf()
     */
    @Override
    public long getDocumentsOpened_Presentation_ODF() {
        return m_statistics.getDocumentsOpened(DocumentType.PRESENTATION, OpenType.ODF);
    }

    /* (non-Javadoc)
     * @see com.openexchange.office.getDocumentsOperations_Presentation_Outgoing_Distributeted#getDocumentsClosed_Presentation_Total()
     */
    @Override
    public long getDocumentsClosed_Presentation_Total() {
        return m_statistics.getDocumentsClosed(DocumentType.PRESENTATION, CloseType.CLOSE);
    }

    /* (non-Javadoc)
     * @see com.openexchange.office.getDocumentsOperations_Presentation_Outgoing_Distributeted#getDocumentsClosed_Presentation_Timeout()
     */
    @Override
    public long getDocumentsClosed_Presentation_Timeout() {
        return m_statistics.getDocumentsClosed(DocumentType.PRESENTATION, CloseType.TIMEOUT);
    }

    /* (non-Javadoc)
     * @see com.openexchange.office.IInformation#getDocumentsSaved_Presentation_Total()
     */
    @Override
    public long getDocumentsSaved_Presentation_Total() {
        return m_statistics.getDocumentsSaved(DocumentType.PRESENTATION, null);
    }

    /* (non-Javadoc)
     * @see com.openexchange.office.getDocumentsOperations_Presentation_Outgoing_Distributeted#getDocumentsSaved_Presentation_Close()
     */
    @Override
    public long getDocumentsSaved_Presentation_Close() {
        return m_statistics.getDocumentsSaved(DocumentType.PRESENTATION, SaveType.CLOSE);
    }

    /* (non-Javadoc)
     * @see com.openexchange.office.getDocumentsOperations_Presentation_Outgoing_Distributeted#getDocumentsSaved_Presentation_100ops()
     */
    @Override
    public long getDocumentsSaved_Presentation_100ops() {
        return m_statistics.getDocumentsSaved(DocumentType.PRESENTATION, SaveType.OPS_100);
    }

    /* (non-Javadoc)
     * @see com.openexchange.office.getDocumentsOperations_Presentation_Outgoing_Distributeted#getDocumentsSaved_Presentation_15min()
     */
    @Override
    public long getDocumentsSaved_Presentation_15mins() {
        return m_statistics.getDocumentsSaved(DocumentType.PRESENTATION, SaveType.OPS_15MINS);
    }

    /* (non-Javadoc)
     * @see com.openexchange.office.getDocumentsOperations_Presentation_Outgoing_Distributeted#getDocumentOperations_Presentation_Incoming()
     */
    @Override
    public long getDocumentsOperations_Presentation_Incoming() {
        return m_statistics.getDocumentsOperations(DocumentType.PRESENTATION, OperationsType.INCOMING);
    }

    /* (non-Javadoc)
     * @see com.openexchange.office.getDocumentsOperations_Presentation_Outgoing_Distributeted#getDocumentOperations_Presentation_Outgoing_Distributeted()
     */
    @Override
    public long getDocumentsOperations_Presentation_Distributed() {
        return m_statistics.getDocumentsOperations(DocumentType.PRESENTATION, OperationsType.DISTRIBUTED);
    }

	@Override
	public long getDocumentsStarted_RemotePresenter_Total() {
		return m_statistics.getDocumentsStarted_RemotePresenterTotal();
	}

	@Override
	public long getDocumentsOpened_Presenter_Total() {
		return m_statistics.getDocumentsOpenedPresenterTotal();
	}

	@Override
	public long getDocumentsClosed_Presenter_Total() {
		return m_statistics.getDocumentsClosedPresenterTotal();
	}

	@Override
	public long getDocumentsClosed_Presenter_Timeout() {
		return m_statistics.getDocumentsClosedPresenterTimeout();
	}

	@Override
	public long getDocumentsOperations_Presenter_Incoming() {
		return m_statistics.getDocumentsOperationsPresenterIncoming();
	}

	@Override
	public long getDocumentsOperations_Presenter_Distributed() {
		return m_statistics.getDocumentsOperationsPresenterDistributed();
	}

	@Override
	public long getRestoreDocsOpsMsgs_StoredLastCycle() {
		return m_statistics.getRestoreDocOpsMsgStored();
	}

	@Override
	public long getRestoreDocsOpsMsgs_StoreTimeNeededLastCycle() {
		return m_statistics.getRestoreDocOpsMsgStoredTime();
	}

	@Override
	public long getBackgroundSave_SavedLastCycle() {
		return m_statistics.getBackgroundSaveLastCycleDocCount();
	}

	@Override
	public long getBackgroundSave_TimeNeededLastCycle() {
		return m_statistics.getBackgroundSaveLastCycleTime();
	}

	@Override
	public long getBackgroundSave_AverageTimePerDoc() {
		return m_statistics.getBackgroundSaveAverageTimePerDoc();
	}

	@Override
	public long getBackgroundSave_Setting_MaxTimeForSave() {
		return m_statistics.getBackgroundSaveMaxTimeForSave();
	}

	@Override
	public long getBackgroundSave_Setting_MinTimeForFasterSave() {
		return m_statistics.getBackgroundSaveMinTimeFasterSave();
	}

	@Override
	public long getBackgroundSave_Setting_MinNumOfMessagesForSave_Text() {
		return m_statistics.getBackgroundSaveMinNumOfMsgForFasterSaveText();
	}

	@Override
	public long getBackgroundSave_Setting_MinNumOfMessagesForSave_Spreadsheet() {
		return m_statistics.getBackgroundSaveMinNumOfMsgForFasterSaveSpreadsheet();
	}

	@Override
	public long getBackgroundSave_Setting_MinNumOfMessagesForSave_Presentation() {
		return m_statistics.getBackgroundSaveMinNumOfMsgForFasterSavePresentation();
	}

	@Override
	public long getRestoreDocsManagedFiles_RemovedLastCycle() {
		return m_statistics.getRestoreDocManagedFilesRemovedLastCycle();
	}

	@Override
	public long getRestoreDocsManagedFiles_InRemoveQueue() {
		return m_statistics.getRestoreDocManagedFilesInRemoveQueue();
	}

	@Override
	public long getRestoreDocsManagedFiles_RuntimeLastCycle() {
		return m_statistics.getRestoreDocManagedFilesRuntimeLastCycle();
	}

	@Override
	public long getRestoreDocsManagedFiles_Current() {
		return m_statistics.getRestoreDocManagedFilesCurrent();
	}

	// - IMonitoringOT ---------------------------------------------------------

    @Override
    public long getDocumentsOperations_Total_ServerTransformed() {
        return m_statistics.getTransformationCount(null);
    }

    @Override
    public long getDocumentsOperations_Text_ServerTransformed() {
        return m_statistics.getTransformationCount(DocumentType.TEXT);
    }

    @Override
    public long getDocumentsOperations_Spreadsheet_ServerTransformed() {
        return m_statistics.getTransformationCount(DocumentType.SPREADSHEET);
    }

    @Override
    public long getDocumentsOperations_Presentation_ServerTransformed() {
        return m_statistics.getTransformationCount(DocumentType.PRESENTATION);
    }

    // - Implementation

    /**
     * @return The statistics object, all office runtime information is collected at
     */
    public @NonNull Statistics getStatistics() {
        return m_statistics;
    }

    /**
     * Terminating all internally used threads and freeing resources
     */
    public void shutdown() {
        if (null != m_statistics) {
            m_statistics.shutdown();
        }
    }

    // - Members ---------------------------------------------------------------

    final private Statistics m_statistics;
}

