/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */
package com.openexchange.office.message;

import org.apache.commons.lang3.Validate;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;
import com.openexchange.exception.OXException;
import com.openexchange.office.tools.error.ErrorCode;
import com.openexchange.office.tools.user.UserData;
import com.openexchange.server.services.ServerServiceRegistry;
import com.openexchange.session.Session;
import com.openexchange.sessiond.SessiondService;
import com.openexchange.tools.session.ServerSession;
import com.openexchange.tools.session.ServerSessionAdapter;


public class MessageHelper {

    public final static String KEY_SESSION = "session";
    public final static String KEY_RESOURCE = "resource";


    // - Members ---------------------------------------------------------------
    static protected final org.apache.commons.logging.Log LOG = com.openexchange.log.LogFactory.getLog(MessageHelper.class);

    /**
     * Creates the final JSON result object to be sent to the client-side.
     * This function is for special purpose where a method already created
     * a JSON result object with error data. In that case finalize shall not
     * overwrite the first error data.
     *
     * @param jsonRequest
     *  The request to be processed.
     *
     * @param jsonExtra
     *  Extra data to be added to the JSON result object.
     *
     * @return
     *  The final JSON result object to be sent to the client-side.
     */
    static public JSONObject finalizeJSONResult(JSONObject tmpResult, JSONObject jsonRequest, JSONObject jsonExtra) {
        if (null != tmpResult) {
            try {
                // add unique message id, if given in request
                if ((null != jsonRequest) && jsonRequest.has(MessagePropertyKey.KEY_UNIQUE_ID)) {
                    tmpResult.put(MessagePropertyKey.KEY_UNIQUE_ID, jsonRequest.get(MessagePropertyKey.KEY_UNIQUE_ID));
                }
            } catch (JSONException e) {
                LOG.error("RT2: getJSfinalizeJSONResult catches exception trying to to put unique id to the JSON object. ", e);
            }

            // add extra data, if set
            if (null != jsonExtra) {
                OperationHelper.appendJSON(tmpResult, jsonExtra);
            }
        }

        return tmpResult;
    }

    /**
     * Creates the final JSON result object to be sent to the client-side.
     *
     * @param errorCode The result of the request as error code.
     * @param jsonRequest The request to be processed.
     * @param jsonExtra Extra data to be added to the JSON result object.
     *
     * @return The final JSON result object to be sent to the client-side.
     */
    static public JSONObject finalizeJSONResult(ErrorCode errorCode, JSONObject jsonRequest, JSONObject jsonExtra) {
        JSONObject jsonResult = new JSONObject();
        return finalizeJSONResult(jsonResult, errorCode, jsonRequest, jsonExtra);
    }

    /**
     * Creates the final JSON result object to be sent to the client-side.
     *
     * @param errorCode The result of the request as error code.
     * @return The final JSON result object to be sent to the client-side.
     */
    static public JSONObject finalizeJSONResult(final ErrorCode errorCode) {
        JSONObject jsonResult = new JSONObject();
        return finalizeJSONResult(jsonResult, errorCode, null, null);
    }

    /**
     * Creates the final JSON result object to be sent to the client-side.
     *
     * @param tmpResult
     *  A json object which is used to merge the following data into it.
     *
     * @param errorCode
     *  The result of the request as error code.
     *
     * @param jsonRequest
     *  The request to be processed.
     *
     * @param jsonExtra
     *  Extra data to be added to the JSON result object.
     *
     * @return
     *  The final JSON result object to be sent to the client-side. The object
     *  is the same as tmpResult extended with the properties provided as
     *  arguments.
     */
    static public JSONObject finalizeJSONResult(JSONObject tmpResult, ErrorCode errorCode, JSONObject jsonRequest, JSONObject jsonExtra) {
        if (null != tmpResult) {
            try {
                // add unique message id, if given in request
                if ((null != jsonRequest) && jsonRequest.has(MessagePropertyKey.KEY_UNIQUE_ID)) {
                    tmpResult.put(MessagePropertyKey.KEY_UNIQUE_ID, jsonRequest.get(MessagePropertyKey.KEY_UNIQUE_ID));
                }
                tmpResult.put(MessagePropertyKey.KEY_ERROR_DATA, errorCode.getAsJSON());
            } catch (JSONException e) {
                LOG.error("RT2: getJSfinalizeJSONResult catches exception trying to to put unique id to the JSON object. ", e);
            }

            // add extra data, if set
            if (null != jsonExtra) {
                OperationHelper.appendJSON(tmpResult, jsonExtra);
            }
        }

        return tmpResult;
    }

    /**
     * Checks the json result object is empty and sets the "hasErrors"
     * property to indicate an errornous state.
     *
     * @param jsonResult
     *  A json object containing the result of a request which should be
     *  checked.
     *
     * @return
     *  The jsonResult which contains a "hasError" property if it doesn't
     *  contain any properties.
     */
    static public JSONObject checkJSONResult(JSONObject jsonResult) {
        if ((null != jsonResult) && jsonResult.isEmpty()) {
            try {
                jsonResult.put(MessagePropertyKey.KEY_HAS_ERRORS, true);
            } catch (JSONException e) {
                LOG.error("RT2: checkJSONResult catches exception trying to to put hasErrors to the JSON object. ", e);
            }
        }

        return jsonResult;
    }

}
