package com.openexchange.office.filter.ooxml.xlsx.components;
/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */



import javax.xml.bind.JAXBException;
import org.docx4j.XmlUtils;
import org.docx4j.dml.CTRegularTextRun;
import org.docx4j.dml.CTTextCharacterProperties;
import org.docx4j.dml.CTTextLineBreak;
import org.docx4j.dml.CTTextParagraph;
import org.docx4j.dml.CTTextParagraphProperties;
import org.docx4j.dml.ITextCharacterProperties;
import org.docx4j.jaxb.Context;
import org.docx4j.openpackaging.exceptions.InvalidFormatException;
import org.docx4j.openpackaging.exceptions.PartUnrecognisedException;
import org.json.JSONException;
import org.json.JSONObject;
import org.xlsx4j.sml.CTTextField;
import com.openexchange.office.filter.api.FilterException;
import com.openexchange.office.filter.api.OCKey;
import com.openexchange.office.filter.core.DLNode;
import com.openexchange.office.filter.core.IContentAccessor;
import com.openexchange.office.filter.core.DLList;
import com.openexchange.office.filter.core.SplitMode;
import com.openexchange.office.filter.core.component.Component;
import com.openexchange.office.filter.core.component.ComponentContext;
import com.openexchange.office.filter.ooxml.OfficeOpenXMLOperationDocument;
import com.openexchange.office.filter.ooxml.components.OfficeOpenXMLComponent;
import com.openexchange.office.filter.ooxml.components.IParagraph;
import com.openexchange.office.filter.ooxml.drawingml.DMLHelper;

public class ParagraphComponent extends XlsxComponent implements IParagraph {

    final CTTextParagraph paragraph;

    public ParagraphComponent(ComponentContext<OfficeOpenXMLOperationDocument> parentContext, DLNode<Object> _node, int _componentNumber) {
        super(parentContext, _node, _componentNumber);

        paragraph = (CTTextParagraph)getObject();
    }

    public CTTextParagraph getParagraph() {
        return paragraph;
    }

    @Override
    public Component<OfficeOpenXMLOperationDocument> getNextChildComponent(ComponentContext<OfficeOpenXMLOperationDocument> previousChildContext, Component<OfficeOpenXMLOperationDocument> previousChildComponent) {
        final DLNode<Object> paragraphNode = getNode();
        final DLList<Object> nodeList = (DLList<Object>)((IContentAccessor)paragraphNode.getData()).getContent();
        final int nextComponentNumber = previousChildComponent!=null?previousChildComponent.getNextComponentNumber():0;
        DLNode<Object> childNode = previousChildContext!=null ? previousChildContext.getNode().getNext() : nodeList.getFirstNode();

        Component<OfficeOpenXMLOperationDocument> nextComponent = null;
        for(; nextComponent==null&&childNode!=null; childNode = childNode.getNext()) {
            final Object o = getContentModel(childNode, paragraphNode.getData());
            if(o instanceof CTRegularTextRun) {
                if(((CTRegularTextRun)o).getT()!=null&&!((CTRegularTextRun)o).getT().isEmpty()) {
                    nextComponent = new TextComponent(this, childNode, nextComponentNumber);
                }
            }
            else if(o instanceof CTTextField) {
                nextComponent = new TextFieldComponent(this, childNode, nextComponentNumber);
            }
            else if(o instanceof CTTextLineBreak) {
                nextComponent = new HardBreakComponent(this, childNode, nextComponentNumber);
            }
       }
        return nextComponent;
    }

    @Override
    public void insertText(int textPosition, String text, JSONObject attrs) throws Exception {

        if(text.length()>0) {
            Component<OfficeOpenXMLOperationDocument> childComponent = getNextChildComponent(null, null);
            Component<OfficeOpenXMLOperationDocument> cRet = null;

            if(childComponent!=null) {
                if(textPosition>0) {
                    childComponent = childComponent.getComponent(textPosition-1);
                }
                // check if the character could be inserted into an existing text:
                if(childComponent instanceof TextComponent) {
                    final CTRegularTextRun t = (CTRegularTextRun)childComponent.getObject();
                    final StringBuffer s = new StringBuffer(t.getT());
                    s.insert(textPosition-((TextComponent)childComponent).getComponentNumber(), text);
                    t.setT(s.toString());
                    cRet = childComponent;
                }
                else {
                    final CTRegularTextRun newRun = Context.getDmlObjectFactory().createCTRegularTextRun();
                    newRun.setParent(getObject());
                    newRun.setT(text);
                    (paragraph.getContent()).addNode(childComponent.getNode(), new DLNode<Object>(newRun), textPosition == 0 ? true : false);

                    CTTextCharacterProperties referenceRPr = null;
                    if(childComponent.getObject() instanceof ITextCharacterProperties) {
                        referenceRPr = ((ITextCharacterProperties)childComponent.getObject()).getRPr(false);
                    }
                    if(referenceRPr!=null) {
                        final CTTextCharacterProperties newRPr = XmlUtils.deepCopy(referenceRPr, operationDocument.getPackage());
                        if(newRPr!=null) {
                            newRun.setRPr(newRPr);
                        }
                    }
                    if(textPosition==0) {
                        cRet = getNextChildComponent(null, null);
                    }
                    else {
                        cRet = childComponent.getNextComponent();
                    }
                }
          }
          else {
              // the paragraph is empty, we have to create R and its text
              final CTRegularTextRun newRun = Context.getDmlObjectFactory().createCTRegularTextRun();
              newRun.setParent(getObject());
              paragraph.getContent().add(newRun);
              if(paragraph.getEndParaRPr(false)!=null) {
                  newRun.setRPr(XmlUtils.deepCopy(paragraph.getEndParaRPr(true), operationDocument.getPackage()));
              }
              else if(paragraph.getPPr(false)!=null&&paragraph.getPPr(false).getDefRPr(false)!=null) {
                  newRun.setRPr(XmlUtils.deepCopy(paragraph.getPPr(false).getDefRPr(true), operationDocument.getPackage()));
              }
              newRun.setT(text);
              cRet = getNextChildComponent(null, null);
            }
            ((CTRegularTextRun)cRet.getObject()).getRPr(true).setDirty(Boolean.valueOf(false));
            if(attrs!=null) {
                cRet.splitStart(textPosition, SplitMode.ATTRIBUTE);
                cRet.splitEnd(textPosition+text.length()-1, SplitMode.ATTRIBUTE);
                cRet.applyAttrsFromJSON(attrs);
            }
        }
    }

    @Override
    public void splitParagraph(int textPosition) {
        // creating and inserting our new paragraph
        final CTTextParagraph destParagraph = Context.getDmlObjectFactory().createCTTextParagraph();
        destParagraph.setParent(paragraph.getParent());
        final DLNode<Object> destParagraphNode = new DLNode<Object>(destParagraph);
        ((DLList<Object>)((IContentAccessor)paragraph.getParent()).getContent()).addNode(getNode(), destParagraphNode, textPosition==0);

        CTTextCharacterProperties lastRPr = null;
        if(textPosition>0) {
            // splitting the paragraph
            Component<OfficeOpenXMLOperationDocument> childComponent = getChildComponent(textPosition-1);
            if(childComponent!=null) {
                lastRPr = ((ITextCharacterProperties)childComponent.getObject()).getRPr(false);
                childComponent.splitEnd(textPosition-1, SplitMode.DELETE);

                // moving text runs into the new paragraph
                DLNode<Object> sourceNode = childComponent.getNode().getNext();
                if(sourceNode!=null) {
                    paragraph.getContent().moveNodes(sourceNode, -1, destParagraph.getContent(), null, true, destParagraph);
                }
            }
        }

        final CTTextParagraphProperties sourceParagraphProperties = paragraph.getPPr(false);
        if(sourceParagraphProperties!=null){
            final CTTextParagraphProperties destParagraphProperties = XmlUtils.deepCopy(sourceParagraphProperties, getOperationDocument().getPackage());
            destParagraph.setPPr(destParagraphProperties);
        }

        // taking care of paragraph attributes
        if(lastRPr!=null) {
            // if available, we have to get the character attributes from the last textrun
            destParagraph.getPPr(true).setDefRPr(XmlUtils.deepCopy(lastRPr, getOperationDocument().getPackage()));
        }
    }

    @Override
    public void mergeParagraph() {
        final Component<OfficeOpenXMLOperationDocument> nextParagraphComponent = getNextComponent();
        if(nextParagraphComponent instanceof ParagraphComponent) {
            final DLList<Object> sourceContent = ((CTTextParagraph)nextParagraphComponent.getObject()).getContent();
            final DLList<Object> destContent = paragraph.getContent();
            if(sourceContent.size()>0) {
                sourceContent.moveNodes(destContent, paragraph);
            }
            DLNode<Object> parentContextNode = nextParagraphComponent.getParentContext().getNode();
            ((IContentAccessor)parentContextNode.getData()).getContent().remove(nextParagraphComponent.getObject());
        }
    }

    @Override
    public void applyAttrsFromJSON(JSONObject attrs)
        throws JSONException, InvalidFormatException, PartUnrecognisedException {

        DMLHelper.applyTextParagraphPropertiesFromJson(paragraph.getPPr(true), attrs, operationDocument);
    }

    @Override
    public JSONObject createJSONAttrs(JSONObject attrs)
            throws JSONException, FilterException {

        DMLHelper.createJsonFromTextParagraphProperties(operationDocument, attrs, paragraph.getPPr(false));
        if(paragraph.getEndParaRPr(false)!=null&&paragraph.getContent().isEmpty()) {
            DMLHelper.createJsonFromTextCharacterProperties(getOperationDocument(), attrs, paragraph.getEndParaRPr(false));
        }
        return attrs;
    }

    public void checkSetLevel(JSONObject attrs) {
        if(attrs!=null) {
            final JSONObject paragraphAttrs = attrs.optJSONObject(OCKey.PARAGRAPH.value());
            if(paragraphAttrs!=null) {
                final Object l = paragraphAttrs.opt(OCKey.LEVEL.value());
                if(l!=null) {
                    paragraph.getPPr(true).setLvl(l instanceof Number ? ((Number)l).intValue() : 0);
                }
            }
        }
    }

    public int getLevel() {
        final CTTextParagraphProperties properties = paragraph.getPPr(false);
        final Integer lvl = properties!=null ? properties.getLvl() : null;
        return lvl!=null ? lvl.intValue() : 0;
    }

    @Override
    public Component<OfficeOpenXMLOperationDocument> insertChildComponent(int textPosition, JSONObject attrs, Type childType) throws Exception {

        Object newChild = null;

        CTTextCharacterProperties textCharacterProperties = null;
        if(childType == Type.HARDBREAK_DEFAULT) {
            newChild = Context.getDmlObjectFactory().createCTTextLineBreak();
        }
        else if(childType == Type.SIMPLE_FIELD) {
            newChild = Context.getDmlObjectFactory().createCTTextField();
        }
        Component<OfficeOpenXMLOperationDocument> childComponent = getNextChildComponent(null, null);
        if(childComponent!=null) {
            if(textPosition>0) {
                childComponent = childComponent.getComponent(textPosition-1);
                childComponent.splitEnd(textPosition-1, SplitMode.ATTRIBUTE);
            }
            if(childComponent.getObject() instanceof ITextCharacterProperties) {
                textCharacterProperties = ((ITextCharacterProperties)childComponent.getObject()).getRPr(false);
            }
            final IContentAccessor parentContextObject = (IContentAccessor)childComponent.getParentContext().getNode().getData();
            ((DLList<Object>)parentContextObject.getContent()).addNode(childComponent.getNode(), new DLNode<Object>(newChild), textPosition==0);
        }
        else {
            paragraph.getContent().addNode(new DLNode<Object>(newChild));
            textCharacterProperties = paragraph.getEndParaRPr(false);
        }
        if(textPosition>0) {
            childComponent = childComponent.getNextComponent();
        }
        else {
            childComponent = getNextChildComponent(null, null);
        }
        if(textCharacterProperties!=null&&childComponent.getObject() instanceof ITextCharacterProperties) {
            ((ITextCharacterProperties)childComponent.getObject()).setRPr(XmlUtils.deepCopy(textCharacterProperties, operationDocument.getPackage()));
        }
        if(attrs!=null) {
            childComponent.applyAttrsFromJSON(attrs);
        }
        return childComponent;
    }
}
