/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.filter.ooxml.pptx.components;

import org.docx4j.dml.CTNonVisualDrawingProps;
import org.docx4j.dml.CTTextBody;
import org.docx4j.dml.CTTextParagraph;
import org.docx4j.jaxb.Context;
import org.json.JSONException;
import org.json.JSONObject;
import org.pptx4j.pml.Shape;
import org.pptx4j.pml.Shape.NvSpPr;
import com.openexchange.office.filter.api.OCKey;
import com.openexchange.office.filter.core.DLNode;
import com.openexchange.office.filter.core.IContentAccessor;
import com.openexchange.office.filter.core.DLList;
import com.openexchange.office.filter.core.component.Child;
import com.openexchange.office.filter.core.component.Component;
import com.openexchange.office.filter.core.component.ComponentContext;
import com.openexchange.office.filter.ooxml.OfficeOpenXMLOperationDocument;
import com.openexchange.office.filter.ooxml.components.OfficeOpenXMLComponent;
import com.openexchange.office.filter.ooxml.components.IShapeType;
import com.openexchange.office.filter.ooxml.components.ShapeType;
import com.openexchange.office.filter.ooxml.drawingml.DMLHelper;
import com.openexchange.office.filter.ooxml.pptx.tools.PMLShapeHelper;

public class ShapeComponent extends PptxComponent implements IShapeComponent, IShapeType {

	final Shape shape;

	public ShapeComponent(ComponentContext<OfficeOpenXMLOperationDocument> parentContext, DLNode<Object> _node, int _componentNumber) {
        super(parentContext, _node, _componentNumber);

        this.shape = (Shape)getObject();
	}

    @Override
	public Component<OfficeOpenXMLOperationDocument> getNextChildComponent(ComponentContext<OfficeOpenXMLOperationDocument> previousChildContext, Component<OfficeOpenXMLOperationDocument> previousChildComponent) {
        final DLNode<Object> shapeNode = getNode();
        final DLList<Object> nodeList = (DLList<Object>)((IContentAccessor)shapeNode.getData()).getContent();
        final int nextComponentNumber = previousChildComponent!=null?previousChildComponent.getNextComponentNumber():0;
        DLNode<Object> childNode = previousChildContext!=null ? previousChildContext.getNode().getNext() : nodeList.getFirstNode();

        OfficeOpenXMLComponent nextComponent = null;
        for(; nextComponent==null&&childNode!=null; childNode = childNode.getNext()) {
            final Object o = getContentModel(childNode, ((Shape)shapeNode.getData()).getTxBody(false));
            if(o instanceof CTTextParagraph) {
            	nextComponent = new ParagraphComponent(this, childNode, nextComponentNumber);
            }
        }
        return nextComponent;
    }

    @Override
    public Component<OfficeOpenXMLOperationDocument> insertChildComponent(ComponentContext<OfficeOpenXMLOperationDocument> parentContext, DLNode<Object> contextNode, int number, Component<OfficeOpenXMLOperationDocument> child, Component.Type type, JSONObject attrs) {

    	DLList<Object> DLList = (DLList<Object>)((IContentAccessor)contextNode.getData()).getContent();
        DLNode<Object> referenceNode = child!=null ? child.getNode() : null;

        switch(type) {
        	case PARAGRAPH: {
	            final Child newChild = Context.getDmlObjectFactory().createCTTextParagraph();
	            newChild.setParent(contextNode.getData());
	            final DLNode<Object> newChildNode = new DLNode<Object>(newChild);
	            DLList.addNode(referenceNode, newChildNode, true);
	            return new ParagraphComponent(parentContext, newChildNode, number);
        	}
        	default : {
                throw new UnsupportedOperationException();
            }
        }
    }

    @Override
    public void applyAttrsFromJSON(JSONObject attrs) throws Exception {

    	PMLShapeHelper.applyAttrsFromJSON(operationDocument, attrs, shape);
    	final JSONObject listStyle = attrs.optJSONObject(OCKey.LIST_STYLE.value());
    	if(listStyle!=null) {
    		DMLHelper.applyTextListStyleFromJson(shape.getTxBody(true), listStyle, operationDocument);
    	}
		com.openexchange.office.filter.ooxml.drawingml.Shape.applyAttrsFromJSON(operationDocument, attrs, shape, !(getParentComponent() instanceof ShapeGroupComponent));
    }

    @Override
    public JSONObject createJSONAttrs(JSONObject attrs)
    	throws JSONException {

    	PMLShapeHelper.createJSONAttrs(attrs, shape);

    	final CTTextBody txBody = shape.getTxBody(false);
        if(txBody!=null) {
        	final JSONObject style = DMLHelper.createJsonFromTextListStyle(getOperationDocument(), txBody.getLstStyle());
        	if(style!=null&&!style.isEmpty()) {
        		attrs.put(OCKey.LIST_STYLE.value(), style);
        	}
        }
        com.openexchange.office.filter.ooxml.drawingml.Shape.createJSONAttrs(getOperationDocument(), attrs, shape, !(getParentComponent() instanceof ShapeGroupComponent));
        return attrs;
    }

	@Override
	public ShapeType getType() {
		return ShapeType.SHAPE;
	}

	@Override
	public Integer getId() {
		final CTNonVisualDrawingProps nvdProps = shape.getNonVisualDrawingProperties(false);
		return nvdProps!=null ? nvdProps.getId() : null;
	}

	@Override
	public boolean isPresentationObject() {
		final NvSpPr nvSpPr = shape.getNvSpPr();
		return nvSpPr!=null&&nvSpPr.getNvPr()!=null&&nvSpPr.getNvPr().getPh()!=null ? true : false;
	}

	public CTTextBody getTextBody(boolean forceCreate) {
		return shape.getTxBody(true);
	}
}
