/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.filter.core.spreadsheet;

import java.util.ArrayList;
import java.util.List;
import org.apache.commons.lang3.mutable.MutableInt;

public final class RowRef extends Number implements Comparable<RowRef> {

    public static final int   MIN_VALUE = 0x0;

    public static final int   MAX_VALUE = 0xFFFFF;

    private final int row;

    public RowRef(int row) {
        this.row = row;
    }

    /*
     * The offset gives the position up from where the row should be parsed.
     *
     * After parsing the offset points to last parsed character + 1 or -1 if
     * no row could be parsed. If nothing could be parsed the row is initialized
     * with 0.
     *
     */
    public RowRef(String row, MutableInt offset) {

        int y = 0;
        int numbers = 0;
        int numberStartIndex = offset.intValue();

        if(numberStartIndex>=0) {
            if(numberStartIndex<row.length()&&row.charAt(numberStartIndex) == '$') {
                numberStartIndex++;
            }
            while(numbers+numberStartIndex<row.length()) {
                char character = row.charAt(numberStartIndex+numbers);
                if((character<'0')||(character>'9')) {
                    break;
                }
                numbers++;
            }
            if(numbers<=0||numbers>=8) {
                offset.setValue(-1);
            }
            else {
                int i;
                for(i = 0; i < numbers; i++) {
                    y *= 10;
                    y += row.charAt(i+numberStartIndex)-'0';
                }
                offset.setValue(i+numberStartIndex);
                y--;
            }
        }
        this.row = y;
    }

    @Override
    public int hashCode() {
        return row;
    }

    @Override
    public String toString() {
        return Long.toString(row + 1);
    }

    public static Interval<RowRef> createInterval(String interval) {
        final MutableInt offset = new MutableInt(0);
        final RowRef min = new RowRef(interval, offset);
        if(offset.intValue()==-1) {
            return null;
        }
        offset.setValue(interval.indexOf(':', offset.intValue()));
        if(offset.intValue()==-1) {
            return new Interval<RowRef>(min, min);
        }
        if(offset.intValue()+1>=interval.length()) {
            return null;
        }
        offset.increment();
        final RowRef max = new RowRef(interval, offset);
        if(offset.intValue()==-1) {
            return null;
        }
        return new Interval<RowRef>(min, max);
    }

    public static List<Interval<RowRef>> createIntervals(String intervals) {
        if(intervals==null) {
            return new ArrayList<Interval<RowRef>>();
        }
        final String[] tokenArray = intervals.split(" ", -1);
        final ArrayList<Interval<RowRef>> list = new ArrayList<Interval<RowRef>>(tokenArray.length);
        for(int i=0; i<tokenArray.length; i++) {
            list.add(createInterval(tokenArray[i]));
        }
        return list;
    }

    @Override
    public boolean equals(Object obj) {
        if (obj instanceof RowRef) {
            return row == ((RowRef)obj).intValue();
        }
        return false;
    }

    @Override
    public int compareTo(RowRef arg0) {
        return Integer.compare(this.row, arg0.row);
    }

    @Override
    public double doubleValue() {
        return row;
    }

    @Override
    public float floatValue() {
        return row;
    }

    @Override
    public int intValue() {
        return row;
    }

    @Override
    public long longValue() {
        return row;
    }
}
