/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.filter.core.spreadsheet;

import java.util.ArrayList;
import java.util.List;
import org.apache.commons.lang3.mutable.MutableInt;

public final class ColumnRef extends Number implements Comparable<ColumnRef>  {

    public static final int   MIN_VALUE = 0x0;

    public static final int   MAX_VALUE = 0x3FFF;

    private final int column;

    public ColumnRef(int column) {
        this.column = column;
    }

    /*
     * The offset gives the position up from where the row should be parsed.
     *
     * After parsing the offset points to last parsed character + 1 or -1 if
     * no column could be parsed. If nothing could be parsed the column is initialized
     * with 0.
     *
     */
    public ColumnRef(String column, MutableInt offset) {

        int x = 0;
        int characters = 0;
        int characterStartIndex = offset.intValue();
        if(characterStartIndex>=0) {
            if(column.charAt(characterStartIndex) == '$') {
                characterStartIndex++;
            }
            while((characters+characterStartIndex)<column.length()) {
                char character = column.charAt(characterStartIndex+characters);
                if((character<'A')||(character>'Z')) {
                    break;
                }
                characters++;
            }
            if(characters<=0||characters>=6) {
                offset.setValue(-1);
            }
            else {

                final int[] exMap = { 1, 26, 676, 17576, 456976 };

                int i, ex = characters - 1;
                x = -1;

                for(i = 0; i < characters; i++) {
                    x +=((column.charAt(i+characterStartIndex)-'A')+1)*exMap[ex-i];
                }
                offset.setValue(i+characterStartIndex);
            }
        }
        this.column = x;
    }

    @Override
    public int hashCode() {
        return column;
    }

    @Override
    public String toString() {
        final StringBuilder builder = new StringBuilder(5);
        int i = column;
        while(i>=0) {
            builder.insert(0, (char)('A' + (i % 26)));
            i /= 26;
            i--;
        }
        return builder.toString();
    }

    public static Interval<ColumnRef> createInterval(String interval) {
        final MutableInt offset = new MutableInt(0);
        final ColumnRef min = new ColumnRef(interval, offset);
        if(offset.intValue()==-1) {
            return null;
        }
        offset.setValue(interval.indexOf(':', offset.intValue()));
        if(offset.intValue()==-1) {
            return new Interval<ColumnRef>(min, min);
        }
        if(offset.intValue()+1>=interval.length()) {
            return null;
        }
        offset.increment();
        final ColumnRef max = new ColumnRef(interval, offset);
        if(offset.intValue()==-1) {
            return null;
        }
        return new Interval<ColumnRef>(min, max);
    }

    public static List<Interval<ColumnRef>> createIntervals(String intervals) {
        if(intervals==null) {
            return new ArrayList<Interval<ColumnRef>>();
        }
        final String[] tokenArray = intervals.split(" ", -1);
        final ArrayList<Interval<ColumnRef>> list = new ArrayList<Interval<ColumnRef>>(tokenArray.length);
        for(int i=0; i<tokenArray.length; i++) {
            list.add(createInterval(tokenArray[i]));
        }
        return list;
    }

    @Override
    public boolean equals(Object obj) {
        if (obj instanceof ColumnRef) {
            return column == ((ColumnRef)obj).intValue();
        }
        return false;
    }

    @Override
    public int compareTo(ColumnRef arg0) {
        return Integer.compare(this.column, arg0.column);
    }

    @Override
    public double doubleValue() {
        return column;
    }

    @Override
    public float floatValue() {
        return column;
    }

    @Override
    public int intValue() {
        return column;
    }

    @Override
    public long longValue() {
        return column;
    }
}
