/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.filter.core;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.openexchange.office.filter.api.DocumentProperties;
import com.openexchange.office.imagemgr.IResourceManager;
import com.openexchange.office.tools.common.memory.MemoryObserver;
import com.openexchange.office.tools.common.memory.MemoryObserver.MemoryListener;
import com.openexchange.session.Session;

public abstract class OperationDocument implements AutoCloseable {

    final protected static Logger log = LoggerFactory.getLogger(OperationDocument.class);

    protected final Session               session;
    protected final IResourceManager      resourceManager;
    protected final DocumentProperties    documentProperties;

    private MemoryObserver              memoryObserver = null;
    private MemoryListener              memoryListener = null;

    protected boolean                   createFinalDocument = false;

    protected OperationDocument(Session session, IResourceManager resourceManager, DocumentProperties documentProperties) {
        this.session = session;
        this.resourceManager = resourceManager;
        this.documentProperties = documentProperties!=null ? documentProperties : new DocumentProperties();

        registerMemoryListener();
    }

    public IResourceManager getResourceManager() {
        return resourceManager;
    }

    public DocumentProperties getDocumentProperties() {
        return documentProperties;
    }

    public String getUserLanguage() {
        return documentProperties.optString(DocumentProperties.PROP_USER_LANGUAGE, "");
    }

    final public void setCreateFinalDocument(boolean createFinalDocument) {
        this.createFinalDocument = createFinalDocument;
    }

    final public void registerMemoryListener() {
        getMemoryObserver();
        if(memoryObserver!=null) {
            if(memoryListener==null) {
                memoryListener = new MemoryListener() {
                    @Override
                    public void memoryTresholdExceeded(long usedMemory, long maxMemory) {
                        notifyLowMemory();
                    }
                };
                memoryObserver.addListener(memoryListener);
            }
            if(memoryObserver.isUsageThresholdExceeded()) {
                notifyLowMemory();
            }
        }
    }

    private void removeMemoryListener() {
        if(memoryListener!=null && memoryObserver!=null) {
            memoryObserver.removeListener(memoryListener);
            memoryListener = null;
        }
        memoryObserver = null;
    }

    abstract protected void notifyLowMemory();

    private MemoryObserver getMemoryObserver () {
        if (memoryObserver == null) {
            memoryObserver = MemoryObserver.getMemoryObserver();
        }
        return memoryObserver;
    }

    @Override
    public void close() {
        removeMemoryListener();
    }
}
