/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.filter.api;

import java.io.InputStream;
import java.util.Map;
import com.openexchange.session.Session;

/**
 * A partitioning importer can provide the operations and meta data
 * of a document stream in parts. The importer must be initialized
 * with the necessary data and meta data, the active part and all
 * remaining parts can be retrieved. The importer must provide the
 * number of parts within the meta data.
 *
 * @author <a href="mailto:carsten.driesner@open-xchange.com">Carsten Driesner
 * @since 7.8.2
 */
public interface IPartImporter {

	/**
	 * Initializes the importer with the necessary data to provide
	 * partition support for the document.
	 *
	 * @param session the session of the client requesting to import the document
	 * @param inputDocument the document stream which must be seekable
	 * @param documentProperties the document properties that specifies certain
	 *                           information to the filter. E.g. how much memory
	 *                           can be used, etc.
	 * @throws FilterException
	 */
    public void initPartitioning(final Session session, final InputStream inputDocument, final DocumentProperties documentProperties)
		throws FilterException;

    /**
     * Provides the meta data for the provided document. One of the meta data
     * properties must specify in how many parts the document will be sliced.
     *
	 * @param documentProperties the document properties that specifies certain
	 *                           information to the filter. E.g. how much memory
	 *                           can be used, etc.
	 *
     * @return a map containing the meta data of the document as properties. The
     *         must contain the property "numberOfParts" which provides the number
     *         of parts including the active part.
     */
    public Map<String, Object> getMetaData(DocumentProperties documentProperties)
    	throws FilterException;

    /**
     * Provides the active part of the provided document stream. It's up to
     * the implementation to decide which is the active part. Normally this
     * would be the active sheet, active slide or the first part of a text
     * document.
     *
	 * @param documentProperties the document properties that specifies certain
	 *                           information to the filter. E.g. how much memory
	 *                           can be used, etc.
	 *
     * @return the active part of the provided document stream. The map must
     *         at least provide the property "operations" which is a JSONArray
     *         containing operations. The array may be empty.
     */
    public Map<String, Object> getActivePart(DocumentProperties documentProperties)
    	throws FilterException;

    /**
     * Provides the next part of the provided document stream. The filter should
     * chose a logical way to provide the next part. E.g. for a text document the next
     * part after the active part would be the following elements, for spreadsheet the
     * following sheet or the previous one and the same would be true for presentation
     * documents.
     *
	 * @param documentProperties the document properties that specifies certain
	 *                           information to the filter. E.g. how much memory
	 *                           can be used, etc.
	 *
     *
     * @return the next part of the provided document stream or null if there is
     *         no part. The map must at least provide the property "operations" 
     *         which is a JSONArray containing operations. The array may be empty.
     */
    public Map<String, Object> getNextPart(DocumentProperties documentProperties)
    	throws FilterException;
}
