/*
 *
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package org.docx4j.openpackaging.parts;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.InputStream;
import java.util.Map;
import javax.xml.bind.JAXBContext;
import javax.xml.bind.JAXBException;
import javax.xml.bind.Marshaller;
import javax.xml.bind.Unmarshaller;
import javax.xml.stream.XMLInputFactory;
import javax.xml.stream.XMLOutputFactory;
import javax.xml.stream.XMLStreamReader;
import javax.xml.transform.Templates;
import javax.xml.transform.stream.StreamResult;
import org.apache.commons.io.IOUtils;
import org.docx4j.XmlUtils;
import org.docx4j.jaxb.Context;
import org.docx4j.jaxb.FragmentPrefixMapper;
import org.docx4j.jaxb.JaxbValidationEventHandler;
import org.docx4j.openpackaging.exceptions.Docx4JException;
import org.docx4j.openpackaging.exceptions.InvalidFormatException;
import org.docx4j.openpackaging.io3.stores.PartStore;

public abstract class DocumentPart<E extends DocumentSerialization> extends SerializationPart<E> {

    public DocumentPart(PartName partName) throws InvalidFormatException {
        super(partName);
        setJAXBContext(Context.getJc());
    }

    public DocumentPart(PartName partName, JAXBContext jc) throws InvalidFormatException {
        super(partName);
        setJAXBContext(jc);
    }

    protected JAXBContext jc;

    public void setJAXBContext(JAXBContext jc) {
        this.jc = jc;
    }

    public JAXBContext getJAXBContext() {
        return jc;
    }

    public Unmarshaller.Listener createUnmarshalListener() {
        return null;
    }

    protected E document = null;

    @Override
    public boolean isUnmarshalled() {
        return document!=null;
    }

    protected abstract E createDocument();

    @Override
    public void setJaxbElement(E document) {
        this.document = document;
    }

    @Override
    public E getJaxbElement() {
        InputStream is = null;
        if (document==null) {
            final PartStore partStore = this.getPackage().getSourcePartStore();
            try {
                final String name = this.partName.getName();
                try {
                    if (partStore!=null) {
                        this.setContentLengthAsLoaded(partStore.getPartSize(name.substring(1)));
                    }
                } catch (UnsupportedOperationException uoe) {
                    //
                }
                is = partStore.loadPart(name.substring(1));
                if (is==null) {
                    log.info(name + " missing from part store");
                } else {
                    log.debug("Lazily unmarshalling " + name);
                    unmarshal(is);
                }
            } catch (Docx4JException e) {
                log.error(e.getMessage(), e);
            } finally {
                IOUtils.closeQuietly(is);
            }
        }
        return document;
    }

    @Override
    public E unmarshal(java.io.InputStream is) {
        try {
            final XMLInputFactory xif = XMLInputFactory.newFactory();
            xif.setProperty(XMLInputFactory.IS_SUPPORTING_EXTERNAL_ENTITIES, false);
            xif.setProperty(XMLInputFactory.SUPPORT_DTD, false);
            try {
                final XMLStreamReader reader = xif.createXMLStreamReader(is);
                document = createDocument();
                document.readObject(reader, this);
            }
            catch (Exception ue) {

                Context.abortOnLowMemory(getPackage());

                final Templates mcPreprocessorXslt = JaxbValidationEventHandler.getMcPreprocessor();
                is.reset();
                final ByteArrayOutputStream bos = new ByteArrayOutputStream();
                final StreamResult streamResult = new StreamResult();
                streamResult.setOutputStream(bos);
                XmlUtils.transform(is, mcPreprocessorXslt, null, streamResult);
                final XMLStreamReader reader = xif.createXMLStreamReader(new ByteArrayInputStream(bos.toByteArray()));
                document = createDocument();
                document.readObject(reader, this);

                Context.abortOnLowMemory(getPackage());
            }
            return document;

        } catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

    @Override
    public void marshal(java.io.OutputStream os) {
        try {
            getJaxbElement().writeObject(XMLOutputFactory.newFactory().createXMLStreamWriter(os, "UTF-8"), this);
        }
        catch (Exception ue) {
            throw new RuntimeException();
        }
    }

    private JaxbValidationEventHandler jaxbValidationEventHandler;

    public JaxbValidationEventHandler getJaxbValidationEventHandler() {
        if(jaxbValidationEventHandler==null) {
            jaxbValidationEventHandler = new JaxbValidationEventHandler(getPackage());
        }
        return jaxbValidationEventHandler;
    }

    public Unmarshaller getUnmarshaller() throws JAXBException {
        final Unmarshaller unmarshaller = Context.createUnmarshaller(getJAXBContext(), getJaxbValidationEventHandler());
        unmarshaller.setListener(createUnmarshalListener());
        return unmarshaller;
    }

    public Marshaller getFragmentMarshaller(Map<String, String> prefixToUri, Map<String, String> uriToPrefix) throws JAXBException {
        final FragmentPrefixMapper prefixMapper = prefixToUri!=null&&uriToPrefix!=null ? new FragmentPrefixMapper() : null;
        final Marshaller marshaller = Context.createMarshaller(getJAXBContext(), prefixMapper, getJaxbValidationEventHandler());
        marshaller.setProperty(Marshaller.JAXB_FRAGMENT, Boolean.TRUE);
        return marshaller;
    }
}
