/*
/*
*
*    OPEN-XCHANGE legal information
*
*    All intellectual property rights in the Software are protected by
*    international copyright laws.
*
*
*    In some countries OX, OX Open-Xchange, open xchange and OXtender
*    as well as the corresponding Logos OX Open-Xchange and OX are registered
*    trademarks.
*    The use of the Logos is not covered by the GNU General Public License.
*    Instead, you are allowed to use these Logos according to the terms and
*    conditions of the Creative Commons License, Version 2.5, Attribution,
*    Non-commercial, ShareAlike, and the interpretation of the term
*    Non-commercial applicable to the aforementioned license is published
*    on the web site http://www.open-xchange.com/EN/legal/index.html.
*
*    Please make sure that third-party modules and libraries are used
*    according to their respective licenses.
*
*    Any modifications to this package must retain all copyright notices
*    of the original copyright holder(s) for the original code used.
*
*    After any such modifications, the original and derivative code shall remain
*    under the copyright of the copyright holder(s) and/or original author(s)per
*    the Attribution and Assignment Agreement that can be located at
*    http://www.open-xchange.com/EN/developer/. The contributing author shall be
*    given Attribution for the derivative code and a license granting use.
*
*     Copyright (C) 2016-2020 OX Software GmbH
*     Mail: info@open-xchange.com
*
*
*     This program is free software; you can redistribute it and/or modify it
*     under the terms of the GNU General Public License, Version 2 as published
*     by the Free Software Foundation.
*
*     This program is distributed in the hope that it will be useful, but
*     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
*     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
*     for more details.
*
*     You should have received a copy of the GNU General Public License along
*     with this program; if not, write to the Free Software Foundation, Inc., 59
*     Temple Place, Suite 330, Boston, MA 02111-1307 USA
*
*/

package com.openexchange.office.documents.impl;

import org.json.JSONException;

import com.openexchange.exception.OXException;
import com.openexchange.filemanagement.ManagedFile;
import com.openexchange.office.documents.DocRequest;
import com.openexchange.office.documents.DocRequests;
import com.openexchange.office.documents.Document;
import com.openexchange.office.documents.DocumentRequestProcessor;
import com.openexchange.office.documents.Documents;
import com.openexchange.office.documents.FlushableDocument;
import com.openexchange.office.rt2.protocol.value.RT2CliendUidType;
import com.openexchange.office.tools.doc.OXDocumentException;
import com.openexchange.office.tools.error.ErrorCode;
import com.openexchange.office.tools.logging.v2.ELogLevel;
import com.openexchange.office.tools.logging.v2.impl.Logger;
import com.openexchange.office.tools.logging.v2.impl.Slf4JLogger;
import com.openexchange.office.tools.osgi.ServiceLookupRegistry;

public class DocumentRequestProcessorImpl implements DocumentRequestProcessor {

    // ---------------------------------------------------------------
    private static final Logger LOG = Slf4JLogger.create(DocumentRequestProcessorImpl.class);

    //-------------------------------------------------------------------------
	@Override
	public String processDocRequest(final DocRequest aDocRequest) throws OXException {
        String result = null;

        if (null == aDocRequest)
            throw new OXDocumentException("", ErrorCode.GENERAL_ARGUMENTS_ERROR);

        try {
            final String sReqCmd = aDocRequest.getRequestCommand();

            if (DocRequests.DOCREQUEST_FLUSHTOMANAGEDFILE.equals(sReqCmd)) {
                result = handleFlushToManagedFile(aDocRequest);
            } else {
                throw new OXDocumentException("", ErrorCode.GENERAL_ARGUMENTS_ERROR);
            }
        } catch (OXException e) {
            LOG.forLevel(ELogLevel.E_ERROR).withError(e).log();
            throw e;
        } catch (Exception e) {
            LOG.forLevel(ELogLevel.E_ERROR).withError(e).log();
            throw new OXDocumentException("Exception caught while trying to process request", ErrorCode.GENERAL_UNKNOWN_ERROR);
        }

        return result;
	}

    //-------------------------------------------------------------------------
	private String handleFlushToManagedFile(final DocRequest aDocRequest) throws OXException, JSONException {
        String result = null;

        final Documents aDocumentsAccess = ServiceLookupRegistry.get().getService(Documents.class);
        if (null == aDocumentsAccess)
            throw new OXDocumentException("", ErrorCode.GENERAL_SERVER_COMPONENT_NOT_WORKING_ERROR);

        final RT2CliendUidType clientUID = new RT2CliendUidType(aDocRequest.getRequestClientUID());
        final String docUID = aDocRequest.getRequestDocUID();

        final Document aDoc = aDocumentsAccess.getDocument(docUID);
        final FlushableDocument aFlushableDoc = (aDoc instanceof FlushableDocument) ? (FlushableDocument)aDoc : null;

        if (null != aFlushableDoc) {
            try {
                final ManagedFile aManagedFile = aFlushableDoc.flushToManagedFile(clientUID);

                if (null != aManagedFile)
                    result = aManagedFile.getID();
            }
            catch (Exception e) {
                LOG.forLevel(ELogLevel.E_ERROR).withError(e).log();
                throw new OXDocumentException("Exception caught while trying to process flushToManagedFile", ErrorCode.GENERAL_UNKNOWN_ERROR);
            }
        } else {
            throw new OXDocumentException("", ErrorCode.GENERAL_DOCUMENT_ALREADY_DISPOSED_ERROR);
        }

        return result;
	}

}
