/*
/*
*
*    OPEN-XCHANGE legal information
*
*    All intellectual property rights in the Software are protected by
*    international copyright laws.
*
*
*    In some countries OX, OX Open-Xchange, open xchange and OXtender
*    as well as the corresponding Logos OX Open-Xchange and OX are registered
*    trademarks.
*    The use of the Logos is not covered by the GNU General Public License.
*    Instead, you are allowed to use these Logos according to the terms and
*    conditions of the Creative Commons License, Version 2.5, Attribution,
*    Non-commercial, ShareAlike, and the interpretation of the term
*    Non-commercial applicable to the aforementioned license is published
*    on the web site http://www.open-xchange.com/EN/legal/index.html.
*
*    Please make sure that third-party modules and libraries are used
*    according to their respective licenses.
*
*    Any modifications to this package must retain all copyright notices
*    of the original copyright holder(s) for the original code used.
*
*    After any such modifications, the original and derivative code shall remain
*    under the copyright of the copyright holder(s) and/or original author(s)per
*    the Attribution and Assignment Agreement that can be located at
*    http://www.open-xchange.com/EN/developer/. The contributing author shall be
*    given Attribution for the derivative code and a license granting use.
*
*     Copyright (C) 2016-2020 OX Software GmbH
*     Mail: info@open-xchange.com
*
*
*     This program is free software; you can redistribute it and/or modify it
*     under the terms of the GNU General Public License, Version 2 as published
*     by the Free Software Foundation.
*
*     This program is distributed in the hope that it will be useful, but
*     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
*     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
*     for more details.
*
*     You should have received a copy of the GNU General Public License along
*     with this program; if not, write to the Free Software Foundation, Inc., 59
*     Temple Place, Suite 330, Boston, MA 02111-1307 USA
*
*/

package com.openexchange.office.documents.impl;

import java.util.Optional;
import java.util.Set;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import org.json.JSONException;
import com.hazelcast.core.HazelcastInstance;
import com.hazelcast.core.IExecutorService;
import com.hazelcast.core.Member;
import com.openexchange.exception.OXException;
import com.openexchange.office.documents.DocRequest;
import com.openexchange.office.documents.DocRequester;
import com.openexchange.office.documents.Document;
import com.openexchange.office.documents.DocumentRequestProcessor;
import com.openexchange.office.documents.Documents;
import com.openexchange.office.documents.hazelcast.serialization.PortableDocRequest;
import com.openexchange.office.rt2.hazelcast.RT2DocOnNodeMap;
import com.openexchange.office.tools.doc.OXDocumentException;
import com.openexchange.office.tools.error.ErrorCode;
import com.openexchange.office.tools.logging.v2.ELogLevel;
import com.openexchange.office.tools.logging.v2.impl.Logger;
import com.openexchange.office.tools.logging.v2.impl.Slf4JLogger;
import com.openexchange.office.tools.osgi.ServiceLookupRegistry;

public class DocRequesterImpl implements DocRequester {

    // ---------------------------------------------------------------
    private static final Logger LOG = Slf4JLogger.create(DocRequesterImpl.class);

    //-------------------------------------------------------------------------
	private static final String HZ_EXECUTORSERVICE = "default";

    //-------------------------------------------------------------------------
	private static final long   MAX_TIMEOUT_FOR_REQUEST = 30000;

    //-------------------------------------------------------------------------
	public DocRequesterImpl() {
		// nothing to do
	}

    //-------------------------------------------------------------------------
	@Override
	public String sendSyncRequest(DocRequest aDocRequest) throws OXException {
        String result = null;
        String docUID = null;

        try {
            docUID = aDocRequest.getRequestDocUID();
        } catch (JSONException e) {
            throw new OXDocumentException("DocUID not set on doc request", ErrorCode.GENERAL_ARGUMENTS_ERROR);
        }

        LOG.forLevel(ELogLevel.E_DEBUG).withMessage("RT2: sendSyncRequest " + aDocRequest.toString()).log();

        final Document aDoc = getLocalDocumentFor(docUID);
        if (null == aDoc) {
            final HazelcastInstance hazelcastInstance = ServiceLookupRegistry.get().getService(HazelcastInstance.class);
            final Member aHzMember = getMemberOfDoc(hazelcastInstance, docUID);
            final IExecutorService executorService = hazelcastInstance.getExecutorService(HZ_EXECUTORSERVICE);

            final Future<String> future = executorService.submitToMember(new PortableDocRequest(aDocRequest), aHzMember);

            try {
                result = future.get(MAX_TIMEOUT_FOR_REQUEST, TimeUnit.MILLISECONDS);
            } catch (TimeoutException e) {
                throw new OXDocumentException("", ErrorCode.GENERAL_SYSTEM_BUSY_ERROR);
            } catch (Exception e) {
                LOG.forLevel(ELogLevel.E_ERROR).withError(e).log();
                throw new OXDocumentException("", ErrorCode.GENERAL_UNKNOWN_ERROR);
            }
        } else {
            final DocumentRequestProcessor aDocRequestProcessor = ServiceLookupRegistry.get().getService(DocumentRequestProcessor.class);

            if (null == aDocRequestProcessor)
                throw new OXDocumentException("", ErrorCode.GENERAL_SERVER_COMPONENT_NOT_WORKING_ERROR);

            result = aDocRequestProcessor.processDocRequest(aDocRequest);
        }

        return result;
    }

    //-------------------------------------------------------------------------
	private Document getLocalDocumentFor(String docUID) throws OXDocumentException {
        final Documents aDocuments = ServiceLookupRegistry.get().getService(Documents.class);

        if (null == aDocuments)
            throw new OXDocumentException("", ErrorCode.GENERAL_SERVER_COMPONENT_NOT_WORKING_ERROR);

        return aDocuments.getDocument(docUID);
	}

    //-------------------------------------------------------------------------
	private Member getMemberOfDoc(HazelcastInstance hzInstance, String docUID) throws OXException {
        final RT2DocOnNodeMap aDocOnNodeMap = ServiceLookupRegistry.get().getService(RT2DocOnNodeMap.class);
        final String sNodeUUID = aDocOnNodeMap.get(docUID);
        final Set<Member> aClusterMembers = hzInstance.getCluster().getMembers();
        final Optional<Member> aClusterMemberForDoc = aClusterMembers.stream()
                                                                     .filter(m -> memberHasUUID(m, sNodeUUID))
                                                                     .findFirst();
        return aClusterMemberForDoc.isPresent() ? aClusterMemberForDoc.get() : null;
	}

    //-------------------------------------------------------------------------
	private boolean memberHasUUID(Member aMember, String aUUID) {
		return (null != aMember) ? aMember.getUuid().equals(aUUID) : false;
	}
}
