package com.openexchange.office.dcs.registry.database;

import java.util.Map;
import com.openexchange.config.ConfigurationService;
import com.openexchange.office.tools.database.DatabaseConfiguration;
import com.openexchange.office.tools.database.DatabaseException;
import com.openexchange.office.tools.osgi.ServiceLookupRegistry;

public class DCSDatabaseFactory {

    /**
     * {@link DcsDatabaseConfiguration}
     *
     * @author <a href="mailto:kai.ahrens@open-xchange.com">Kai Ahrens</a>
     * @since v7.10.3
     */
    private static class DcsDatabaseConfiguration extends DatabaseConfiguration {
        final private static int DEFAULT_CONNECTION_POOLSIZE = 5;

        DcsDatabaseConfiguration(final ConfigurationService configService, final String configPrefix) {
            super(configService, configPrefix);
        }

        /**
         * Overrides the max. connection pool size method with a default
         * of DEFAULT_CONNECTION_POOLSIZE (5) since we don't need a
         * super class default pool size (30) that large for this database
         *
         * @return The number of open connections within the database connection pool.
         *  Default is 5.
         */
        @Override
        public int getConnectionPoolSize() {
            return this.getConfigurationService().getIntProperty(this.getConfigPrefix() + "connectionPool.maxPoolSize",
                DcsDatabaseConfiguration.DEFAULT_CONNECTION_POOLSIZE);
        }
    }

    private static DCSDatabase instance = null;

    public static DCSDatabase getDcsDatabase(final String configPrefix) throws DatabaseException {
        if (instance == null) {
            final ConfigurationService configService = configurationService();
            try {
                instance = new DCSDatabase(new DcsDatabaseConfiguration(configService, configPrefix));
            } catch (DatabaseException e) {
                final Map<String, String> dbProps = e.getDBProperties();

                final StringBuffer tmp = new StringBuffer(e.getMessage());
                tmp.append(", database connection properties: ");
                tmp.append(getDatabasePropertiesAsString(dbProps));

                throw new DatabaseException(tmp.toString(), e);
            }
        }
        return instance;
    }

    public static ConfigurationService configurationService() {
        final ConfigurationService aConfig = ServiceLookupRegistry.get ().getService(ConfigurationService.class);
        return aConfig;
    }

    private static String getDatabasePropertiesAsString(final Map<String, String> props) {
        final StringBuffer tmp = new StringBuffer(256);
        if ((null != props) && !props.isEmpty()) {
            boolean first = true;
            for (String key : props.keySet()) {
                if (first) {
                   first = false;
                } else {
                    tmp.append(", ");
                }
                tmp.append(key);
                tmp.append("=");
                tmp.append(props.get(key));
            }
        }

        return tmp.toString();
    }

    /**
     * For testing purpose
     * @param instance
     */
    static void setInstance(DCSDatabase dcsDbInstance) {
		instance = dcsDbInstance;
	}
}
