/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.tools.user;

import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.Timer;
import java.util.TimerTask;
import java.util.function.Predicate;
import org.apache.commons.logging.Log;

/**
 * Cache data needed for decrypting files from zips in get_file action
 *
 * @author Malte Timmermann
 * @since 7.8.3
 */
public class AuthorizationCache {

    @SuppressWarnings("deprecation")
    static protected final Log LOG = com.openexchange.log.LogFactory.getLog(AuthorizationCache.class);
    private static final long MAX_AGE_FOR_AUTOMATIC_GC = 24 * 60 * 60 * 1000; // 24h
    private static final long PERIOD_AUTOMATIC_GC      = 1 * 60 * 60 * 1000;  // 1h

    private final Map<String, AuthorizationEntry> hmap = new HashMap<>();
    private Timer timer = new Timer("com.openexchange.office.AuthorizationCache.Cleanup");
    private TimerTask timerTask = null;

    private static class Holder {
        private static final AuthorizationCache INSTANCE = new AuthorizationCache();
    }

    private static class AuthorizationEntry {
        private String  value;
        private long    timeStamp;
        private int     refCount;
        private int     refCountPersistent;

        public AuthorizationEntry(String value, boolean persistent) {
            this.value              = value;
            this.timeStamp          = System.currentTimeMillis();
            this.refCount           = 1;
            this.refCountPersistent = (persistent ? 1 : 0);
        }

        String getValue() {
            timeStamp = System.currentTimeMillis();
            return value;
        }

        long getLastAccessTime() { return timeStamp; }

        boolean isPersistent() { return (refCountPersistent > 0); }

        boolean canBeGC() {
            return ((refCount == 0) ||
                    (((Math.abs(System.currentTimeMillis() - getLastAccessTime())) > MAX_AGE_FOR_AUTOMATIC_GC) &&
                     !isPersistent()));
        }

        void inc(boolean persistent) {
            ++refCount;
            refCountPersistent += (persistent ? 1 : 0);
            timeStamp = System.currentTimeMillis();
        }

        boolean dec(boolean persistent) {
            boolean removeEntry = (--refCount <= 0);
            refCountPersistent -= (persistent ? 1 : 0);
            return removeEntry;
        }
    }

    private AuthorizationCache() {
        // create the task to do a gc of aged entries
        timerTask = new TimerTask() {

            @Override
            public void run() {
                try {
                    AuthorizationCache.getInstance().gc();
                }
                catch (Exception e)
                {
                    LOG.error("RT connection: exception caught trying to gc authorization cache entries", e);
                }
            }
        };

        // start the periodic timer with the specified timeout
        timer.schedule(timerTask, PERIOD_AUTOMATIC_GC, PERIOD_AUTOMATIC_GC);
    }

    public static AuthorizationCache getInstance() {
        return Holder.INSTANCE;
    }

    public static void addKey(String context, String user, String fileId, String value) {
        getInstance().addKey(createKey(context, user, fileId), false, value);
    }

    public static void addPersistentKey(String context, String user, String fileId, String value) {
        getInstance().addKey(createKey(context, user, fileId), true, value);
    }

    public static String getKey(String context, String user, String fileId) {
        return getInstance().getKey(createKey(context, user, fileId));
    }

    public static void removeKey(String context, String user, String fileId) {
        getInstance().removeKey(createKey(context, user, fileId), false);
    }

    public static void removePersistentKey(String context, String user, String fileId) {
        getInstance().removeKey(createKey(context, user, fileId), true);
    }

    private static String createKey(String context, String user, String fileId) {
        return "key:" + context + ":" + user + ":" + fileId;
    }

    private synchronized void addKey(String key, boolean persistent, String value) {
        final AuthorizationEntry entry = hmap.get(key);
        if (null == entry)
            hmap.put(key, new AuthorizationEntry(value, persistent));
        else {
            entry.inc(persistent);
        }
    }

    private synchronized String getKey(String key) {
        final AuthorizationEntry entry = hmap.get(key);
        return (null != entry) ? entry.getValue() : null;
    }

    private synchronized void removeKey(String key, boolean persistent) {
        final AuthorizationEntry entry = hmap.get(key);
        if (null != entry) {
            if (entry.dec(persistent))
                hmap.remove(key);
        }
    }

    protected synchronized void gc() {
        if (!hmap.isEmpty()) {
            final Set<String> keys = new HashSet<>(hmap.keySet());
            keys.stream()
                .filter(expiredEntry(hmap))
                .forEach(k -> hmap.remove(k));
        }
    }

    private static Predicate<String> expiredEntry(Map<String, AuthorizationEntry> aMap) {
        return e -> {
            final AuthorizationEntry entry = aMap.get(e);
            return (null != entry) ? entry.canBeGC() : false;
        };
    }
}
