/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * This file originates from http://dren.dk/hunspell.html (https://github.com/dren-dk/HunspellJNA)
 *
 * The Hunspell java bindings are licensed under LGPL, see the file COPYING.txt
 * in the root of the distribution for the exact terms.
 *
 *
 * @author Flemming Frandsen (flfr at stibo dot com)
 * @author L'or'and Somogyi < lorand dot somogyi at gmail dot com >
 *         http://lorands.com
 * @author Olivier Gattaz < olivier dot gattaz at isandlatech dot com >
 * @date 28/04/2011 (dd/mm/yy)
 */
package com.openexchange.spellchecker.hunspell.jna;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.UnsupportedEncodingException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.openexchange.exception.ExceptionUtils;
import com.openexchange.spellchecker.hunspell.SpellChecker;
import com.sun.jna.Native;
import com.sun.jna.Pointer;
import com.sun.jna.ptr.PointerByReference;

/**
 * The simple hunspell library frontend which takes care of creating and
 * singleton'ing the library instance (no need to load it more than once per
 * process) .
 *
 * The Hunspell java bindings are licensed under LGPL, see the file COPYING.txt
 * in the root of the distribution for the exact terms.
 *
 * @author Flemming Frandsen (flfr at stibo dot com)
 * @author L'or'and Somogyi < lorand dot somogyi at gmail dot com >
 *         http://lorands.com
 * @author Olivier Gattaz < olivier dot gattaz at isandlatech dot com >
 * @date 28/04/2011 (dd/mm/yy)
 */

public class Hunspell {

    final static Logger LOG = LoggerFactory.getLogger(SpellChecker.class);

	/**
	 * Class representing a single dictionary.
	 */
	public class Dictionary {

		/**
		 * The encoding used by this dictionary
		 */
		private final String encoding;

		/**
		 * The pointer to the hunspell object as returned by the hunspell
		 * constructor.
		 */
		private Pointer hunspellDict = null;

		/**
		 * The encoding of this dictionary
		 */
		private final Locale pLocale;

		/**
		 * Creates an instance of the dictionary.
		 *
		 * @param baseFileName
		 *            the base name of the dictionary. the base name of the
		 *            dictionary, passing /dict/da_DK means that the files
		 *            /dict/da_DK.dic and /dict/da_DK.aff get loaded
		 */
		Dictionary(String baseFileName) throws FileNotFoundException,
				UnsupportedEncodingException {
			File dic = new File(baseFileName + ".dic");
			File aff = new File(baseFileName + ".aff");

			if (!dic.canRead() || !aff.canRead()) {
				throw new FileNotFoundException("The dictionary files "
						+ baseFileName + "(.aff|.dic) could not be read");
			}

			hunspellDict = hsl.Hunspell_create(aff.toString(), dic.toString());
			encoding = hsl.Hunspell_get_dic_encoding(hunspellDict);

			pLocale = initLocaleFromDicFilename(dic.getName());

			// This will blow up if the encoding doesn't exist
			stringToBytes("test");
		}

		/**
		 * Deallocate the dictionary.
		 */
		public void destroy() {
			if (hsl != null && hunspellDict != null) {
				hsl.Hunspell_destroy(hunspellDict);
				hunspellDict = null;
			}
		}

		/**
		 * @return
		 */
		public String getEncoding() {
			return encoding;
		}

		/**
		 * @return the Locale of this dictionary
		 */
		public Locale getLocale() {
			return pLocale;
		}

		/**
		 * @param wDicFileName
		 * @return
		 */
		private Locale initLocaleFromDicFilename(String wDicFileName) {

			String wLocaleId = wDicFileName.substring(0,
					wDicFileName.lastIndexOf('.'));

			int wPosUnderScore = wLocaleId.indexOf('_');
			boolean wHasCountry = wPosUnderScore > -1
					&& wPosUnderScore < wLocaleId.length() - 1;

			String wLangage = (wHasCountry) ? wLocaleId.substring(0,
					wPosUnderScore) : wLocaleId;

			if (!wHasCountry)
				return new Locale(wLangage);

			String wCountry = wLocaleId.substring(wPosUnderScore + 1);

			return new Locale(wLangage, wCountry);
		}

		/**
		 * Check if a word is spelled correctly
		 *
		 * @param word
		 *            The word to check.
		 */
		public boolean misspelled(String word) {
			try {
				return hsl.Hunspell_spell(hunspellDict, stringToBytes(word)) == 0;
			} catch (UnsupportedEncodingException e) {
				return true; // this should probably never happen.
			}
		}

		/**
		 * Returns a list of stems
		 *
		 * @param word
		 *            The word to get stems for
		 * @return List of stems or null if the word doesn't exist in dictionary
		 */
		public List<String> stem(String word) {
			List<String> res = new ArrayList<String>();
			try {
				int stemsCount = 0;

				PointerByReference stems = new PointerByReference();
				stemsCount = hsl.Hunspell_stem(hunspellDict, stems,
						stringToBytes(word));

				if (stemsCount == 0)
					return null;

				Pointer[] pointerArray = stems.getValue().getPointerArray(0,
						stemsCount);

				for (int i = 0; i < stemsCount; i++) {
					/* Flemming's comment... */
					/*
					 * This only works for 8 bit chars, luckily hunspell uses
					 * either 8 bit encodings or utf8, if someone implements
					 * support in hunspell for utf16 we are in trouble.
					 */
					long len = pointerArray[i].indexOf(0, (byte) 0);
					if (len != -1) {
						if (len > Integer.MAX_VALUE) {
							throw new RuntimeException(
									"String improperly terminated: " + len);
						}
						byte[] data = pointerArray[i]
								.getByteArray(0, (int) len);
						res.add(new String(data, encoding));
					}
				}
			} catch (UnsupportedEncodingException e) {
                LOG.error(e.getMessage());
			}

			return res;
		}

		/**
		 * Convert a Java string to a zero terminated byte array, in the
		 * encoding of the dictionary, as expected by the hunspell functions.
		 */
		protected byte[] stringToBytes(String str)
				throws UnsupportedEncodingException {
			return (str + "\u0000").getBytes(encoding);
		}

		/**
		 * Returns a list of suggestions
		 *
		 * @param word
		 *            The word to check and offer suggestions for
		 */
		public List<String> suggest(String word) {
			List<String> res = new ArrayList<String>();
			// #68432# hunspell is having a bug in his memory management, we will not try to get replacements for words that exceeds normal word lengths
			if (word==null || word.length() > 32) {
			    return res;
			}
			int suggestionsCount = 0;
			try {
				PointerByReference suggestions = new PointerByReference();
				suggestionsCount = hsl.Hunspell_suggest(hunspellDict,
						suggestions, stringToBytes(word));

				// ogattaz - if no suggestion the "pointer array" is null, so
				// break
				if (suggestionsCount == 0)
					return res;

				// Get each of the suggestions out of the pointer array.
				Pointer[] pointerArray = suggestions.getValue()
						.getPointerArray(0, suggestionsCount);

				for (int i = 0; i < suggestionsCount; i++) {

					/*
					 * This only works for 8 bit chars, luckily hunspell uses
					 * either 8 bit encodings or utf8, if someone implements
					 * support in hunspell for utf16 we are in trouble.
					 */
					long len = pointerArray[i].indexOf(0, (byte) 0);
					if (len != -1) {
						if (len > Integer.MAX_VALUE) {
							throw new RuntimeException(
									"String improperly terminated: " + len);
						}
						byte[] data = pointerArray[i]
								.getByteArray(0, (int) len);
						res.add(new String(data, encoding));
					}
				}

			} catch (UnsupportedEncodingException ex) {
				// Shouldn't happen...
			}

			return res;
		}

		@Override
		public String toString() {
			return String.format("Dictionary: Locale=[%s] encoding=[%s]",
					getLocale().toString(), getEncoding());
		}
	}

	/**
	 * The Singleton instance of Hunspell
	 */
	private static Hunspell hunspell = null;

	/**
	 * The instance of the HunspellManager, looks for the native lib in the
	 * default directories
	 */
	public static Hunspell getInstance() throws UnsatisfiedLinkError,
			UnsupportedOperationException {
		return getInstance(null);
	}

	/**
	 * The instance of the HunspellManager, looks for the native lib in the
	 * directory specified.
	 *
	 * @param libDir
	 *            Optional absolute directory where the native lib can be found.
	 */
	public static Hunspell getInstance(String libPath)
			throws UnsatisfiedLinkError, UnsupportedOperationException {
		if (hunspell != null) {
			return hunspell;
		}

		hunspell = new Hunspell(libPath);
		return hunspell;
	}


	/**
	 * The native library instance, created by JNA.
	 */
	private HunspellLibrary hsl = null;

	/**
	 * This is the cache where we keep the already loaded dictionaries around
	 */
	private final HashMap<String, Dictionary> map = new HashMap<String, Dictionary>();

	/**
	 * Constructor for the library, loads the native lib.
	 *
	 * Loading is done in the first of the following three ways that works: 1)
	 * Unmodified load in the provided directory. 2) libFile stripped back to
	 * the base name (^lib(.*)\.so on unix) 3) The library is searched for in
	 * the classpath, extracted to disk and loaded.
	 *
	 * @param libDir
	 *            Optional absolute directory where the native lib can be found.
	 * @throws UnsupportedOperationException
	 *             if the OS or architecture is simply not supported.
	 */
	protected Hunspell(String libFile) throws UnsatisfiedLinkError,
			UnsupportedOperationException {

		try {
			hsl = (HunspellLibrary) Native.loadLibrary(libFile,
					HunspellLibrary.class);
		}
		catch (Throwable e) {
        	ExceptionUtils.handleThrowable(e);
		    throw(e);
		}
	}

	/**
	 * Removes all the dictionaries from the internal cache
	 *
	 */
	public void destroyAllDictionaries() {
		int wSize = map.size();
		map.clear();
	}

	/**
	 * Removes a dictionary from the internal cache
	 *
	 * @param baseFileName
	 *            the base name of the dictionary, as passed to getDictionary()
	 */
	public void destroyDictionary(String baseFileName) {
		if (map.containsKey(baseFileName)) {
			map.remove(baseFileName);
		}
	}

	/**
	 * Gets an instance of the dictionary.
	 *
	 * @param baseFileName
	 *            the base name of the dictionary, passing /dict/da_DK means
	 *            that the files /dict/da_DK.dic and /dict/da_DK.aff get loaded
	 */
	public Dictionary getDictionary(String baseFileName)
			throws FileNotFoundException, UnsupportedEncodingException {

		Dictionary d;
		// TODO: Detect if the dictionary files have changed and reload if they
		// have.
		if (map.containsKey(baseFileName)) {
			d = map.get(baseFileName);
		} else {
			d = new Dictionary(baseFileName);
			map.put(baseFileName, d);
		}

		return d;
	}
}
