/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.rt2.rest;

import javax.servlet.http.HttpServletRequest;

import org.json.JSONException;
import org.json.JSONObject;

import com.openexchange.ajax.requesthandler.AJAXActionService;
import com.openexchange.ajax.requesthandler.AJAXRequestData;
import com.openexchange.ajax.requesthandler.AJAXRequestDataTools;
import com.openexchange.ajax.requesthandler.AJAXRequestResult;
import com.openexchange.exception.OXException;
import com.openexchange.office.documents.ClientConnectionRegistry;
import com.openexchange.office.rt2.protocol.RT2Message;
import com.openexchange.office.rt2.protocol.RT2MessageFactory;
import com.openexchange.office.rt2.protocol.RT2MessageGetSet;
import com.openexchange.office.rt2.protocol.value.RT2CliendUidType;
import com.openexchange.office.rt2.protocol.value.RT2DocUidType;
import com.openexchange.office.rt2.protocol.value.RT2MessageType;
import com.openexchange.office.rt2.proxy.RT2DocProxy;
import com.openexchange.office.rt2.proxy.RT2DocProxyRegistry;
import com.openexchange.office.rt2.ws.RT2ChannelId;
import com.openexchange.office.rt2.ws.RT2EnhDefaultWebSocket;
import com.openexchange.office.rt2.ws.RT2WSApp;
import com.openexchange.office.rt2.ws.RT2WebSocketListener;
import com.openexchange.office.session.SessionService;
import com.openexchange.office.tools.error.ErrorCode;
import com.openexchange.office.tools.error.HttpStatusCode;
import com.openexchange.office.tools.logging.v2.ELogLevel;
import com.openexchange.office.tools.logging.v2.impl.Logger;
import com.openexchange.office.tools.logging.v2.impl.Slf4JLogger;
import com.openexchange.office.tools.osgi.ServiceLookupRegistry;
import com.openexchange.office.tools.osgi.ServiceLookupRegistryService;
import com.openexchange.tools.session.ServerSession;

public class EmergencyLeaveAction implements AJAXActionService {

    // ---------------------------------------------------------------
    private static final Logger   LOG              = Slf4JLogger.create(EmergencyLeaveAction.class);    
    
    // ---------------------------------------------------------------
    private static final String[] aMandatoryParams = { RESTParameters.PARAMETER_CLIENTUID,
                                                       RESTParameters.PARAMETER_DOCUID,
    		                                           RESTParameters.PARAMETER_CHANNELUID };
    
    private final SessionService sessionService;

    // ---------------------------------------------------------------
    public EmergencyLeaveAction() {
        this.sessionService = ServiceLookupRegistryService.getInstance().getService(SessionService.class);
    }

    // ---------------------------------------------------------------
	@Override
	public AJAXRequestResult perform(AJAXRequestData requestData, ServerSession session) throws OXException {

		if (!ParamValidator.areAllParamsNonEmpty(requestData, aMandatoryParams))
            return ParamValidator.getResultFor(HttpStatusCode.BAD_REQUEST.getStatusCode());

        final String channelUID = requestData.getParameter(RESTParameters.PARAMETER_CHANNELUID);
        final RT2WSApp wsApp = ServiceLookupRegistry.get().getService(RT2WSApp.class);
        final RT2WebSocketListener webSocketListener = ServiceLookupRegistry.get().getService(RT2WebSocketListener.class);
        
        if (null == wsApp) {
            return new AJAXRequestResult(ErrorCode.GENERAL_MISSING_SERVICE_ERROR.getAsJSONResultObject());
        }

        final String channelUrl = ClientConnectionRegistry.WS_URL_ROOT + channelUID;
        final String clientUID = requestData.getParameter(RESTParameters.PARAMETER_CLIENTUID);
        final String docUID = requestData.getParameter(RESTParameters.PARAMETER_DOCUID);
        final HttpServletRequest servletRequest = requestData.optHttpServletRequest();

        JSONObject leaveData = (JSONObject) requestData.getData();
        if (null == leaveData && (servletRequest != null)) {
        	// try to extract body data using a different approach (e.g.
        	// support POST with preferStream = false)
            requestData.setUploadStreamProvider(null);
            AJAXRequestDataTools.loadRequestBody(requestData);
            leaveData = (JSONObject) requestData.getData();
        }

    	try {
    		if (leaveData == null) {
    			leaveData = new JSONObject();
    		}

            LOG.forLevel(ELogLevel.E_DEBUG).withMessage("RT2: emergencyLeave request received for docUID=" + docUID + ", clientUID="+clientUID).log();

            final RT2EnhDefaultWebSocket clientConnection = wsApp.getWebSocketWithId(new RT2ChannelId(channelUrl));
            if (null != clientConnection) {
                webSocketListener.emergencyLeave(clientConnection, docUID, clientUID, session.getSessionID(), leaveData);
            } else {
            	sendEmergencyLeaveViaDocProxy(session, new RT2CliendUidType(clientUID), new RT2DocUidType(docUID), leaveData);
            }
    	} catch (final JSONException e) {
    		LOG.forLevel(ELogLevel.E_INFO).withMessage("RT2: JSONException caught trying to decode leave data from emergency request").withError(e).log();
    		return new AJAXRequestResult(ErrorCode.GENERAL_ARGUMENTS_ERROR.getAsJSONResultObject());
    	} catch (final DocProxyNotFoundException e) {
    		LOG.forLevel(ELogLevel.E_INFO).withMessage("RT2: DocProxyNotFoundException caught trying trying to send emergency request").withError(e).log();
    		return new AJAXRequestResult(ErrorCode.GENERAL_CLIENT_UID_UNKNOWN_ERROR.getAsJSONResultObject());
    	} catch (final ServiceNotAvailableException e) {
    		LOG.forLevel(ELogLevel.E_ERROR).withMessage("RT2: ServiceNotAvailableException caught trying trying to send emergency request").withError(e).log();
    		return new AJAXRequestResult(ErrorCode.GENERAL_MISSING_SERVICE_ERROR.getAsJSONResultObject());
    	} catch (final Exception e) {
    		LOG.forLevel(ELogLevel.E_WARNING).withMessage("RT2: Exception caught trying to send emergency request").withError(e).log();
    		return new AJAXRequestResult(ErrorCode.GENERAL_UNKNOWN_ERROR.getAsJSONResultObject());
    	}

        return ParamValidator.getResultFor(HttpStatusCode.OK.getStatusCode());
	}

    // ---------------------------------------------------------------
	private void sendEmergencyLeaveViaDocProxy(final ServerSession session, RT2CliendUidType clientUID, RT2DocUidType docUID, final JSONObject leaveData) throws Exception {
		final RT2DocProxyRegistry docProxyRegistry = ServiceLookupRegistry.get().getService(RT2DocProxyRegistry.class);

		if (docProxyRegistry != null) {
        	final String docProxyID = RT2DocProxy.calcID(clientUID, docUID);
        	final RT2DocProxy docProxy = docProxyRegistry.getDocProxy(docProxyID);
        	if (docProxy == null) {
        		throw new DocProxyNotFoundException("RT2DocProxy " + docProxyID + " not found in registry - must drop emergency leave message");
        	}

	  		final RT2Message aEmergencyLeaveRequest = RT2MessageFactory.newMessage(RT2MessageType.REQUEST_EMERGENCY_LEAVE, clientUID, docUID);
			RT2MessageGetSet.setSessionID(aEmergencyLeaveRequest, sessionService.getSessionId(session));
			aEmergencyLeaveRequest.setBody(leaveData);
		
        	docProxy.addMsgFromWebSocket(aEmergencyLeaveRequest);
            docProxy.leave(aEmergencyLeaveRequest);
		} else {
			throw new ServiceNotAvailableException("RT2DocProxyRegistry service not found emergency leave message cannot be processed - please check your installation!", RT2DocProxyRegistry.class.getCanonicalName());
		}
	}
}
