package com.openexchange.office.rt2.hazelcast;

import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.hazelcast.core.HazelcastInstance;
import com.hazelcast.core.IMap;
import com.hazelcast.core.Member;
import com.openexchange.management.ManagementAware;
import com.openexchange.management.ManagementObject;
import com.openexchange.office.rt2.hazelcast.management.HzDocOnNodeMapMBean;
import com.openexchange.office.rt2.hazelcast.management.HzDocOnNodeMapManagement;
import com.openexchange.office.rt2.hazelcast.serialization.PortableNodeDocsPredicate;
import com.openexchange.office.tools.osgi.ServiceLookupRegistry;

public class HzDocOnNodeMap implements RT2DocOnNodeMap, ManagementAware<HzDocOnNodeMapMBean>
{
	private static final Logger log = LoggerFactory.getLogger(HzDocOnNodeMap.class);

    //-------------------------------------------------------------------------
	private final String mapIdentifier;

    //-------------------------------------------------------------------------
    private final HzDocOnNodeMapManagement managementObject;

    //-------------------------------------------------------------------------
    public HzDocOnNodeMap(String mapIdentifier) throws Exception {
        this.mapIdentifier = mapIdentifier;
        this.managementObject = new HzDocOnNodeMapManagement(this);
    }

    //-------------------------------------------------------------------------
	@Override
	public String get(String sDocUID) {
        final IMap<String, String> allDocToNodeMappings = getDocNodeMapping();
        return allDocToNodeMappings.get(sDocUID);
	}

    //-------------------------------------------------------------------------
	@Override
	public Map<String, String> get(Collection<String> aDocUIDs)  {
              Map<String, String> aFoundNodesMap = new HashMap<>();
        final Set<String>         aDocUIDSet     = new HashSet<>();

        for (final String sDocUID : aDocUIDs) {
        	aDocUIDSet.add(sDocUID);
        }

        if (!aDocUIDSet.isEmpty()) {
            final IMap<String, String> allDocToNodeMappings = getDocNodeMapping();
            aFoundNodesMap = allDocToNodeMappings.getAll(aDocUIDSet);
        }

        return aFoundNodesMap;
	}

    //-------------------------------------------------------------------------
	@Override
	public String set(String sDocUID)  {
        final IMap<String, String> allDocToNodeMappings = getDocNodeMapping();
        String sPrevNodeUID = allDocToNodeMappings.get(sDocUID);

        if (sPrevNodeUID != null) {
            // but the previous resource provides a presence
        	allDocToNodeMappings.put(sDocUID, getLocalNodeUUID());
        } else {
        	sPrevNodeUID = allDocToNodeMappings.putIfAbsent(sDocUID, getLocalNodeUUID());
        }
        return sPrevNodeUID;
	}

    //-------------------------------------------------------------------------
	@Override
	public void setIfAbsent(String sDocUID) {
       final IMap<String, String> allDocToNodeMappings = getDocNodeMapping();
       String aPrevNodeUUID = allDocToNodeMappings.get(sDocUID);

       if (aPrevNodeUUID == null) {
          	allDocToNodeMappings.put(sDocUID, getLocalNodeUUID());
        }
	}

    //-------------------------------------------------------------------------
	@Override
	public String remove(String sDocUID) {
        final IMap<String, String> allDocToNodeMappings = getDocNodeMapping();
        String sPrevNodeUUID = allDocToNodeMappings.get(sDocUID);

        if (sPrevNodeUUID != null) {
           	allDocToNodeMappings.remove(sDocUID);
        }
        return sPrevNodeUUID;
	}

    //-------------------------------------------------------------------------
	@Override
	public Map<String, String> remove(Collection<String> aDocUIDs)  {
        final Map<String, String> aRemovedMap = new HashMap<>();
        final Set<String>         aDocUIDSet  = new HashSet<>();

        for (final String sDocUID : aDocUIDs) {
        	aDocUIDSet.add(sDocUID);
        }

        if (!aDocUIDSet.isEmpty()) {
            final IMap<String, String> allDocToNodeMappings = getDocNodeMapping();
            final Map<String, String>  aMatchStates         = allDocToNodeMappings.getAll(aDocUIDSet);

            if (aMatchStates != null) {
                for (Entry<String, String> entry : aMatchStates.entrySet()) {
                	final String sFoundUID = entry.getKey();
                	allDocToNodeMappings.remove(sFoundUID);
                	aRemovedMap.put(sFoundUID, entry.getValue());
                }
            }
        }

        return aRemovedMap;
	}

    //-------------------------------------------------------------------------
    @Override
	public Set<String> getDocsOfMember()
    {
    	return getDocsOfMember(getLocalNodeUUID());
    }

    //-------------------------------------------------------------------------
    @Override
	public Set<String> getDocsOfMember(final String sNodeUUID)
    {
        final IMap<String, String> allResources = getDocNodeMapping();
        final PortableNodeDocsPredicate nodeDocsPredicate = new PortableNodeDocsPredicate(sNodeUUID);
        final Set<Entry<String, String>> matchingResources = allResources.entrySet(nodeDocsPredicate);

        final Set<String> foundIds = new HashSet<String>();
        final Iterator<Entry<String, String>> iterator = matchingResources.iterator();
        while(iterator.hasNext()) {
            try {
                Entry<String, String> next = iterator.next();

                foundIds.add(next.getKey());
            } catch (Exception e) {
            	log.error("Couldn't add resource that was found for member " + sNodeUUID, e);
            }
        }

        return foundIds;
    }

    //-------------------------------------------------------------------------
    @Override
    public String getUniqueMapName() {
        return mapIdentifier;
	}

    //-------------------------------------------------------------------------
	@Override
	public ManagementObject<HzDocOnNodeMapMBean> getManagementObject() {
		return managementObject;
	}

    //-------------------------------------------------------------------------
    public IMap<String, String> getDocNodeMapping() {
        final HazelcastInstance aHZCore = ServiceLookupRegistry.get ().getService(HazelcastInstance.class);
        return aHZCore.getMap(mapIdentifier);
    }

    //-------------------------------------------------------------------------
    public String getLocalNodeUUID() {
        final HazelcastInstance aHZCore = ServiceLookupRegistry.get ().getService(HazelcastInstance.class);
        final Member aMySelfNode = aHZCore.getCluster().getLocalMember();
        return aMySelfNode.getUuid();
    }

    //-------------------------------------------------------------------------
	@Override
	public Set<String> getMember() {
		return new HashSet<>(getDocNodeMapping().values());
	}
}
