/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */
package com.openexchange.office.rt2.proxy;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

import org.apache.commons.text.TextStringBuilder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.codahale.metrics.Gauge;
import com.codahale.metrics.MetricRegistry;
import com.openexchange.exception.ExceptionUtils;
import com.openexchange.office.rt2.core.RT2ThreadFactoryBuilder;
import com.openexchange.office.rt2.exception.RT2Exception;
import com.openexchange.office.rt2.exception.RT2TypedException;
import com.openexchange.office.rt2.jms.JmsMessageSender;
import com.openexchange.office.rt2.jms.RT2AdminJmsConsumer;
import com.openexchange.office.rt2.logging.IMessagesLoggable;
import com.openexchange.office.rt2.logging.IMessagesObjectManager;
import com.openexchange.office.rt2.logging.MessagesLogger;
import com.openexchange.office.rt2.protocol.RT2Message;
import com.openexchange.office.rt2.protocol.RT2MessageFactory;
import com.openexchange.office.rt2.protocol.value.RT2CliendUidType;
import com.openexchange.office.rt2.protocol.value.RT2DocUidType;
import com.openexchange.office.rt2.protocol.value.RT2MessageType;
import com.openexchange.office.rt2.protocol.value.RT2SessionIdType;
import com.openexchange.office.rt2.ws.RT2ChannelId;

//=============================================================================
/** knows all open documents within the RT2 cluster ...
 *  It works internal with a cluster wide cache to hold it's state.
 */
public class RT2DocProxyRegistry implements IMessagesObjectManager
{
    private static final Logger log = LoggerFactory.getLogger(RT2DocProxyRegistry.class);

    //-------------------------------------------------------------------------
	public static final String  PERSISTENCE_SUBSET_DOC_STATES  = "doc.states";

	//-------------------------------------------------------------------------
	private static final long   MAX_TIME_FOR_SHUTDOWN_PROXIES  = 30000; // time in ms

    //-------------------------------------------------------------------------
    private final ConcurrentHashMap< String, RT2DocProxy > docProxyRegistry = new ConcurrentHashMap<>();

    //-------------------------------------------------------------------------
    private final JmsMessageSender jmsMessageSender;
    
    private final ScheduledExecutorService logExecutor = Executors.newSingleThreadScheduledExecutor(new RT2ThreadFactoryBuilder("RT2DocProxyRegistryLogger-%d").build());
    
    //-------------------------------------------------------------------------
    public RT2DocProxyRegistry(MetricRegistry metricRegistry, JmsMessageSender jmsMessageSender) {
    	this.jmsMessageSender = jmsMessageSender;
    	if (metricRegistry != null) {
	    	metricRegistry.register(MetricRegistry.name("DocProxyRegistry", "size"), 
	    		new Gauge<Integer>() {
	
					@Override
					public Integer getValue() {
						return docProxyRegistry.size();
					}
	    		});
    	}    	
    	logExecutor.scheduleAtFixedRate(new MessagesLogger(this, log), 3, 3, TimeUnit.MINUTES); 
    }
    
    //-------------------------------------------------------------------------
    public void stop () {
        log.debug("RT2DocProxyRegistry shutdown - will close all DocProxy instances registered on this node.");

        shutdownDocProxies   ();

        clear();
        
        logExecutor.shutdown();
    }

    //-------------------------------------------------------------------------
    public void clear() {
        docProxyRegistry.clear ();
    }

    //-------------------------------------------------------------------------
    /** look for an existing doc proxy instance or create new one otherwise.
     *
     *  @param  clientUID [IN]
     *          the unique ID of the client.
     *
     *  @param  docUID [IN]
     *          the unique ID of the document.
     *
     *  @return a doc proxy instance always.
     * @throws RT2Exception 
     * @throws RT2TypedException 
     */
    public RT2DocProxy getOrCreateDocProxy (final RT2Message rt2Msg, boolean goOnlineIfNew, RT2ChannelId channelId, RT2SessionIdType sessionId) throws RT2TypedException, RT2Exception {
    	synchronized (docProxyRegistry) {
	        final String proxyID  = RT2DocProxy.calcID(rt2Msg.getClientUID(), rt2Msg.getDocUID());
	        RT2DocProxy docProxy = docProxyRegistry.get(proxyID);

	        if (docProxy == null) {
	        	log.info("Creating DocProxy ");
	            docProxy = RT2DocProxy.create(rt2Msg.getClientUID(), rt2Msg.getDocUID(), channelId, sessionId);
	            docProxyRegistry.put(proxyID, docProxy);
	            if (goOnlineIfNew) {
	            	docProxy.goOnline();
	            }
	            docProxy.incCountUsedBy();
	            return docProxy;
	        }
	 
	        log.debug("use existing doc proxy {}", docProxy);
	        docProxy.incCountUsedBy();
	        return docProxy;
       	}
    }

    //-------------------------------------------------------------------------        
    public void registerDocProxy(final RT2DocProxy docProxy) {
    	docProxyRegistry.put(docProxy.getProxyID(), docProxy);
    }
    
    //-------------------------------------------------------------------------
    /** deregister the given doc proxy from this registry ...
     *
     *  @param aDocProxy [IN]
     *         the proxy to be deregistered here.
     */
    public void deregisterDocProxy (final RT2DocProxy aDocProxy, boolean force) {
    	if (force || (aDocProxy.decCountUsedBy() <= 0)) {
	    	List<String> msgsLogs = aDocProxy.formatMsgsLogInfo();
	    	if (!msgsLogs.isEmpty()) {
		    	TextStringBuilder builder = new TextStringBuilder(msgsLogs.get(0));
		    	msgsLogs.remove(0);
		    	msgsLogs.forEach(s -> builder.appendln("    " + s));
		    	log.info("deregister doc proxy {}, {}...", aDocProxy.getProxyID(), builder);	        
	    	} else {
	    		log.info("deregister doc proxy {}...", aDocProxy.getProxyID());
	    	}
	    	
	
	
	        // remove proxy from registry only
	        // do not call any methods on proxy : no clear, no dispose, ... !!!
	        docProxyRegistry.remove(aDocProxy.getProxyID());
	        
	        RT2Message msg = RT2MessageFactory.newAdminMessage(RT2MessageType.ADMIN_TASK_COMPLETED_CLOSE_DOC_ROUTE);
	        msg.setDocUID(aDocProxy.getDocUID());
	        msg.setClientUID(aDocProxy.getClientUID());
	        jmsMessageSender.sendToAdminTopic(msg, RT2AdminJmsConsumer.generateAdminID());
    	}
    }

    //-------------------------------------------------------------------------
    /** @return the doc proxy instance bound to the given ID (if it exists - null otherwise)
     *
     *  @param sProxyID []
     *          the ID of the requested doc proxy object.
     */
    public RT2DocProxy getDocProxy (final RT2CliendUidType sClientUID, final RT2DocUidType sDocUID) {
        final String sProxyID  = RT2DocProxy.calcID(sClientUID, sDocUID);
        final RT2DocProxy aDocProxy = docProxyRegistry.get(sProxyID);
        return aDocProxy;
    }

    //-------------------------------------------------------------------------
    public RT2DocProxy getDocProxy (String proxyId) {
        final RT2DocProxy aDocProxy = docProxyRegistry.get(proxyId);
        return aDocProxy;
    }


    //-------------------------------------------------------------------------
    /** @return the a list of doc proxy instances bound to the given document
     *          Can be an empty list - but wont be null.
     *
     *  @param  sDocUID [IN]
     *          the UID of the document where all proxy instances are searched for.
     */
    public List< RT2DocProxy > getDocProxies4DocUID (final RT2DocUidType sDocUID, final RT2SessionIdType sessionId) {
        final String sIDMatch  = RT2DocProxy.calcIDMatch4DocUID(sDocUID.getValue());
        synchronized (docProxyRegistry) {
	        Set<RT2DocProxy> allDocProxies = new HashSet<>(docProxyRegistry.values());
	        List< RT2DocProxy > rt2DocProxies = allDocProxies.stream().filter(p -> p.getProxyID().matches(sIDMatch)).collect(Collectors.toList());
	        if (sessionId != null) {
	        	rt2DocProxies = rt2DocProxies.stream().filter(p -> p.getSessionId().equals(sessionId)).collect(Collectors.toList());
	        }
	        return rt2DocProxies;
        }
    }

    //-------------------------------------------------------------------------
    public List< RT2DocProxy > listAllDocProxies () {
        final List< RT2DocProxy > res  = new ArrayList<> ();
        res.addAll(docProxyRegistry.values());
        return res;
    }

	//-------------------------------------------------------------------------
    private void shutdownDocProxies () {
    	synchronized (docProxyRegistry) {
	        final List< RT2DocProxy > lDocProxies = listAllDocProxies();
	
	        for (final RT2DocProxy aDocProxy : lDocProxies) {
	            try {
	                aDocProxy.closeHard (true     );
	            } catch (final Throwable exIgnore) {
	                ExceptionUtils.handleThrowable(exIgnore);
	                log.error("... closeHard for doc proxy [{}] failed", aDocProxy);
	            }
	        }
	
	        // Synchronize with asynchronous closeHard which needs two complete
	        // round-trips for close/leave. Don't wait forever and use a timeout
	        // to not block the shutdown process for a longer period of time.
	        long nStartTime   = System.currentTimeMillis();
	        long nElapsedTime = 0;
	        int nCount        = lDocProxies.size();
	
	        while ((nCount != 0) && (nElapsedTime < MAX_TIME_FOR_SHUTDOWN_PROXIES)) {
	            final Iterator<RT2DocProxy> aIter = lDocProxies.iterator();
	            while (aIter.hasNext()) {
	                final RT2DocProxy aDocProxy = aIter.next();
	                if (aDocProxy.isLeft()) {
	                    aIter.remove();
	                    nCount--;
	                }
	            }
	
	            try {
	                Thread.sleep(100);
	            } catch (InterruptedException e) {
	                Thread.currentThread().interrupt();
	            }
	            nElapsedTime = System.currentTimeMillis() - nStartTime;
	        }
	
	        for (final RT2DocProxy aDocProxy : lDocProxies) {
	            try {
	                deregisterDocProxy  (aDocProxy, true);
	            } catch (final Throwable exIgnore) {
	                ExceptionUtils.handleThrowable(exIgnore);
	                log.error("... deregister for doc proxy [{}] failed", aDocProxy);
	            }
	        }
    	}
    }

	@Override
	public Collection<IMessagesLoggable> getManagedCollection() {
		return new HashSet<>(docProxyRegistry.values());
	}
}
