/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */
package com.openexchange.office.rt2.proxy;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicReference;
import java.util.stream.Collectors;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.collect.EvictingQueue;
import com.openexchange.exception.ExceptionUtils;
import com.openexchange.log.LogPropertyName.LogLevel;
import com.openexchange.office.rt2.cache.ClusterLockService;
import com.openexchange.office.rt2.cache.ClusterLockService.ClusterLock;
import com.openexchange.office.rt2.cache.RT2DocInfo;
import com.openexchange.office.rt2.core.RT2Constants;
import com.openexchange.office.rt2.core.RT2GarbageCollector;
import com.openexchange.office.rt2.core.RT2LogInfo;
import com.openexchange.office.rt2.core.RT2NodeInfoService;
import com.openexchange.office.rt2.core.doc.IDocProcessorContainer;
import com.openexchange.office.rt2.core.doc.IRT2DocProcessorManager;
import com.openexchange.office.rt2.exception.RT2Exception;
import com.openexchange.office.rt2.exception.RT2TypedException;
import com.openexchange.office.rt2.hazelcast.RT2DocOnNodeMap;
import com.openexchange.office.rt2.jms.JmsMessageSender;
import com.openexchange.office.rt2.jms.RT2AdminJmsConsumer;
import com.openexchange.office.rt2.logging.IMessagesLoggable;
import com.openexchange.office.rt2.protocol.RT2Message;
import com.openexchange.office.rt2.protocol.RT2MessageFactory;
import com.openexchange.office.rt2.protocol.RT2MessageGetSet;
import com.openexchange.office.rt2.protocol.RT2Protocol;
import com.openexchange.office.rt2.protocol.value.RT2CliendUidType;
import com.openexchange.office.rt2.protocol.value.RT2DocUidType;
import com.openexchange.office.rt2.protocol.value.RT2MessageIdType;
import com.openexchange.office.rt2.protocol.value.RT2MessageType;
import com.openexchange.office.rt2.protocol.value.RT2NodeUuidType;
import com.openexchange.office.rt2.protocol.value.RT2SessionIdType;
import com.openexchange.office.rt2.protocol.value.RT2UnavailableTimeType;
import com.openexchange.office.rt2.proxy.RT2DocProxyLogInfo.Direction;
import com.openexchange.office.rt2.ws.RT2ChannelId;
import com.openexchange.office.rt2.ws.RT2WSChannelDisposer;
import com.openexchange.office.tools.error.ErrorCode;
import com.openexchange.office.tools.logging.annotation.LogMethodCallHelper;
import com.openexchange.office.tools.logging.annotation.Loggable;
import com.openexchange.office.tools.osgi.ServiceLookupRegistry;
import com.openexchange.office.tools.osgi.ServiceLookupRegistryService;

//=============================================================================
public class RT2DocProxy implements Loggable, IMessagesLoggable {

    private static final long TIMEOUT_HARD_CLOSE = 30000;

    //-------------------------------------------------------------------------

    private static final Logger log = LoggerFactory.getLogger(RT2DocProxy.class);

    private final RT2CliendUidType clientUID;
    private final RT2DocUidType docUID;
    private final RT2ChannelId channelId;
    private final String proxyID;
    private final long creationTime;

    private final RT2DocProxyRegistry docProxyRegistry;
    private final RT2DocInfoRegistry docInfoRegistry;
    private final RT2NodeInfoService rt2NodeInfo;

    private final Map< RT2MessageIdType, Deferred< RT2Message > > deferreds = new ConcurrentHashMap<>();

    private final JmsMessageSender jmsMessageSender;

    private final RT2WSChannelDisposer channelDisposer;

    private final MessageCounters messageCounters;

    private final ClusterLockService clusterLockService;

    //-------------------------------------------------------------------------
    protected RT2DocProxyStateHolder docProxyStateHolder;

    private RT2DocProxyConnectionStateHolder docProxyConnectionStateHolder = new RT2DocProxyConnectionStateHolder();

    //-------------------------------------------------------------------------
    private AtomicInteger m_nOutStandingRequests = new AtomicInteger(0);

    //-------------------------------------------------------------------------
    private CountDownLatch m_bJoinLeaveSyncPoint = new CountDownLatch(1);

    //-------------------------------------------------------------------------
    private CountDownLatch m_LeaveSyncPoint = new CountDownLatch(1);

    //-------------------------------------------------------------------------
    private AtomicBoolean sysShutdown = new AtomicBoolean(false);

    private AtomicInteger countUsedBy = new AtomicInteger(0);

    //-------------------------------------------------------------------------
    private EvictingQueue<RT2DocProxyLogInfo> msgs = EvictingQueue.create(20);

    private final AtomicReference<RT2SessionIdType> sessionId;

    private RT2DocOnNodeMap rt2DocOnNodeMap;

    //-------------------------------------------------------------------------
    RT2DocProxy (RT2CliendUidType clientUID, RT2DocUidType docUID, RT2ChannelId channelId, RT2SessionIdType sessionId) {
    	msgs.add(new RT2DocProxyLogInfo(Direction.UNKNOWN, "create RT2DocProxy", clientUID, docUID));
        this.clientUID = clientUID;
        this.docUID = docUID;
        this.proxyID = calcID(clientUID, docUID);
        this.channelId = channelId;
        this.creationTime = System.currentTimeMillis();
        this.sessionId = new AtomicReference<RT2SessionIdType>(sessionId);
        this.docProxyStateHolder = new RT2DocProxyStateHolder(clientUID, docUID);
        ServiceLookupRegistryService serviceLookupRegistryService = ServiceLookupRegistryService.getInstance();
        this.docProxyRegistry = serviceLookupRegistryService.getService(RT2DocProxyRegistry.class);
        this.docInfoRegistry = serviceLookupRegistryService.getService(RT2DocInfoRegistry.class);
        this.rt2NodeInfo = serviceLookupRegistryService.getService(RT2NodeInfoService.class);
        this.jmsMessageSender = serviceLookupRegistryService.getService(JmsMessageSender.class);
        this.channelDisposer = serviceLookupRegistryService.getService(RT2WSChannelDisposer.class);
        this.messageCounters = serviceLookupRegistryService.getService(MessageCounters.class);
        this.clusterLockService = serviceLookupRegistryService.getService(ClusterLockService.class);
        this.rt2DocOnNodeMap = serviceLookupRegistryService.getService(RT2DocOnNodeMap.class);
    }

    //-------------------------------------------------------------------------
    public static String calcID (final RT2CliendUidType sClientUID, final RT2DocUidType sDocUID   )
    {
    	return sClientUID.getValue() + "_" + sDocUID.getValue();
    }

    //-------------------------------------------------------------------------
    public static String calcIDMatch4DocUID (final String sDocUID)
    {
        return "(.*)("+sDocUID+")($)";
    }

    //-------------------------------------------------------------------------
    public static RT2DocProxy create (final RT2CliendUidType clientUID, final RT2DocUidType docUID, final RT2ChannelId channelId, RT2SessionIdType sessionId) {
    	LogMethodCallHelper.logMethodCall(log, RT2DocProxy.class, "create", LogLevel.INFO, clientUID, docUID, channelId, sessionId);
    	final RT2DocProxy aContext = new RT2DocProxy(clientUID, docUID, channelId, sessionId);//, docType);
    	return aContext;
    }

    //-------------------------------------------------------------------------
    public void addMsgFromWebSocket(RT2Message msg) {
    	synchronized (msgs) {
    		msgs.add(new RT2DocProxyLogInfo(Direction.FROM_WS, msg));
    	}
   		goOnline();
    }

    //-------------------------------------------------------------------------
    public void closeHard (boolean sysShutdown)
        throws Exception
    {
    	synchronized (msgs) {
    		msgs.add(new RT2DocProxyLogInfo(Direction.UNKNOWN, "closeHard", clientUID, docUID));
    	}
    	if (sysShutdown) {
    		LogMethodCallHelper.logMethodCall(this, "closeHard", LogLevel.INFO, sysShutdown);
    	} else {
    		LogMethodCallHelper.logMethodCall(this, "closeHard", LogLevel.DEBUG, sysShutdown);
    	}

        final CountDownLatch aSync = new CountDownLatch (1);

        if (sysShutdown) {
            this.sysShutdown.set(true);
        }

        final Deferred.Func< RT2Message > aLeaveDeferredFunc = new Deferred.Func< RT2Message > ()
        {
           @Override
            public void always(final RT2Message aResponse) {
                log.debug("RT2: hard-close : got leave response for proxy with id ''{}''...", getProxyID());
                aSync.countDown();
            }
        };

        switch (docProxyStateHolder.getProxyState()) {
            // Don't continue in case a client already is leaving/has left in the meantime
            case E_IN_LEAVE:
            case E_LEFT: return;
            // just trigger leave in case we are already in a close(ing) state or haven't reached
            // the open state
            case E_CLOSED:
            case E_IN_CLOSE:
            case E_JOINED: {
                final RT2Message aLeaveRequest = RT2MessageFactory.newLeaveRequest(clientUID, docUID);
                RT2MessageGetSet.setInternalHeader(aLeaveRequest, RT2Protocol.HEADER_INTERNAL_FORCE, true);
                leave (aLeaveRequest).setAutoClose(true).always(aLeaveDeferredFunc);
                break;
            }
            default: {
                final RT2Message aCloseRequest = RT2MessageFactory.newCloseDocRequest(clientUID, docUID);
                RT2MessageGetSet.setInternalHeader(aCloseRequest, RT2Protocol.HEADER_INTERNAL_FORCE, true);

                final RT2Message aLeaveRequest = RT2MessageFactory.createRequestFromMessage(aCloseRequest, RT2MessageType.REQUEST_LEAVE);
                RT2MessageGetSet.setInternalHeader(aLeaveRequest, RT2Protocol.HEADER_INTERNAL_FORCE, true);

                log.debug("RT2: hard-close : trigger close for proxy with id ''{}''...", getProxyID());

                final Deferred.Func< RT2Message > aCloseDeferredFunc = new Deferred.Func< RT2Message > () {
                    @Override
                    public void always(final RT2Message aResponse) throws Exception {
                        log.debug("RT2: hard-close : trigger leave for proxy with id ''{}''...", getProxyID());
                        leave (aLeaveRequest).setAutoClose(true).always(aLeaveDeferredFunc);
                    }
                };

                close (aCloseRequest).setAutoClose(true).always(aCloseDeferredFunc);
            }
        }

        log.debug("RT2: hard-close : wait for the results for proxy with id ''{}''...", getProxyID());

        final boolean bOK = aSync.await(TIMEOUT_HARD_CLOSE, TimeUnit.MILLISECONDS);
        if ( ! bOK) {
            log.warn("RT2: hard-close : timeout : result of this operation for docProxy with id ''{}'' is undefined !", getProxyID());
        }

        log.debug("RT2: hard-close for docProxy with id ''{}'': DONE", getProxyID());
    }

    //-------------------------------------------------------------------------
    public boolean isLeft() {
        return m_LeaveSyncPoint.getCount() <= 0;
    }

    //-------------------------------------------------------------------------
    public void join (final RT2Message aJoinRequest) throws Exception {
        LogMethodCallHelper.logMethodCall(this, "join", aJoinRequest.getHeader());
        this.messageCounters.incJoinRequestCounter();
        // UUID from Hazelcast
        aJoinRequest.setNodeUUID(new RT2NodeUuidType(UUID.fromString(rt2NodeInfo.getNodeUUID())));

        // Tricky part :
        // If this instance is already in "shutdown mode" ...
        // we need to sync with that operation.
        // Otherwise some resources (as e.g. camel routes) will be closed ...
        // during our try to join/open a same document within this instance.

        // IMPORTANT to call this BEFORE switch into new proxy state !
        syncJoinWithLeaveIfNeeded ();

        long clientRefCount = 0;

        final RT2DocInfo aDocInfo = docInfoRegistry.getDocInfo(docUID);
        final ClusterLock clusterLock = clusterLockService.getLock(docUID);
        boolean locked = false;
        boolean triggerGc = false;
        try {
            log.debug("Lock call by client [{}] for doc with id [{}]", clientUID, docUID);
            locked = clusterLock.lock();
            if (locked) {
                docProxyStateHolder.switchProxyStateOrFail(EDocProxyState.E_IN_JOIN);

                clientRefCount  = aDocInfo.incRefCount4Clients();
                if (clientRefCount < 0) {
                	clusterLock.unlock();
               		try {
    					Thread.sleep(3000);
    				} catch (InterruptedException e) {}
                	try {
    					join(aJoinRequest);
    				} catch (Exception e) {
    					throw new RT2TypedException(ErrorCode.GENERAL_SYSTEM_BUSY_ERROR, getMsgsAsList());
    				}
                	return;
                }
                final boolean bIsFirstJoin = (clientRefCount == 1L);

                if (bIsFirstJoin) {
                    log.debug("RT2: join context - accepted : is first");
                    rt2NodeInfo.registerDocOnNode(docUID);
                }
                else if ((clientRefCount < RT2Constants.REF_COUNT_TRESHHOLD) && (clientRefCount > -1)) {
                	String nodeId = rt2DocOnNodeMap.get(docUID.getValue());
                	if (nodeId == null) {
                		log.warn("RT2: join context - clientRefCount is greater 0 but rt2DocOnNodeMap has no entry for docUID {}", docUID);
                		triggerGc = true;
                	} else {
                		log.debug("RT2: join context - accepted : is {}", clientRefCount);
                	}
                }
                else {
                    // The ref-count value indicates that the document is part of a clean-up
                    // process, therefore we need to bail-out now with an error. The ref-count
                    // will be reset by the clean-up process, therefore we don't need to decrement
                    // it here (see RT2NodeHealthMonitor/MasterCleanupTask/CleanupTask).
                    //
                    // We have to make sure that the ref-count will be removed in a defined time-frame even
                    // if the clean-up process was not completed. Therefore the ref-count will be also
                    // registered at the garbage collector service for verification and alternative
                    // clean-up.
                    ServiceLookupRegistry.get().getService(RT2GarbageCollector.class).registerDocRefCountForRemovalVerification(docUID);

                    log.debug("RT2: join context - detected ref-count set by gc process - join must be aborted");
               		throw new RT2TypedException(ErrorCode.GENERAL_SYSTEM_BUSY_ERROR, getMsgsAsList());
                }
            } else {
            	// we didn't get the lock -> bail out with an error
           		throw new RT2TypedException(ErrorCode.GENERAL_SYSTEM_BUSY_ERROR, getMsgsAsList());
            }
        }
        finally {
        	if (locked) {
                log.debug("Unlock call by client [{}] for doc with id [{}]", clientUID, docUID);
                clusterLock.unlock();
        	}
        }
        if (triggerGc) {
        	if (!ServiceLookupRegistry.get().getService(RT2GarbageCollector.class).doGcForDocUid(docUID, true)) {
        		throw new RT2TypedException(ErrorCode.GENERAL_SYSTEM_BUSY_ERROR, getMsgsAsList());
        	}
        	join(aJoinRequest);
        	return;
        }
        // forward join request to doc processor (which might exist on a different node).
        sendRequest (aJoinRequest);
    }

    //-------------------------------------------------------------------------
    public Deferred< RT2Message > leave (final RT2Message aLeaveRequest) throws Exception {
        LogMethodCallHelper.logMethodCall(this, "leave", aLeaveRequest.getHeader());
        this.messageCounters.incLeaveRequestCounter();
       	docProxyStateHolder.checkDocumentAlreadyDisposed(getMsgsAsList());

        Boolean bForce = RT2MessageGetSet.getInternalHeader(aLeaveRequest, RT2Protocol.HEADER_AUTO_CLOSE, false) ||
                aLeaveRequest.isType(RT2MessageType.REQUEST_EMERGENCY_LEAVE);
        docProxyStateHolder.setForceQuit(bForce);
        docProxyStateHolder.switchProxyStateOrFail(EDocProxyState.E_IN_LEAVE);
        return sendRequestWithDefered(aLeaveRequest);
    }

    //-------------------------------------------------------------------------
    public void open (final RT2Message aOpenRequest)
        throws Exception
    {
        LogMethodCallHelper.logMethodCall(this, "open", aOpenRequest.getHeader());
        this.messageCounters.incOpenRequestCounter();
        docProxyStateHolder.switchProxyStateOrFail(EDocProxyState.E_IN_OPEN);
        sendRequest (aOpenRequest);
    }

    //-------------------------------------------------------------------------
    public Deferred< RT2Message > close (final RT2Message aCloseRequest) throws RT2Exception
    {
        LogMethodCallHelper.logMethodCall(this, "close", aCloseRequest.getHeader());
        this.messageCounters.incCloseRequestCounter();
       	docProxyStateHolder.checkDocumentAlreadyDisposed(getMsgsAsList());
        docProxyStateHolder.switchProxyStateOrFail(EDocProxyState.E_IN_CLOSE);
        return sendRequestWithDefered(aCloseRequest);
    }

    //-------------------------------------------------------------------------
    public void abortOpen (final RT2Message aAbortOpenRequest) throws RT2TypedException
    {
    	LogMethodCallHelper.logMethodCall(this, "abortOpen", aAbortOpenRequest.getHeader());
    	this.messageCounters.incAbortCounter();
   		docProxyStateHolder.checkDocumentAlreadyDisposed(getMsgsAsList());
       	docProxyStateHolder.setAbortOpen(true);
        sendRequest (aAbortOpenRequest);
    }

	//-------------------------------------------------------------------------
	public RT2SessionIdType getSessionId() {
		return sessionId.get();
	}

	//-------------------------------------------------------------------------
	public void setSessionId(RT2SessionIdType sessionId) {
		this.sessionId.set(sessionId);
	}

    //-------------------------------------------------------------------------
    public long getOfflineTimeInMS () {
    	return docProxyConnectionStateHolder.getOfflineTimeInMS();
    }

    //-------------------------------------------------------------------------
    public EDocProxyState getDocState() {
        return docProxyStateHolder.getProxyState();
    }

    //-------------------------------------------------------------------------
    public long getCreationTimeInMS() {
        return creationTime;
    }

  //-------------------------------------------------------------------------
    public RT2ChannelId getChannelId() {
		return channelId;
	}

	//-------------------------------------------------------------------------
    public boolean isOnline() {
    	return docProxyConnectionStateHolder.isOnline();
    }

    //-------------------------------------------------------------------------
    public void goOffline () {
    	if (docProxyConnectionStateHolder.isOnline()) {
	    	synchronized (msgs) {
	    		msgs.add(new RT2DocProxyLogInfo(Direction.INTERNAL, "goOffline", clientUID, docUID));
	    	}
	        docProxyConnectionStateHolder.goOffline();
    	}
    }

    //-------------------------------------------------------------------------
    public void goOnline () {
    	if (docProxyConnectionStateHolder.isOffline()) {
	    	synchronized (msgs) {
	    		msgs.add(new RT2DocProxyLogInfo(Direction.INTERNAL, "goOnline", clientUID, docUID));
	    	}
	    	log.debug("RT2: doc proxy [{}"+getProxyID()+"] is offline : enable online mode. ClientUid: [{}], DocUid: [{}]", getProxyID(), getClientUID(), getDocUID());
	        docProxyConnectionStateHolder.goOnline();
    	}
    }

    //-------------------------------------------------------------------------
    public void notifyUnavailibility(Direction direction, long nUnavailTime)
        throws Exception
    {
    	synchronized (msgs) {
    		msgs.add(new RT2DocProxyLogInfo(direction, "notifyUnavailibility", clientUID, docUID));
    	}
        LogMethodCallHelper.logMethodCall(this, "notifyUnavailibility", nUnavailTime);
        final RT2Message aUnavailRequest = RT2MessageFactory.newUnavailabilityRequest(getClientUID(), getDocUID(), new RT2UnavailableTimeType(nUnavailTime));
        sendRequest(aUnavailRequest);
    }

    //-------------------------------------------------------------------------
    public void sendCrashedResponseToClient () throws RT2Exception, RT2TypedException {
        LogMethodCallHelper.logMethodCall(this, "sendCrashedResponseToClient", sysShutdown.get(), docProxyStateHolder.isInShutdown());
        if (!sysShutdown.get() && !docProxyStateHolder.isInShutdown()) {
            LogMethodCallHelper.logMethodCall(this, "sendCrashedResponseToClient");
            final RT2Message aCrashedBroadcast = RT2MessageFactory.newBroadcastMessage(getClientUID(), getDocUID(), RT2MessageType.BROADCAST_CRASHED);
            forwardResponseToHandler(aCrashedBroadcast);
        }
    }

    //-------------------------------------------------------------------------
    public void handleResponse(RT2Message aResponse) {
        LogMethodCallHelper.logMethodCall(this, "handleResponse", aResponse.getHeader());
        synchronized (msgs) {
        	msgs.add(new RT2DocProxyLogInfo(Direction.FROM_JMS, aResponse));
        }
        try
        {
            // count request/response pairs (if possible : e.g. broadcasts cant and has not to be counted)
            final boolean bIsFinalResponse = aResponse.isFinalMessage();
            if (bIsFinalResponse){
                m_nOutStandingRequests.decrementAndGet();
            }
            boolean bForwardResponseToHandler = true;
            switch (aResponse.getType()) {
                case RESPONSE_UNAVAILABILITY: bForwardResponseToHandler = false;
                    break;
                case RESPONSE_JOIN:
                	docProxyStateHolder.switchProxyStateOrFail(EDocProxyState.E_JOINED);
                	this.messageCounters.incJoinResponseCounter();
                    break;
                case RESPONSE_OPEN_DOC:
                	docProxyStateHolder.switchProxyStateOrFail(EDocProxyState.E_OPEN);
                	this.messageCounters.incOpenResponseCounter();
                    break;
                case RESPONSE_CLOSE_DOC:
                	docProxyStateHolder.switchProxyStateOrFail(EDocProxyState.E_CLOSED);
                	this.messageCounters.incCloseResponseCounter();
                    break;
                case RESPONSE_LEAVE:
                case RESPONSE_EMERGENCY_LEAVE:
                	handleFinalLeave(aResponse);
                	this.messageCounters.incLeaveResponseCounter();
                    break;
                case BROADCAST_SHUTDOWN: handleRemoteNodeShutdown (aResponse);
                    break;
                default: /* nothing to do */;
            }
            forwardResponseToDeferreds (aResponse);

            if (bForwardResponseToHandler)
                forwardResponseToHandler   (aResponse);
        }
        catch (Throwable ex)
        {
            log.error("RT2: Exception caught in handleResponse. msgs: {}", this.formatMsgsLogInfo(), ex);
            ExceptionUtils.handleThrowable(ex);
            handleResponseErrors (aResponse, ex);
        }
    }

    //-------------------------------------------------------------------------
    public Deferred< RT2Message > sendRequestWithDefered (final RT2Message aRequest) throws RT2TypedException
    {
        m_nOutStandingRequests.incrementAndGet();

        // For sync request check if proxy state is sufficient to handle it
        if (RT2MessageType.REQUEST_SYNC.equals(aRequest.getType()) && !docProxyStateHolder.isProxyInOpenState()) {
       		throw new RT2TypedException(ErrorCode.GENERAL_DOCUMENT_ALREADY_DISPOSED_ERROR, getMsgsAsList());
        }

        final Deferred< RT2Message > aDeferred  = new Deferred<> ();
        deferreds.put (aRequest.getMessageID(), aDeferred);
       	jmsMessageSender.sendToDocumentQueue(aRequest, getMsgsAsList());
        return aDeferred;
    }

    //-------------------------------------------------------------------------
    public void sendRequest (final RT2Message aRequest) throws RT2TypedException
    {
        m_nOutStandingRequests.incrementAndGet();

        // For sync request check if proxy state is sufficient to handle it
        if (RT2MessageType.REQUEST_SYNC.equals(aRequest.getType()) && !docProxyStateHolder.isProxyInOpenState()) {
        	throw new RT2TypedException(ErrorCode.GENERAL_DOCUMENT_ALREADY_DISPOSED_ERROR, getMsgsAsList());
        }
        jmsMessageSender.sendToDocumentQueue(aRequest, getMsgsAsList());
    }


    //-------------------------------------------------------------------------
    private void forwardResponseToDeferreds(final RT2Message aResponse) throws Exception {
        final Deferred< RT2Message > aDeferred  = deferreds.get(aResponse.getMessageID());
        if (aDeferred != null) {
            aDeferred.resolve(aResponse);
            if (aDeferred.isClosed())
                deferreds.remove (aResponse.getMessageID());
        }
    }

    //-------------------------------------------------------------------------
    public void forceCleanup () throws RT2Exception {
    	synchronized (msgs) {
    		msgs.add(new RT2DocProxyLogInfo(Direction.INTERNAL, "forceCleanup", clientUID, docUID));
    	}
        docProxyStateHolder.switchProxyStateOrFail(EDocProxyState.E_SHUTDOWN);
    }

    //-------------------------------------------------------------------------
    private void handleResponseErrors (final RT2Message aMessage, final Throwable  aEx) {
    	LogMethodCallHelper.logMethodCall(this, "handleResponseErrors", aMessage.getHeader(), aEx);
        try {
            final RT2Message aErrorResponse = RT2MessageFactory.cloneMessage(aMessage, RT2MessageType.RESPONSE_GENERIC_ERROR);
                  ErrorCode  aError         = ErrorCode.GENERAL_UNKNOWN_ERROR;

            if (RT2TypedException.class.isAssignableFrom(aEx.getClass ())) {
                final RT2TypedException aRT2Ex = (RT2TypedException) aEx;
                aError = aRT2Ex.getError();
            }
            RT2MessageGetSet.setError(aErrorResponse, aError);
            forwardResponseToHandler   (aErrorResponse);
        }
        catch (Throwable exIgnore) {
            ExceptionUtils.handleThrowable(exIgnore);
            log.error(exIgnore.getMessage() + ", proxyId: " + getProxyID(), exIgnore);
            // error in error (handling) ?
            // Do not do anything then log those error for debug purposes.
        }
    }

    //-------------------------------------------------------------------------
    public void handleFinalLeave (final RT2Message aLeaveResponse)
    {
        try {
            final String       sDocUID    = RT2MessageGetSet.getDocUID   (aLeaveResponse);
            final String       sClientUID = RT2MessageGetSet.getClientUID(aLeaveResponse);
            final RT2DocInfo   aDocInfo   = docInfoRegistry.getDocInfo(aLeaveResponse.getDocUID());
            final RT2DocUidType docUid 	  = new RT2DocUidType(sDocUID);

            final ClusterLock clusterLock = clusterLockService.getLock(docUID);
            boolean locked = false;
            try {
            	locked = clusterLock.lock();
            	if (locked) {
                    log.debug("tryLock for client with id [{}] and document with id [{}]", sClientUID, sDocUID);

                    final long nClientRefCount = aDocInfo.decRefCount4Clients();
                    log.debug("Lock aquired...clientRefCount: {}", nClientRefCount);

                    if (nClientRefCount <= 0L) {
                    	aDocInfo.setRefCount4Clients(RT2Constants.IN_REMOVE_REF_COUNT_TRESHHOLD);
                        log.debug("Last leave for doc with id {}", docUid);
                        final RT2Message  aCloseTask = RT2MessageFactory.newAdminMessage(RT2MessageType.ADMIN_TASK_CLOSE_DOC_PROCESSOR);
                        aCloseTask.setDocUID(aLeaveResponse.getDocUID());
                        ServiceLookupRegistry.get().getService(RT2AdminJmsConsumer.class).send(aCloseTask);
                        docInfoRegistry.freeDocInfos(aLeaveResponse.getDocUID());
                    }
            	} else {
                    log.warn("Lock couldn't acquired in for id[{}] and document id [{}]", sClientUID, sDocUID);
            	}
            }
            finally {
                deregisterAndSwitchToLeftState(docProxyRegistry, aLeaveResponse);
                log.debug("RT2: local close finished - unlock call for client with id [{}] and document with id [{}]", sClientUID, sDocUID);
                if (locked) {
                    clusterLock.unlock();
                }
                m_LeaveSyncPoint.countDown();
            }
            log.debug("... unblock join/leave sync point");
            m_bJoinLeaveSyncPoint.countDown();
        } catch (Throwable ex) {
            ExceptionUtils.handleThrowable(ex);
            log.error(ex.getMessage() + ", msgs: {}", formatMsgsLogInfo(), ex);
        }
    }

    //-------------------------------------------------------------------------
    private void deregisterAndSwitchToLeftState(final RT2DocProxyRegistry aDocRegistry, final RT2Message aLeaveResponse) throws RT2Exception
    {
        LogMethodCallHelper.logMethodCall(this, "deregisterAndSwitchToLeftState", aLeaveResponse.getHeader());

        // now it's time to deregister THIS proxy from global registry
        aDocRegistry.deregisterDocProxy(this, false);
        docProxyStateHolder.setForceQuit(false);
        docProxyStateHolder.switchProxyStateOrFail(EDocProxyState.E_LEFT);
    }

    //-------------------------------------------------------------------------
    private void handleRemoteNodeShutdown (final RT2Message aShutdownMsg) throws Exception {
        LogMethodCallHelper.logMethodCall(this, "handleRemoteNodeShutdown", aShutdownMsg.getHeader());

        final RT2DocUidType docUID      = aShutdownMsg.getDocUID();
        final boolean       isRemoteDoc = !(ServiceLookupRegistry.get().getService(IRT2DocProcessorManager.class).contains(docUID));

        // ATTENTION:
        // We only handle shutdown notifications of remote nodes - if our
        // own node is going to be shutdown, then doc proxies will be closed
        // by RT2Activator.deactivate()!
        if (isRemoteDoc) {
        	final ClusterLock clusterLock = clusterLockService.getLock(docUID);

        	boolean locked = false;
            try {
                rt2NodeInfo.setNodeShutdown(true);

                final RT2DocInfo aDocInfo = docInfoRegistry.getDocInfo(docUID);

                log.debug("Lock call for remote node shutdown of RT2DocProxy [{}]", this);
                locked = clusterLock.lock();
                if (locked) {
                    final long        nClientRefCount = aDocInfo.decRefCount4Clients();
                    final boolean     bLastLeave      = (nClientRefCount <= 0L);

                    if (bLastLeave)
                    	docInfoRegistry.freeDocInfos(aShutdownMsg.getDocUID());

                    // now it's time to deregister THIS proxy from global registry
                    docProxyRegistry.deregisterDocProxy(this, true);
                    log.debug("RT2DocProxy [{}] was shutdown...", this);
                    docProxyStateHolder.switchProxyStateOrFail(EDocProxyState.E_SHUTDOWN);
                } else {
                	log.warn("RT2DocProxy [{}] couldn't get lock for document id [{}] for client uid [{}]", docUID, getClientUID());
                }
            } finally {
            	if (locked) {
                	clusterLock.unlock();
                    log.debug("unlock called for remote node shutdown");
            	}

                rt2NodeInfo.setNodeShutdown(false);
                m_bJoinLeaveSyncPoint.countDown();
            }
        }
    }

    //-------------------------------------------------------------------------
    private void syncJoinWithLeaveIfNeeded () throws RT2TypedException, InterruptedException {

        if ( !docProxyStateHolder.isInShutdown () && !rt2NodeInfo.isNodeShutdown()) {
            return;
        }

        log.info("RT2Proxy [{}] is in shutdown - wait for finish", this);

        // wait for final leave signal 5s max
        // This feature is not thought to handle long running save operations within close/leave !
        // It's thought to handle "fast reload clashes" in some rare situations !

        final boolean bOK = m_bJoinLeaveSyncPoint.await(5000, TimeUnit.MILLISECONDS);
        if ( ! bOK) {
        	log.error("System busy error!", new RT2TypedException (ErrorCode.GENERAL_SYSTEM_BUSY_ERROR, getMsgsAsList()));
        	throw new RT2TypedException (ErrorCode.GENERAL_SYSTEM_BUSY_ERROR, getMsgsAsList());
        }
        m_bJoinLeaveSyncPoint = new CountDownLatch((int) m_bJoinLeaveSyncPoint.getCount());
    }

    //-------------------------------------------------------------------------
    /** response handler is a 'dynamic resource'.
     *  It will be (de)registered from RT2WSChannel at runtime on cached RT2DocProxy
     *  instances. So it can happen there is no response handler known for some milliseconds.
     *
     *  On the other side we are on a stack where camel read a message from JMS and pass it to this method.
     *  If we fail ... this message is lost.
     *
     *  So what can we do ?
     *
     *  We can retry to find a suitable (might be new registered) response handler.
     *  We do it synchronous to be on the JMS-Camel-Process-Chain.
     *  But we give up after some time.
     *
     *  @param  aResponse [IN]
     *          the response to be forwarded to our response handler.
     */
    private void forwardResponseToHandler (final RT2Message aResponse) throws RT2Exception, RT2TypedException {
    	log.debug("forwardResponseToHandler: {}", aResponse.getHeader());
    	channelDisposer.addMessageToResponseQueue(aResponse);
    }

    //-------------------------------------------------------------------------
    @Override
    public String toString() {
    	String res = "RT2DocProxy [ClientUID=" + clientUID + ", DocUID=" + docUID + docProxyStateHolder + docProxyConnectionStateHolder;
        List<String> msgsLogInfos = formatMsgsLogInfo();
        if (!msgsLogInfos.isEmpty()) {
        	res += msgsLogInfos.toString();
        }
        return res;
    }

    //-------------------------------------------------------------------------
    @Override
    public Map<String, Object> getAdditionalLogInfo() {
        Map<String, Object> res = new HashMap<>();
        res.put("clientUid", clientUID);
        res.put("docUid", docUID);
        return res;
    }

  //-------------------------------------------------------------------------
    public List<String> formatMsgsLogInfo() {
    	synchronized (msgs) {
    		List<RT2DocProxyLogInfo> res = new ArrayList<>();
    		res.addAll(this.msgs);
    		Collections.sort(res);
    		return res.stream().map(m -> m.toString()).collect(Collectors.toList());
    	}
    }

    private List<RT2LogInfo> getMsgsAsList() {
    	synchronized (msgs) {
    		return new ArrayList<>(msgs);
		}
    }

    //-------------------------------------------------------------------------
    @Override
    public Logger getLogger() {
        return log;
    }

    //-------------------------------------------------------------------------
    public String getProxyID () {
        return proxyID;
    }

    //-------------------------------------------------------------------------
    public RT2CliendUidType getClientUID () {
        return clientUID;
    }

    //-------------------------------------------------------------------------
    public RT2DocUidType getDocUID () {
        return docUID;
    }

    //-------------------------------------------------------------------------
    public int getOutStandingRequests () {
        return m_nOutStandingRequests.get();
    }

    //-------------------------------------------------------------------------
    public int incCountUsedBy() {
    	return this.countUsedBy.incrementAndGet();
    }

    //-------------------------------------------------------------------------
    public int decCountUsedBy() {
    	return this.countUsedBy.decrementAndGet();
    }
}
