package com.openexchange.office.rt2.jms;

import javax.jms.ConnectionFactory;
import javax.jms.Message;
import javax.jms.MessageListener;
import javax.validation.ValidationException;

import org.apache.activemq.command.ActiveMQQueue;
import org.apache.activemq.jms.pool.PooledConnectionFactory;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.core.task.SimpleAsyncTaskExecutor;
import org.springframework.jms.listener.DefaultMessageListenerContainer;
import org.springframework.util.backoff.ExponentialBackOff;
import com.openexchange.log.LogProperties;
import com.openexchange.office.rt2.core.RT2ClientUidUnknownException;
import com.openexchange.office.rt2.core.RT2NodeInfoService;
import com.openexchange.office.rt2.core.RT2ThreadFactoryBuilder;
import com.openexchange.office.rt2.core.doc.IRT2DocProcessorManager;
import com.openexchange.office.rt2.core.doc.RT2DocProcessor;
import com.openexchange.office.rt2.core.doc.RT2DocProcessorFactory;
import com.openexchange.office.rt2.core.doc.RT2DocProcessorManager;
import com.openexchange.office.rt2.core.metric.DocProcessorMetricService;
import com.openexchange.office.rt2.protocol.RT2Message;
import com.openexchange.office.rt2.protocol.RT2MessageFactory;
import com.openexchange.office.rt2.protocol.RT2MessageGetSet;
import com.openexchange.office.rt2.protocol.value.RT2MessageType;
import com.openexchange.office.tools.osgi.ServiceLookupRegistryService;

public class RT2DocProcessorJmsConsumer implements MessageListener {

    private static final Logger log = LoggerFactory.getLogger(RT2DocProcessorJmsConsumer.class);

    private final ConnectionFactory connFactory;

    private final IRT2DocProcessorManager docProcessorManager;

    private final String nodeUUID;

    private final ActiveMQQueue queue;

    private final DocProcessorMetricService docProcessorMetricService;

    private DefaultMessageListenerContainer msgListenerCont;

    public RT2DocProcessorJmsConsumer() {
        final RT2NodeInfoService nodeInfoService = ServiceLookupRegistryService.getInstance().getService(RT2NodeInfoService.class);
        this.nodeUUID = nodeInfoService.getNodeUUID();
        this.docProcessorManager = ServiceLookupRegistryService.getInstance().getService(IRT2DocProcessorManager.class);
        this.connFactory = ServiceLookupRegistryService.getInstance().getService(PooledConnectionFactory.class);
        String queueName = JmsComponentFactory.DOCUMENT_QUEUE_NAME + nodeUUID;
        this.queue = new ActiveMQQueue(queueName);
        log.debug("Consuming from queue with name {}", queueName);
        this.docProcessorMetricService = ServiceLookupRegistryService.getInstance().getService(DocProcessorMetricService.class);
    }

    // For unit test purposes only
    public RT2DocProcessorJmsConsumer(String nodeUUID, ConnectionFactory connFactory, RT2DocProcessorManager docProcMgr) {
        this.docProcessorManager = docProcMgr;
        this.connFactory = connFactory;
        this.nodeUUID = nodeUUID;
        String queueName = JmsComponentFactory.DOCUMENT_QUEUE_NAME + nodeUUID;
        this.queue = new ActiveMQQueue(queueName);
        log.debug("Consuming from queue with name {}", queueName);
        this.docProcessorMetricService = ServiceLookupRegistryService.getInstance().getService(DocProcessorMetricService.class);
    }

    @Override
    public void onMessage(Message jmsMsg) {
        try {
            RT2Message rt2Msg = null;
            try {
                rt2Msg = RT2MessageFactory.fromJmsMessage(jmsMsg);
                docProcessorMetricService.startTimer(rt2Msg.getMessageID());
                LogProperties.putProperty(LogProperties.Name.RT2_DOC_UID, rt2Msg.getDocUID().getValue());
                LogProperties.putProperty(LogProperties.Name.RT2_CLIENT_UID, rt2Msg.getClientUID().getValue());
                LogProperties.putProperty(LogProperties.Name.RT2_BACKEND_PART, "processor");
                LogProperties.putProperty(LogProperties.Name.RT2_BACKEND_UID, nodeUUID);
                LogProperties.putProperty(LogProperties.Name.RT2_REQUEST_TYPE, rt2Msg.getType().getValue());
            } catch (Exception ex) {
                log.error(ex.getMessage(), ex);
                return;
            }

            log.debug("Received msg {}", rt2Msg);
            RT2DocProcessor rt2DocProc = docProcessorManager.getDocProcessor(rt2Msg.getDocUID());
            if (rt2DocProc == null) {
                if (RT2MessageType.REQUEST_JOIN.equals(rt2Msg.getType())) {
	                String docType = null;
	                try {
	                	docType = RT2MessageGetSet.getDocType(rt2Msg);
	                    log.debug("Creating new docProcessor for doc with id {}", rt2Msg.getDocUID());
	                    rt2DocProc = RT2DocProcessorFactory.createInstance(docType, rt2Msg.getDocUID());
	                    docProcessorManager.docProcessorCreated(rt2DocProc);
	                } catch (ValidationException ex) {
	                	log.debug("No doc processor found for request message {}", rt2Msg);
	                }
                }
            }

            try {
            	if (rt2DocProc != null) {
            		rt2DocProc.process(rt2Msg);
            	}
            } catch (RT2ClientUidUnknownException ex) {
            	// Nothing todo, it is possible under some circumstances
            } catch (Exception ex) {
                log.error("Exception during processing RT2Message " + rt2Msg + ": " + ex.getMessage(), ex);
            }
        } finally {
            LogProperties.removeLogProperties();
        }
    }

    public synchronized void startReceiveMessages() {
    	if (msgListenerCont == null) {
            msgListenerCont = new DefaultMessageListenerContainer();
            ExponentialBackOff exponentialBackOff = new ExponentialBackOff();
            exponentialBackOff.setMaxInterval(60000);
            msgListenerCont.setBackOff(exponentialBackOff);
            msgListenerCont.setConnectionFactory(connFactory);
            msgListenerCont.setConcurrentConsumers(3);
            msgListenerCont.setDestination(this.queue);
            msgListenerCont.setMaxConcurrentConsumers(3);
            msgListenerCont.setPubSubDomain(false);
            msgListenerCont.setAutoStartup(true);
            msgListenerCont.setupMessageListener(this);
            msgListenerCont.setTaskExecutor(new SimpleAsyncTaskExecutor(new RT2ThreadFactoryBuilder("RT2DocProcessorJmsConsumer-%d").build()));
            msgListenerCont.afterPropertiesSet();
            msgListenerCont.start();
    	}
    }

    public synchronized void destroy() {
        if (msgListenerCont != null) {
            msgListenerCont.destroy();
        }
    }

}
