package com.openexchange.office.rt2.jms;

import javax.jms.JMSException;
import javax.jms.Message;
import javax.jms.MessageListener;

import org.apache.activemq.ActiveMQConnectionFactory;
import org.apache.activemq.command.ActiveMQTextMessage;
import org.apache.activemq.jms.pool.PooledConnectionFactory;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.core.task.SimpleAsyncTaskExecutor;
import org.springframework.jms.listener.DefaultMessageListenerContainer;
import org.springframework.util.backoff.ExponentialBackOff;

import com.codahale.metrics.Counter;
import com.codahale.metrics.MetricRegistry;
import com.openexchange.log.LogProperties;
import com.openexchange.office.rt2.core.RT2NodeInfoService;
import com.openexchange.office.rt2.core.RT2ThreadFactoryBuilder;
import com.openexchange.office.tools.osgi.ServiceLookupRegistry;

public class RT2DcsUpdateConsumer implements MessageListener {

	private static final Logger log = LoggerFactory.getLogger(RT2DcsUpdateConsumer.class);

	private final ActiveMQConnectionFactory connFactory;
	
	private final RT2NodeInfoService nodeInfo;
	
    private DefaultMessageListenerContainer msgListenerCont;
    
    private Counter counter;
    
    public RT2DcsUpdateConsumer(PooledConnectionFactory connFactory, RT2NodeInfoService nodeInfo) {
    	this.connFactory = (ActiveMQConnectionFactory) connFactory.getConnectionFactory();
    	this.nodeInfo = nodeInfo;
    }    
    
	@Override
	public void onMessage(Message msg) {
        LogProperties.putProperty(LogProperties.Name.RT2_BACKEND_PART, "RT2DcsUpdateConsumer");
        LogProperties.putProperty(LogProperties.Name.RT2_BACKEND_UID, nodeInfo.getNodeUUID());      	        
		counter.inc();
		ActiveMQTextMessage txtMsg = (ActiveMQTextMessage) msg;
		try {
			String newBrokerUrl = txtMsg.getText();
			log.info("Received new brokerURl: {}", newBrokerUrl);
			connFactory.setBrokerURL(newBrokerUrl);			
		} catch (JMSException ex) {
			log.error(ex.getErrorCode(), ex);
		}
	}
	
    //-------------------------------------------------------------------------
    public synchronized void receiveDcsUpdateMessages() {
        if (msgListenerCont == null) {
        	MetricRegistry metricRegistry = ServiceLookupRegistry.get().getService(MetricRegistry.class);
        	this.counter = metricRegistry.counter(MetricRegistry.name("RT2DcsUpdateConsumer", "count"));
        	
            msgListenerCont = new DefaultMessageListenerContainer();
            ExponentialBackOff exponentialBackOff = new ExponentialBackOff();
            exponentialBackOff.setMaxInterval(60000);
            msgListenerCont.setBackOff(exponentialBackOff);            
            msgListenerCont.setConnectionFactory(connFactory);
            msgListenerCont.setConcurrentConsumers(3);
            msgListenerCont.setDestination(JmsDestination.brokerChangedTopic);
            msgListenerCont.setMaxConcurrentConsumers(1);
            msgListenerCont.setPubSubDomain(true);
            msgListenerCont.setAutoStartup(true);
            msgListenerCont.setupMessageListener(this);
            msgListenerCont.setTaskExecutor(new SimpleAsyncTaskExecutor(new RT2ThreadFactoryBuilder("RT2DcsUpdateConsumer-%d").build()));
            msgListenerCont.afterPropertiesSet();            
            msgListenerCont.start();
        }
    }	
    
    public synchronized void stopReceivingDcsUpdateMessages() {
        if (msgListenerCont != null) {
            msgListenerCont.destroy();
            msgListenerCont = null;
        }
    }
}
