/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */
package com.openexchange.office.rt2.jms;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.net.InetAddress;
import java.net.UnknownHostException;
import java.util.HashSet;
import java.util.Map;
import java.util.Properties;
import java.util.Set;
import java.util.UUID;
import java.util.stream.Collectors;

import javax.jms.BytesMessage;
import javax.jms.JMSException;
import javax.jms.Message;
import javax.jms.MessageListener;

import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.core.task.SimpleAsyncTaskExecutor;
import org.springframework.jms.JmsException;
import org.springframework.jms.core.JmsTemplate;
import org.springframework.jms.listener.DefaultMessageListenerContainer;
import org.springframework.util.backoff.ExponentialBackOff;

import com.google.protobuf.InvalidProtocolBufferException;
import com.openexchange.log.LogProperties;
import com.openexchange.office.rt2.cache.ClusterLockException;
import com.openexchange.office.rt2.cache.ClusterLockService;
import com.openexchange.office.rt2.cache.ClusterLockService.ClusterLock;
import com.openexchange.office.rt2.cache.RT2DocInfo;
import com.openexchange.office.rt2.config.RT2ConfigItem;
import com.openexchange.office.rt2.core.RT2Constants;
import com.openexchange.office.rt2.core.RT2DocProcessorClientExistsTester;
import com.openexchange.office.rt2.core.RT2DocProcessorExistsTester;
import com.openexchange.office.rt2.core.RT2NodeInfoService;
import com.openexchange.office.rt2.core.RT2ThreadFactoryBuilder;
import com.openexchange.office.rt2.core.control.IRT2NodeHealthManager;
import com.openexchange.office.rt2.core.doc.DocProcessor;
import com.openexchange.office.rt2.core.doc.RT2DocProcessor;
import com.openexchange.office.rt2.core.doc.RT2DocProcessorManager;
import com.openexchange.office.rt2.hazelcast.RT2NodeHealthMap;
import com.openexchange.office.rt2.protocol.GpbMessageJmsPostProcessor;
import com.openexchange.office.rt2.protocol.RT2GoogleProtocol.AdminMessage;
import com.openexchange.office.rt2.protocol.RT2GoogleProtocol.AdminMessageType;
import com.openexchange.office.rt2.protocol.RT2GoogleProtocol.ClientInfo;
import com.openexchange.office.rt2.protocol.RT2GoogleProtocol.ClientUidType;
import com.openexchange.office.rt2.protocol.RT2GoogleProtocol.DocUidType;
import com.openexchange.office.rt2.protocol.RT2GoogleProtocol.MessageTimeType;
import com.openexchange.office.rt2.protocol.RT2GoogleProtocol.ServerIdType;
import com.openexchange.office.rt2.protocol.RT2GoogleProtocol.SessionIdType;
import com.openexchange.office.rt2.protocol.RT2GoogleProtocol.VersionMessageRequest;
import com.openexchange.office.rt2.protocol.RT2GoogleProtocol.VersionMessageResponse;
import com.openexchange.office.rt2.protocol.RT2Message;
import com.openexchange.office.rt2.protocol.RT2MessageFactory;
import com.openexchange.office.rt2.protocol.RT2MessageJmsPostProcessor;
import com.openexchange.office.rt2.protocol.RT2MessagePostProcessor;
import com.openexchange.office.rt2.protocol.value.RT2AdminIdType;
import com.openexchange.office.rt2.protocol.value.RT2CliendUidType;
import com.openexchange.office.rt2.protocol.value.RT2DocUidType;
import com.openexchange.office.rt2.protocol.value.RT2SessionIdType;
import com.openexchange.office.rt2.proxy.RT2DocInfoRegistry;
import com.openexchange.office.rt2.proxy.RT2DocProxy;
import com.openexchange.office.rt2.proxy.RT2DocProxyRegistry;
import com.openexchange.office.tools.osgi.ServiceLookupRegistry;

//=============================================================================
/**
 */
public class RT2AdminJmsConsumer implements MessageListener {
    private static final Logger log = LoggerFactory.getLogger(RT2AdminJmsConsumer.class);

    //-------------------------------------------------------------------------

    private final RT2AdminIdType adminId;

    private final JmsTemplate jmsTemplate;
    
    private final RT2DocProcessorManager docProcMngr;    

    private final RT2DocInfoRegistry docInfoRegistry;
    
    private final RT2DocProxyRegistry docProxyRegistry;

    private final IRT2NodeHealthManager nodeHealthManager;
    
    private final RT2NodeInfoService nodeInfo;
    
    private final RT2NodeHealthMap nodeHealthMap;
    
    private final RT2DocProcessorClientExistsTester docProcClientExistsTester;
    
    private final RT2DocProcessorExistsTester docProcessorExistsTester;
    
    private final Properties versionProperties;
    
    private final ClusterLockService clusterLockService;
    
    private final JmsMessageSender jmsMessageSender;
    
    private DefaultMessageListenerContainer msgListenerCont;     

    public RT2AdminJmsConsumer(JmsTemplate jmsTemplate, RT2DocProcessorManager docProcMngr, RT2NodeInfoService nodeInfo,  RT2DocInfoRegistry docInfoRegistry, 
    		                   RT2DocProxyRegistry docProxyRegistry, IRT2NodeHealthManager nodeHealthManager,   
    		                   RT2DocProcessorClientExistsTester docProcClientExistsTester, RT2DocProcessorExistsTester docProcessorExistsTester,
    		                   Properties versionProperties, ClusterLockService clusterLockService, JmsMessageSender jmsMessageSender) throws IOException {
        this.jmsTemplate = jmsTemplate;
        this.docProcMngr = docProcMngr;
        this.docInfoRegistry = docInfoRegistry;
        this.nodeHealthManager = nodeHealthManager;
        this.docProxyRegistry = docProxyRegistry;
        this.nodeInfo = nodeInfo;
        this.nodeHealthMap = ServiceLookupRegistry.get().getService(RT2NodeHealthMap.class);
        this.docProcClientExistsTester = docProcClientExistsTester;
        this.docProcessorExistsTester = docProcessorExistsTester;
        this.adminId = generateAdminID();
        this.versionProperties = versionProperties;
        this.clusterLockService = clusterLockService;
        this.jmsMessageSender = jmsMessageSender;
    }
    
    //-------------------------------------------------------------------------
    public synchronized void send (final RT2Message aTask) {
        aTask.setAdminID(adminId);
        log.debug("admin send request ''{}''...", aTask);
        String rt2MsgAsStr = aTask.getBodyString();
        jmsTemplate.convertAndSend(JmsDestination.adminTopic, rt2MsgAsStr, new RT2MessageJmsPostProcessor(aTask));
    }

    //-------------------------------------------------------------------------
    public synchronized void receiveAdminMessages() {
        if (msgListenerCont == null) {
            msgListenerCont = new DefaultMessageListenerContainer();
            ExponentialBackOff exponentialBackOff = new ExponentialBackOff();
            exponentialBackOff.setMaxInterval(60000);
            msgListenerCont.setBackOff(exponentialBackOff);            
            msgListenerCont.setConnectionFactory(jmsTemplate.getConnectionFactory());
            msgListenerCont.setConcurrentConsumers(1);
            msgListenerCont.setDestination(JmsDestination.adminTopic);
            msgListenerCont.setMaxConcurrentConsumers(1);
            msgListenerCont.setPubSubDomain(true);
            msgListenerCont.setAutoStartup(true);
            msgListenerCont.setupMessageListener(this);
            msgListenerCont.afterPropertiesSet();
            msgListenerCont.setTaskExecutor(new SimpleAsyncTaskExecutor(new RT2ThreadFactoryBuilder("RT2AdminJmsConsumer-%d").build()));
            msgListenerCont.start();
        }
    }

    //-------------------------------------------------------------------------
    @Override
    public void onMessage(Message jmsMsg) {
        LogProperties.putProperty(LogProperties.Name.RT2_BACKEND_PART, "RT2AdminJmsConsumer");
        LogProperties.putProperty(LogProperties.Name.RT2_BACKEND_UID, nodeInfo.getNodeUUID());      	            	
        log.debug("Received admin message...");
        try {
        	if (jmsMsg.getBooleanProperty(RT2MessagePostProcessor.HEADER_VERSION_MSG_RCV)) {
        		return;
        	}
    		if (jmsMsg.getBooleanProperty(RT2MessagePostProcessor.HEADER_VERSION_MSG_SND)) {
        		BytesMessage byteMsg = (BytesMessage) jmsMsg;
        		byte [] data = new byte[(int)byteMsg.getBodyLength()];
        		byteMsg.readBytes(data);
        		String originator = VersionMessageRequest.parseFrom(data).getOriginator();						
        		String version = "unknown";
        		if ((versionProperties != null) && (!versionProperties.isEmpty())) {
        			version = versionProperties.getProperty("Version") + "(" + versionProperties.get("Buildtime") + ")";
        		}
    			VersionMessageResponse versMsg = VersionMessageResponse.newBuilder()
						.setHost(InetAddress.getLocalHost().getHostName())
						.setVersion(version)
						.setHostId(nodeInfo.getNodeUUID())
						.setOriginator(originator)
						.setCountBackends(nodeHealthMap.getMembersOfState("up").size())
						.build();
    	    	ByteArrayOutputStream baOut = new ByteArrayOutputStream();
    	        try {
    	        	versMsg.writeTo(baOut);
    	        	jmsTemplate.convertAndSend(JmsDestination.adminTopic, baOut.toByteArray(), (newMsg) -> {
    	        		newMsg.setBooleanProperty(RT2MessagePostProcessor.HEADER_VERSION_MSG_SND, false);
    	        		newMsg.setBooleanProperty(RT2MessagePostProcessor.HEADER_VERSION_MSG_RCV, true);
    	        		return newMsg;
    	        	});
    	        } catch (JmsException | IOException ex) {    
    	        	log.error("sending VersionMessageResponse", ex);
    	        }
    	        return;
    		}
        	
        	
	        if (jmsMsg.getBooleanProperty(RT2MessagePostProcessor.HEADER_GPB_MSG)) {
        		BytesMessage byteMsg = (BytesMessage) jmsMsg;
        		byte [] data = new byte[(int)byteMsg.getBodyLength()];
        		byteMsg.readBytes(data);        		
        		AdminMessage adminRequest = AdminMessage.parseFrom(data);
	        	handleResponse(adminRequest);
	        } else {
	        	try {
	        		RT2Message rt2Msg = RT2MessageFactory.fromJmsMessage(jmsMsg);
	        		LogProperties.putProperty(LogProperties.Name.RT2_ADMIN_MSG, rt2Msg.getType());
	        		handleResponse(rt2Msg);
	        	} catch (Exception ex) {
	        		log.error(ex.getMessage(), ex);
	        	}
	        }
        } catch (JMSException | UnknownHostException | InvalidProtocolBufferException ex) {
        	log.error(ex.getMessage(), ex);
        }
    }

    //-------------------------------------------------------------------------
    protected void handleResponse(AdminMessage adminRequest) {

        log.debug("Received admin message ''{}''...", adminRequest);
        switch (adminRequest.getMsgType()) {
        	case UPDATE_CURR_CLIENT_LIST_OF_DOC_PROCESSOR_REQUEST:        		
        		if (!adminRequest.getOriginator().getValue().equals(nodeInfo.getNodeUUID())) {
        			AdminMessage adminResponse = AdminMessage.newBuilder()
        													 .setMsgType(AdminMessageType.UPDATE_CURR_CLIENT_LIST_OF_DOC_PROCESSOR_RESPONSE)
        													 .setMessageTime(MessageTimeType.newBuilder().setValue(System.currentTimeMillis()))
        													 .setServerId(ServerIdType.newBuilder().setValue(nodeInfo.getNodeUUID()))
        													 .setOriginator(adminRequest.getOriginator())
        													 .addAllActiveClients(docProxyRegistry.listAllDocProxies().stream().map(p ->
        													 	ClientInfo.newBuilder()
        													 		.setClientUid(ClientUidType.newBuilder().setValue(p.getClientUID().getValue()))
        													 		.setDocUid(DocUidType.newBuilder().setValue(p.getDocUID().getValue()))
        													 	.build()).collect(Collectors.toSet()))
        													 .build();
        	    	ByteArrayOutputStream baOut = new ByteArrayOutputStream();
        	        try {
        	        	adminResponse.writeTo(baOut);
        	        	jmsTemplate.convertAndSend(JmsDestination.adminTopic, baOut.toByteArray(), new GpbMessageJmsPostProcessor());
        	        } catch (JmsException | IOException jmsEx) {    
        	        	log.error("sendAdminMessage-Exception", jmsEx);
        	        }    	    	        			
        		}
        		break;
        	case UPDATE_CURR_CLIENT_LIST_OF_DOC_PROCESSOR_RESPONSE:
        		if (adminRequest.getOriginator().getValue().equals(nodeInfo.getNodeUUID())) {
        			docProcClientExistsTester.processUpdateCurrClientListOfDocProcResponse(UUID.fromString(adminRequest.getServerId().getValue()), adminRequest.getMessageTime().getValue(), adminRequest.getActiveClientsList());
        		}
        		break;
        	case UPDATE_CURR_DOC_PROCESSORS_REQUEST:
        		if (!adminRequest.getOriginator().getValue().equals(nodeInfo.getNodeUUID())) {
        			AdminMessage adminResponse = AdminMessage.newBuilder()
							 .setMsgType(AdminMessageType.UPDATE_CURR_DOC_PROCESSORS_RESPONSE)
							 .setMessageTime(MessageTimeType.newBuilder().setValue(System.currentTimeMillis()))
							 .setServerId(ServerIdType.newBuilder().setValue(nodeInfo.getNodeUUID()))
							 .setOriginator(adminRequest.getOriginator())
							 .addAllActiveDocProcessors(docProcMngr.getDocProcessors().stream().map(p -> DocUidType.newBuilder().setValue(p.getDocUID().getValue()).build()).collect(Collectors.toSet()))
							 .build();
        			ByteArrayOutputStream baOut = new ByteArrayOutputStream();
        			try {
        				adminResponse.writeTo(baOut);
        				jmsTemplate.convertAndSend(JmsDestination.adminTopic, baOut.toByteArray(), new GpbMessageJmsPostProcessor());
        			} catch (JmsException | IOException jmsEx) {    
        				log.error("sendAdminMessage-Exception", jmsEx);
        			}    	    	        			        			
        		}
        		break;
        	case UPDATE_CURR_DOC_PROCESSORS_RESPONSE:
        		if (adminRequest.getOriginator().getValue().equals(nodeInfo.getNodeUUID())) {
        			docProcessorExistsTester.processUpdateCurrProcResponse(UUID.fromString(adminRequest.getServerId().getValue()), adminRequest.getMessageTime().getValue(), adminRequest.getActiveDocProcessorsList());
        		}
        		break;
        	case ADMIN_TASK_SESSION_REMOVED:
        		if ((adminRequest.getSessionId() != null) && (adminRequest.getSessionId().getValue() != null)) {
        			Set<RT2DocUidType> docUids = new HashSet<>();
        			RT2SessionIdType sessionId = new RT2SessionIdType(adminRequest.getSessionId().getValue());
        			
        			Set<RT2DocProcessor> docProcessors = docProcMngr.getDocProcessorsAssociatedToSessionId(sessionId);        			
        			for (RT2DocProcessor docProcessor : docProcessors) {        				
        				Map<RT2SessionIdType, RT2CliendUidType> sessionToClientUids = docProcessor.getSessionIdToClientUids();        				
        				sessionToClientUids.remove(sessionId);
        				if ((sessionToClientUids.size() > 0) || (docProcessor.getPendingOperationsCount() == 0)) {
       						docUids.add(docProcessor.getDocUID());
        				}
        			}
        			Set<DocUidType> docUidsGpb = new HashSet<>();
        			docUids.forEach(d -> docUidsGpb.add(DocUidType.newBuilder().setValue(d.getValue()).build()));
        			if (!docUids.isEmpty()) {
        				AdminMessage.Builder adminRequestDocProxyRemoveBuilder = AdminMessage.newBuilder()
        						.setMsgType(AdminMessageType.ADMIN_TASK_REMOVE_DOC_PROXIES_WITH_DOC_UUID)
        						.setMessageTime(MessageTimeType.newBuilder().setValue(System.currentTimeMillis()))
        						.setServerId(ServerIdType.newBuilder().setValue(nodeInfo.getNodeUUID()))
        						.setSessionId(SessionIdType.newBuilder().setValue(sessionId.getValue()).build())
        						.setOriginator(ServerIdType.newBuilder().setValue(nodeInfo.getNodeUUID()));        				
        				docUidsGpb.forEach(d -> adminRequestDocProxyRemoveBuilder.addDocProcessorsRemoved(d));
        				AdminMessage adminRequestDocProxyRemove = adminRequestDocProxyRemoveBuilder.build();
        				jmsMessageSender.sendAdminMessage(adminRequestDocProxyRemove);		        				
        			}
        		}        		
        		break;
        	case ADMIN_TASK_REMOVE_DOC_PROXIES_WITH_DOC_UUID:
        		RT2SessionIdType sessionId = new RT2SessionIdType(adminRequest.getSessionId().getValue());
        		adminRequest.getDocProcessorsRemovedList().forEach(d -> {
        			RT2DocUidType docUid = new RT2DocUidType(d.getValue());        			
        			for (RT2DocProxy docProxy : docProxyRegistry.getDocProxies4DocUID(docUid, sessionId)) {        				
        				try {        					
							docProxy.closeHard(false);
						} catch (Exception e) {
							log.error("Cannot close doc Proxy with clientUid {} and docUid {}", docProxy.getClientUID(), docUid, e);
						}
        			}
        		});
        		break;        		
        }
    }
    
    //-------------------------------------------------------------------------
    protected void handleResponse(final RT2Message aTask) {

        log.debug("Received admin message ''{}''...", aTask);

        switch (aTask.getType()) {
            case ADMIN_TASK_CLOSE_DOC_PROCESSOR:
                tryToDisposeDocProcessor(aTask.getDocUID());
                break;
            case ADMIN_TASK_CLEANUP_FOR_CRASHED_NODE:
                nodeHealthManager.startLocalNodeDocCleanup(aTask);
                break;
            case ADMIN_TASK_COMPLETED_CLEANUP_FOR_CRASHED_NODE:
                nodeHealthManager.crashedNodeDocCleanupCompleted(aTask);
                break;
            case ADMIN_TASK_COMPLETED_CLOSE_DOC_ROUTE: // Wrong name!!! Not used anymore so we use this type when a RT2DocProxy was removed.
            	DocProcessor docProcessor = (DocProcessor) docProcMngr.getDocProcessor(aTask.getDocUID());
            	if (docProcessor != null) {
            		docProcessor.removeUser(aTask.getClientUID());
            		docProcessor.deregisterClient(aTask.getClientUID());            		
            	}            	
            	break;
            default:
                log.error("No support for task {} implemented yet.", aTask.getType());
                throw new UnsupportedOperationException ("No support for task '"+aTask.getType()+"' implemented yet.");
        }
    }

    //-------------------------------------------------------------------------
    private void tryToDisposeDocProcessor(RT2DocUidType docUID) {
    	
    	if (docProcMngr.contains(docUID)) {
    		final RT2DocInfo   aDocInfo   = docInfoRegistry.getDocInfo(docUID);
			log.debug("tryLockWithLease for document with id [{}]", docUID);
			final ClusterLock clusterLock = clusterLockService.getLock(docUID);

			boolean locked = false;
    		try {
                locked = clusterLock.lock(RT2Constants.CLUSTERLOCK_WAIT_TIME);
                if (locked) {
                    final long nClientRefCount = aDocInfo.decRefCount4Clients();
                    log.debug("Lock aquired for docUID [{}], clientRefCount: {}", docUID, nClientRefCount);
                    if (nClientRefCount < 0L) {
                    	final RT2DocProcessor docProcessor = docProcMngr.getDocProcessor(docUID);
                        if (docProcessor != null) {
                            docProcessor.dispose();
                        }
                        nodeInfo.deregisterDocOnNode(docUID);
                        if (aDocInfo != null) {
                        	aDocInfo.destroyRefCount4Clients();
                        }
                    }
                } else {
                	log.warn("Lock couldn't be acquired in time for docUID [{}]", docUID);
                }
        	} catch (ClusterLockException e) {
        		log.info("Cannot remove docProcessor with id {}. Try again with other mechanism.", docUID);
        	} finally {
        		if (locked) {
        			clusterLock.unlock();
        		}
            }
    	}
        docInfoRegistry.freeDocInfos(docUID);
    }

    //-------------------------------------------------------------------------

    public static RT2AdminIdType generateAdminID () {
        final RT2ConfigItem aCfg        = RT2ConfigItem.get();
        final String        sNodeID     = aCfg.getOXNodeID();
        String        sNormalized = sNodeID;

        sNormalized = StringUtils.replace(sNormalized, ":" , "-");
        sNormalized = StringUtils.replace(sNormalized, ";" , "-");
        sNormalized = StringUtils.replace(sNormalized, "/" , "-");
        sNormalized = StringUtils.replace(sNormalized, "\\", "-");
        sNormalized = StringUtils.replace(sNormalized, "\"", "-");
        sNormalized = StringUtils.replace(sNormalized, "'" , "-");
        sNormalized = StringUtils.replace(sNormalized, " " , "-");
        return new RT2AdminIdType(sNormalized);
    }

    public synchronized void stop() {
        if (msgListenerCont != null) {
            msgListenerCont.destroy();
        }
    }

}
