/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.rt2.core.sequence;

import java.text.SimpleDateFormat;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import org.apache.commons.lang3.BooleanUtils;
import org.json.JSONArray;
import org.json.JSONException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.openexchange.exception.ExceptionUtils;
import com.openexchange.office.rt2.config.RT2ConfigItem;
import com.openexchange.office.rt2.core.RT2ClientUidUnknownException;
import com.openexchange.office.rt2.core.doc.RT2DocProcessor;
import com.openexchange.office.rt2.exception.RT2TypedException;
import com.openexchange.office.rt2.protocol.RT2GoogleProtocol.BodyType;
import com.openexchange.office.rt2.protocol.RT2GoogleProtocol.BroadcastMessage;
import com.openexchange.office.rt2.protocol.RT2GoogleProtocol.BroadcastMessageReceiver;
import com.openexchange.office.rt2.protocol.RT2GoogleProtocol.ClientUidType;
import com.openexchange.office.rt2.protocol.RT2GoogleProtocol.DocUidType;
import com.openexchange.office.rt2.protocol.RT2GoogleProtocol.MessageIdType;
import com.openexchange.office.rt2.protocol.RT2GoogleProtocol.MessageType;
import com.openexchange.office.rt2.protocol.RT2GoogleProtocol.SequenceNrType;
import com.openexchange.office.rt2.protocol.RT2Message;
import com.openexchange.office.rt2.protocol.RT2MessageFactory;
import com.openexchange.office.rt2.protocol.RT2MessageFlags;
import com.openexchange.office.rt2.protocol.RT2MessageGetSet;
import com.openexchange.office.rt2.protocol.RT2Protocol;
import com.openexchange.office.rt2.protocol.value.RT2CliendUidType;
import com.openexchange.office.rt2.protocol.value.RT2MessageType;
import com.openexchange.office.tools.error.ErrorCode;
import com.openexchange.office.tools.logging.annotation.LogMethodCallHelper;
import com.openexchange.office.tools.logging.annotation.Loggable;
import com.openexchange.office.tools.osgi.ServiceLookupRegistryService;

public abstract class SequenceDocProcessor extends RT2DocProcessor implements Loggable {
	
	private static final Logger log = LoggerFactory.getLogger(SequenceDocProcessor.class);

	private final Map<RT2CliendUidType, ClientSequenceQueue> clientsSeqQueues = Collections.synchronizedMap(new HashMap<>());

	private final QueueProcessorDisposer queueProcessorDisposer;
	
	private final MsgBackupAndACKProcessor ackProcessor;

	private final ClientLeftRemoveThread clientLeftRemoveThread;	
	
	//-------------------------------------------------------------------------
	public SequenceDocProcessor() {
		super();

		ackProcessor = new MsgBackupAndACKProcessor(this);
		ackProcessor.start();

		clientLeftRemoveThread = new ClientLeftRemoveThread(this);
		
		queueProcessorDisposer = ServiceLookupRegistryService.getInstance().getService(QueueProcessorDisposer.class);
	}

	//-------------------------------------------------------------------------
	SequenceDocProcessor(QueueProcessorDisposer queueProcessorDisposer, MsgBackupAndACKProcessor ackProcessor, ClientLeftRemoveThread clientLeftRemoveThread) {
		super();
		this.ackProcessor = ackProcessor;
		this.clientLeftRemoveThread = clientLeftRemoveThread;
		this.queueProcessorDisposer = queueProcessorDisposer;
	}	
	
	//-------------------------------------------------------------------------
	public abstract void handleRequest(final RT2Message aRequest)
	    throws Exception;	
	
	//-------------------------------------------------------------------------
	@Override
	public void dispose() {
		LogMethodCallHelper.logMethodCall(this, "dispose");
		ackProcessor.shutdown();
		clientLeftRemoveThread.shutdown();
	}

	//-------------------------------------------------------------------------
	public Map<RT2CliendUidType, ClientSequenceQueue> getClientsSeqQueues() {
		return new HashMap<>(clientsSeqQueues);
	}

	//-------------------------------------------------------------------------
	public MsgBackupAndACKProcessor getACKProcessor() {
		return ackProcessor;
	}

	//-------------------------------------------------------------------------
	@Override
	public Logger getLogger() {
		return log;
	}

	//-------------------------------------------------------------------------
	@Override
	public Map<String, Object> getAdditionalLogInfo() {
		Map<String, Object> res = new HashMap<>();
		res.put("docUid", getDocUID());
		return res;
	}

	//-------------------------------------------------------------------------
	public boolean hasAbortOpenBeenReceived(RT2CliendUidType sClientUID) throws RT2TypedException {
		return BooleanUtils.toBoolean(getClientSequenceQueueForClient(sClientUID, true).isAborted());
	}

    //-------------------------------------------------------------------------
    @Override
    public void clientAdded(final RT2CliendUidType clientUID) {
        if (clientsSeqQueues.put(clientUID, new ClientSequenceQueue(clientUID, RT2ConfigItem.get().getNackFrequenceOfServer())) != null) {
            log.warn("RT2: Sequence queue overwritten for client UID [{}]!", clientUID);
        }
        ackProcessor.addClient(clientUID);
    }

    //-------------------------------------------------------------------------
    @Override
    public void clientRemoved(final RT2CliendUidType sClientUID) {
    	log.debug("Removing queue for client with id {}", sClientUID);
        ClientSequenceQueue aClientQueue = clientsSeqQueues.remove(sClientUID);
        if (aClientQueue != null) {
        	clientLeftRemoveThread.addClientLeft(sClientUID);
        }
        ackProcessor.removeClient(sClientUID);
    }

	//-------------------------------------------------------------------------
    // Central method to handle every message sent to the higher-level DOC
    // processors.
	//-------------------------------------------------------------------------

    @Override
	public boolean enqueueMessage(final RT2Message msg) throws Exception {
               
        final RT2CliendUidType clientUID          = msg.getClientUID();                      

		// There are special situations where messages with seq-nrs must be processed although missing them.
		// E.g. CLOSE_DOC/LEAVE which are called by the GC which has no knowledge about seq-nrs
        Integer msgSeqNr  = null;
		if (msg.isSequenceMessage() && !RT2MessageGetSet.getInternalHeader(msg, RT2Protocol.HEADER_INTERNAL_FORCE, false))
			msgSeqNr = RT2MessageGetSet.getSeqNumber(msg);

		ackProcessor.updateClientState(msg);
				
		synchronized (clientsSeqQueues) {
			final ClientSequenceQueue clientQueue = getClientSequenceQueueForClient(clientUID, false);

			if (clientQueue == null) {
				return implCheckStateIfClientUnknown(clientUID, msg);
			}
			boolean checkForNeededNACK = false;
			final Integer waitForSeqNr = clientQueue.getInWaitSeqNumber();			
			if (msgSeqNr == null) {
				handleNoneSequenceMessage(msg);
			} else if (msgSeqNr.equals(waitForSeqNr)) {
				checkForNeededNACK = handleExpectedSequenceMessage(msg, clientQueue, waitForSeqNr);
			} else {
				checkForNeededNACK = handleUnexpectedSequenceMessage(msg, clientQueue, waitForSeqNr);
			}
			// request missing messages via NACK
			if (checkForNeededNACK)
				checkForAndRequestMissingMessagesViaNack(clientUID);
		}			
		return true;
	}

	//-------------------------------------------------------------------------    
	boolean handleUnexpectedSequenceMessage(final RT2Message aRT2Message, final ClientSequenceQueue clientQueue, final Integer waitForSeqNr) throws Exception {
		final RT2CliendUidType clientUID = aRT2Message.getClientUID();
		Integer msgSeqNr = aRT2Message.getSeqNumber().getValue();
		boolean seqNrAlreadySeen = clientQueue.wasMsgAlreadySeen(aRT2Message);
		if (!seqNrAlreadySeen) {
			clientQueue.enqueueInMessage(aRT2Message);
			ackProcessor.addReceivedSeqNr(clientUID, msgSeqNr, msgSeqNr);

			log.trace("RT2: Message queued as SEQ-NR: {} not in order (wait for {})" + ", for client {}", msgSeqNr, waitForSeqNr, clientUID);
			return true;
		}
		log.trace("RT2: Message already seen with SEQ-NR: {}, for client {}", msgSeqNr, clientUID);
		return false;
	}

	//-------------------------------------------------------------------------	
	boolean handleExpectedSequenceMessage(final RT2Message aRT2Message, final ClientSequenceQueue clientQueue, final Integer waitForSeqNr) throws Exception {
		final RT2CliendUidType clientUID = aRT2Message.getClientUID();
		Integer msgSeqNr = aRT2Message.getSeqNumber().getValue();		
		boolean checkForNeededNACK;
		clientQueue.enqueueInMessage(aRT2Message);				

		List<RT2Message> canBeProcessed = clientQueue.determinePendingMessagesWhichCanBeProcessed();
		int nextWaitForSeqNr = waitForSeqNr + canBeProcessed.size();

		// store received seq number range for an async ACK
		ackProcessor.addReceivedSeqNr(clientUID, msgSeqNr, Math.max(msgSeqNr, nextWaitForSeqNr - 1));				
		
		// add messages to queue for further processing
		queueProcessorDisposer.addMessages(this, canBeProcessed);
		clientQueue.removedNacksOfReceivedMessages(canBeProcessed);
		
		checkForNeededNACK = clientQueue.hasQueuedInMessages();
		return checkForNeededNACK;
	}

	//-------------------------------------------------------------------------	
	void handleNoneSequenceMessage(final RT2Message aRT2Message) throws Exception, JSONException {
		final RT2CliendUidType clientUID = aRT2Message.getClientUID();
		final ClientSequenceQueue clientQueue = getClientSequenceQueueForClient(clientUID, true);
		boolean bQueueMessage = true;
		switch (aRT2Message.getType()) {
		    case ACK_SIMPLE: {
		        // remove messages from backup map
		        ackProcessor.removeBackupMessagesForACKs(aRT2Message);
		        bQueueMessage = false;
		    } break;
		    case NACK_SIMPLE: {
		        // a simple NACK must be directly processed
		        final JSONArray aNackJSONArray = aRT2Message.getBody().getJSONArray(RT2Protocol.BODYPART_NACKS);
		        if (aNackJSONArray.length() != 0) {
			        List<RT2Message> lResendMessages = ackProcessor.getMessagesForResent(clientUID, aNackJSONArray.asList());
			        implResendMessagesRequestedByNack(clientUID, lResendMessages);
		        }
		        bQueueMessage = false;
		    } break;
		    case REQUEST_ABORT_OPEN: {
		    	log.debug("RT2: REQUEST_ABORT_OPEN received for docUid {} and clientUid {}", getDocUID(), clientUID);
		    	clientQueue.setAborted(true);
		        bQueueMessage = false;

		        final RT2Message aAbortOpenResponse = RT2MessageFactory.createResponseFromMessage(aRT2Message, RT2MessageType.RESPONSE_ABORT_OPEN);
		        sendResponseToClient(clientUID, aAbortOpenResponse);
		    } break;
		    case REQUEST_UNAVAILABILITY: {  // Really queue this message???
				log.debug("RT2: REQUEST_UNAVAILABILITY received for docUid {} and clientUid {}", getDocUID(), clientUID);
				ackProcessor.switchState(clientUID, RT2MessageGetSet.getUnvailableTime(aRT2Message) == 0);
		    } break;
		    default: break;
		}
		if (bQueueMessage) {
			queueProcessorDisposer.addMessages(this, Arrays.asList(aRT2Message));
		}
	}

    //-------------------------------------------------------------------------
    private boolean implCheckStateIfClientUnknown(final RT2CliendUidType sClientUID, final RT2Message msgFromUnknownClient) throws RT2ClientUidUnknownException {
    	final RT2MessageType msgType = msgFromUnknownClient.getType();
    	
		switch (msgType) {
			case ACK_SIMPLE:
			case NACK_SIMPLE: {
				// We ignore low-level msgs if the client uid is not known - this
				// could be a race of msgs where a client is about to leave and we
				// already received the last msg, but the leave-process is still in
				// progress - therefore the RT2DocProxy can send msgs to us!
				return true;
			}
			case REQUEST_CLOSE_DOC:
			case REQUEST_LEAVE: {
				long timeLeft = clientLeftRemoveThread.getClientLeft(sClientUID);
				if (timeLeft >= 0) {
					// We handle messages from clients that try to close/leave although they
					// already left the document instance more gracefully. This can happen if
					// we close/leave a client due to previous errors which the client hasn't
					// received yet.
					final Date date = new Date(timeLeft);
					log.debug("RT2: received message {} with client uid {} that left already at {} - just ignore message", msgFromUnknownClient, sClientUID, SimpleDateFormat.getDateInstance().format(date));
					return true;
				}
				break;
			}
			default: { /* nothing to do - throws exception */ } break;
			}

		log.warn("RT2: received message {} with unknown client uid {}", msgFromUnknownClient, sClientUID);
		throw new RT2ClientUidUnknownException(getMsgsAsList());
    }

    //-------------------------------------------------------------------------
    private void implResendMessagesRequestedByNack(final RT2CliendUidType sClientUID, final List<RT2Message> lResendMessages) throws Exception {
        for (final RT2Message aResendMsg : lResendMessages)
            sendMessageWOSeqNrTo(sClientUID, aResendMsg);
    }

    //-------------------------------------------------------------------------
	private boolean implSetSeqNrOnResponse (final RT2CliendUidType sTo, final RT2Message rMsg) {
		ClientSequenceQueue aClientSeqQueue;
		try {
			aClientSeqQueue = getClientSequenceQueueForClient(sTo, true);
		} catch (RT2TypedException e) {
			return false;
		}
        int nSeq = aClientSeqQueue.incAndGetOutSeqNumber();
        RT2MessageGetSet.setSeqNumber(rMsg, nSeq);
        log.trace("RT2: set SEQ-NR " + nSeq + " for message " + rMsg.getMessageID() + ", for client " + rMsg.getClientUID());
        return true;
	}

    //-------------------------------------------------------------------------
    public void checkForAndRequestMissingMessagesViaNack(final RT2CliendUidType clientUID)	throws Exception {
        ClientSequenceQueue clientSeqQueue = getClientSequenceQueueForClient(clientUID, true);
        Set<Integer> nacks = clientSeqQueue.checkForAndGenerateNacks();
		if (!nacks.isEmpty()) {			
			final RT2Message aNackRequest = RT2MessageFactory.newSimpleNackRequest(clientUID, getDocUID(), nacks);
			
			log.debug("RT2: Sending NACK for client {} with SEQ-NRS: {}", clientUID, nacks.toString());
			sendMessageWOSeqNrTo(clientUID, aNackRequest);
		}
    }

    //-------------------------------------------------------------------------    
	public ClientSequenceQueue getClientSequenceQueueForClient(final RT2CliendUidType clientUID, boolean throwEx) throws RT2TypedException {
		final ClientSequenceQueue aClientSeqQueue = clientsSeqQueues.get(clientUID);
        if (throwEx && (aClientSeqQueue == null)) {
            throw new RT2TypedException(ErrorCode.GENERAL_CLIENT_UID_UNKNOWN_ERROR, getMsgsAsList());
        }
		return aClientSeqQueue;
	}

    //-------------------------------------------------------------------------
	public void hangUpBadClient(RT2CliendUidType clientUid, ErrorCode errorCode) {
		final RT2Message hangupMsg = RT2MessageFactory.newBroadcastMessage(clientUid, getDocUID(), RT2MessageType.BROADCAST_HANGUP);
		sendMessageWOSeqNrTo(clientUid, hangupMsg);
		// a bad client should be immediately removed as member
		clientRemoved(clientUid);
	}

	//-------------------------------------------------------------------------
	@Override
	public void sendMessageTo(final RT2CliendUidType sTo, final RT2Message rMsg) {
	    if (implSetSeqNrOnResponse (sTo, rMsg)) {
	    	ackProcessor.backupMessage(rMsg);
	    	sendResponseToClient    (sTo, rMsg);			
	    }
	}

    //-------------------------------------------------------------------------
    @Override
    public void sendMessageWOSeqNrTo(final RT2CliendUidType sTo, final RT2Message rMsg)  {
		// a prio message is sent without a seq-nr
    	log.trace("Send priority message {} with msg-id: {}", rMsg.getType(), rMsg.getMessageID());
        sendResponseToClient(sTo, rMsg);
    }

	//-------------------------------------------------------------------------
	@Override
	public void broadcastMessageExceptClient(final RT2CliendUidType sExceptClientID, final RT2Message rMsg, final RT2MessageType sMsgType) {
	    final Collection<RT2CliendUidType> aReceiverUIDs = implDetermineReceivers(sExceptClientID);
	    if (!aReceiverUIDs.isEmpty()) {
    		implBroadcastMessage(aReceiverUIDs, rMsg, sMsgType);
	    }
	}

	//-------------------------------------------------------------------------
	@Override
	public void broadcastMessageTo(final Collection<RT2CliendUidType> aReceiverUIDs, final RT2Message rMsg, final RT2MessageType sMsgType) {
		implBroadcastMessage(aReceiverUIDs, rMsg, sMsgType);
	}


	//-------------------------------------------------------------------------
	private Collection<RT2CliendUidType> implDetermineReceivers(final RT2CliendUidType sExceptClientID)	{
	    final Set<RT2CliendUidType> aReceiversSet;
		synchronized(clientsSeqQueues) {
			final Set<RT2CliendUidType> aClientSet = clientsSeqQueues.keySet();
			aReceiversSet = aClientSet.stream()
			                          .filter(c -> !c.equals(sExceptClientID))
			                          .collect(Collectors.toSet());
		}
		return aReceiversSet;
	}

	//-------------------------------------------------------------------------
	private void implBroadcastMessage(final Collection<RT2CliendUidType> aReceiverUIDs, final RT2Message rMsg, final RT2MessageType sMsgType) {
		if (aReceiverUIDs != null) {
    		final List<RT2Message> aBroadcastList = new LinkedList<>();

    		synchronized(clientsSeqQueues) {
    			for (final RT2CliendUidType sClientUID : aReceiverUIDs) {
    				final ClientSequenceQueue aClientSeqQueue = clientsSeqQueues.get(sClientUID);
    				if (null != aClientSeqQueue) {

    					final RT2Message msg = RT2MessageFactory.cloneMessage(rMsg, sMsgType);
    					msg.setClientUID(sClientUID);
    					msg.setDocUID(getDocUID());

    					if (msg.hasFlags(RT2MessageFlags.FLAG_SEQUENCE_NR_BASED)) {
    						final int nOutSeq = aClientSeqQueue.incAndGetOutSeqNumber();
    						RT2MessageGetSet.setSeqNumber(msg, nOutSeq);    						
    					}
    					aBroadcastList.add(msg);
    				}
    			}
    		}
    		BroadcastMessage.Builder broadcastMsgBuilder = BroadcastMessage.newBuilder().setBody(BodyType.newBuilder().setValue(rMsg.getBodyString()).build())
    									 .setDocUid(DocUidType.newBuilder().setValue(getDocUID().getValue()))
    									 .setMessageId(MessageIdType.newBuilder().setValue(rMsg.getMessageID().getValue()))
    									 .setMsgType(getMessageTypeOfRT2Message(sMsgType));
    		
    		for (final RT2Message aMsg : aBroadcastList) {
    			broadcastMsgBuilder.addReceivers(
    					BroadcastMessageReceiver.newBuilder()
    						.setSeqNr(SequenceNrType.newBuilder().setValue(aMsg.hasSeqNumber() ? aMsg.getSeqNumber().getValue() : -1))
    						.setReceiver(ClientUidType.newBuilder().setValue(aMsg.getClientUID().getValue())));
    			try {
    				if (aMsg.hasFlags(RT2MessageFlags.FLAG_SEQUENCE_NR_BASED))
    				    ackProcessor.backupMessage(aMsg);
    				if (aBroadcastList.size() == 1) {
    					sendResponseToClient(aMsg.getClientUID(), aMsg);
    				}
    			}
    			catch (Throwable t) {
    				ExceptionUtils.handleThrowable(t);
    				log.error("RT2: Throwable caught while sending broadcast to client " + aMsg.getClientUID() + ", state unknown!");
    			}
    		}    		
    		if (aBroadcastList.size() > 1) {
	    		BroadcastMessage broadcastMessage = broadcastMsgBuilder.build();
	    		sendBroadcastMessageToClients(broadcastMessage);
    		}
		}
	}

	private MessageType getMessageTypeOfRT2Message(RT2MessageType msgType) {
		switch (msgType) {
			case BROADCAST_CRASHED: return MessageType.BROADCAST_CRASHED;
			case BROADCAST_EDITREQUEST_STATE: return MessageType.BROADCAST_EDITREQUEST_STATE;
			case BROADCAST_HANGUP: return MessageType.BROADCAST_HANGUP;
			case BROADCAST_RENAMED_RELOAD: return MessageType.BROADCAST_RENAMED_RELOAD;
			case BROADCAST_SHUTDOWN: return MessageType.BROADCAST_SHUTDOWN;
			case BROADCAST_UPDATE: return MessageType.BROADCAST_UPDATE;
			case BROADCAST_UPDATE_CLIENTS: return MessageType.BROADCAST_UPDATE_CLIENTS;
			default: throw new RuntimeException("Not a broadcast message type: " + msgType);
		}
	}
}
