/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.rt2.core.sequence;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.openexchange.office.rt2.protocol.RT2Message;
import com.openexchange.office.rt2.protocol.RT2MessageGetSet;
import com.openexchange.office.rt2.protocol.value.RT2CliendUidType;

public class ClientState {
	public static final long NO_TIMESTAMP = -1;
	private static final Logger log = LoggerFactory.getLogger(ClientState.class);
	
	private RT2CliendUidType clientUID;
	private boolean online = true;
	private Map<Integer, RT2Message> msgBackup = new HashMap<>();
	private Set<Integer> ackSet = new HashSet<>();
	private long lastMsgReceived;
	
	//-------------------------------------------------------------------------
	public ClientState(final RT2CliendUidType clientUID, boolean online) {
		this.clientUID = clientUID;
		this.online = online;
		this.lastMsgReceived = NO_TIMESTAMP;
	}

	//-------------------------------------------------------------------------
	public boolean isOnline() {
		return online;
	}

	//-------------------------------------------------------------------------
	public void setOnlineState(boolean bNewState) {
		online = bNewState;
	}

	//-------------------------------------------------------------------------
	public void addReceivedSeqNrCol(Collection<Integer> seqNrCol) {
		ackSet.addAll(seqNrCol);
	}

	//-------------------------------------------------------------------------	
	public Set<RT2Message> getBackupMsgs() {
		return new HashSet<>(msgBackup.values());
	}
	
	//-------------------------------------------------------------------------
	public int getMsgBackupSize() {
		return msgBackup.size();
	}
	
	//-------------------------------------------------------------------------
	public Set<Integer> getAckSet()
	{
		final Set<Integer> aNextAckSet = ackSet;
		ackSet = new HashSet<>();
		return aNextAckSet;
	}

	//-------------------------------------------------------------------------
	public long getLastMsgReceived() {
		return lastMsgReceived;
	}
	
	//-------------------------------------------------------------------------
	public void updateLastMsgReceived() {
		this.lastMsgReceived = System.currentTimeMillis();
	}
	
	//-------------------------------------------------------------------------
	public void backupMessage(final RT2Message rMsg) {
	    final Integer nSeqNr = RT2MessageGetSet.getSeqNumber(rMsg);
		final RT2Message rOld = msgBackup.put(nSeqNr, rMsg);

		if (rOld != null)
			log.error("RT2: Message with seq-nr " + nSeqNr + " already stored!");
	}

	//-------------------------------------------------------------------------
	public void removeStoredMessages(List<Object> ackList) {
		msgBackup.keySet().removeAll(ackList);
	}

	//-------------------------------------------------------------------------
	public void removeStoredMessages(int start, int end) {
		msgBackup.keySet().removeIf(k -> (k >= start && k <= end));
	}

	//-------------------------------------------------------------------------
	public List<RT2Message> getRequestedMessages(List<Object> aNackList) {
		List<RT2Message> lResult = new ArrayList<>();
		for (Object i : aNackList) {
			final RT2Message aMsg = msgBackup.get(i);
			if (aMsg != null)
				lResult.add(aMsg);
			else {
				lResult.clear();
				log.error("RT2: Couldn't find requested message with seq-nr {} for client UID {}. Existing entries: {}", i, clientUID, msgBackup.keySet());
				break;
			}
		}

		return lResult;
	}
}
