/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.rt2.core.osgi;

import java.util.ArrayList;
import java.util.List;

import org.apache.commons.lang3.Validate;
import org.osgi.framework.Bundle;
import org.osgi.framework.BundleContext;
import org.osgi.framework.FrameworkUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.openexchange.exception.ExceptionUtils;

public class BundleHelper
{
	private static final Logger log = LoggerFactory.getLogger(BundleHelper.class);
	
	private BundleHelper() {}

    //-------------------------------------------------------------------------
	public static boolean stopBundle(String sBundleSymbolicName) throws Exception
	{
		Validate.notNull(sBundleSymbolicName);

		final Bundle aBundle = findBundle(sBundleSymbolicName);
		boolean bCanShutdown = (aBundle != null);

		if (bCanShutdown)
			stopBundleAsync(aBundle);

        return bCanShutdown;
	}

    //-------------------------------------------------------------------------
	public static void stopBundles(final List<String> aBundleNames) throws Exception
	{
		Validate.notNull(aBundleNames);

		final List<Bundle> aBundleList = new ArrayList<>();
		for (final String sBundleSymbolicName : aBundleNames)
		{
			final Bundle aBundle = findBundle(sBundleSymbolicName);
			if (null != aBundle)
				aBundleList.add(aBundle);
		}

		aBundleList.forEach(bundle -> {
			try
			{
				stopBundleAsync(bundle);
			}
			catch (Throwable t)
			{
                ExceptionUtils.handleThrowable(t);
                log.error("RT2: Exception caught trying to stop bundle " + bundle.getSymbolicName(), t);
			}
		});

	}

    //-------------------------------------------------------------------------
	public static void stopBundleAsync(final Bundle aBundle) throws Exception
	{
		Validate.notNull(aBundle);

        final Runnable aAsyncShutdown = new Thread() {
            @Override
            public void run ()
            {
                try
                {
                    aBundle.stop();
                }
                catch (Throwable ex)
                {
                    ExceptionUtils.handleThrowable(ex);
                    log.info("RT2: Exception caught trying to stop bundle " + aBundle.getSymbolicName(), ex);
                }
            }
        };
        aAsyncShutdown.run();
	}

    //-------------------------------------------------------------------------
	public static Bundle findBundle(String sBundleSymbolicName) throws Exception
	{
		Bundle aBundleToFind = null;

		final BundleContext aBundleContext = FrameworkUtil.getBundle(BundleHelper.class).getBundleContext();
		final Bundle[]      aBundles       = aBundleContext.getBundles();

		if (null != aBundles)
		{
			for (final Bundle aBundle : aBundles)
			{
				if (aBundle.getSymbolicName().equals(sBundleSymbolicName))
				{
					aBundleToFind = aBundle;
					break;
				}
			}
		}

		return aBundleToFind;
	}
}
