/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.rt2.core.metric;

import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicLong;

import org.apache.commons.lang3.tuple.Pair;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.codahale.metrics.MetricRegistry;
import com.codahale.metrics.Timer;
import com.openexchange.exception.ExceptionUtils;
import com.openexchange.office.rt2.protocol.value.RT2MessageIdType;
import com.openexchange.office.tools.osgi.ServiceLookupRegistry;

public abstract class BaseMetricService {
	private static final long TIME_CLEANUP_FREQUENCY_IN_MS   = 1000 * 60 * 10;
	private static final long TIMEOUT_FOR_CLEANUP_IN_MINUTES = 3;
	private static final Logger log = LoggerFactory.getLogger(BaseMetricService.class);

	protected final ConcurrentHashMap<RT2MessageIdType, Pair<Timer.Context, LocalDateTime>> metricTimerContextMap;
	private final Map<LocalDateTime, Set<RT2MessageIdType>> timeMap = new ConcurrentHashMap<>();
	protected final Timer metricTimer; 
	private final Thread cleanupThread;

	public BaseMetricService(String metricName) {
		MetricRegistry metricRegistry = ServiceLookupRegistry.get().getService(MetricRegistry.class);
		this.metricTimerContextMap = new ConcurrentHashMap<>();
		this.metricTimer = metricRegistry.timer(metricName);
		this.cleanupThread = new Thread(new CleanupTask());
		cleanupThread.setName(metricName + ".CleanupThread");
		cleanupThread.setDaemon(true);
		cleanupThread.start();
	}

	public void startTimer(RT2MessageIdType msgId) {
		LocalDateTime now = LocalDateTime.now();
		Set<RT2MessageIdType> msgIds = timeMap.get(now);
		if (msgIds == null) {
			msgIds = new HashSet<>();
			timeMap.put(now, msgIds);
		} 
		msgIds.add(msgId);
		metricTimerContextMap.put(msgId, Pair.of(metricTimer.time(), now));
	}

	public void stopTimer(RT2MessageIdType msgId) {
		Pair<Timer.Context, LocalDateTime> p = metricTimerContextMap.get(msgId);
		if (p != null) {
			p.getKey().stop();
			metricTimerContextMap.remove(msgId);
			Set<RT2MessageIdType> msgIds = timeMap.get(p.getValue());
			if ((msgIds != null) && (msgIds.size() > 1)) {
				msgIds.remove(msgId);
			} else {
				if ((msgIds != null) &&  msgIds.contains(msgId)) {
					timeMap.remove(p.getValue());
				}
			}
		}
	}

	private class CleanupTask implements Runnable {

		@Override
		public void run() {
			while (true) {
				try {
					Thread.sleep(TIME_CLEANUP_FREQUENCY_IN_MS);
					final LocalDateTime cmpDt = LocalDateTime.now().minusMinutes(TIMEOUT_FOR_CLEANUP_IN_MINUTES);
					final AtomicLong removed = new AtomicLong(0);

					synchronized (timeMap) {
						List<LocalDateTime> timeList = new ArrayList<LocalDateTime>(timeMap.keySet()); 
						Collections.sort(timeList);
						for (LocalDateTime dt : timeList) {
							if (dt.isBefore(cmpDt)) {
								Set<RT2MessageIdType> msgIdSet = timeMap.remove(dt);
								msgIdSet.forEach(msgId -> {
									if (null != metricTimerContextMap.remove(msgId))
										removed.incrementAndGet();
								});
							}
						}
					}

					synchronized (metricTimerContextMap) {
						// remove pending timer after  too
						metricTimerContextMap.keySet().forEach(k -> {
							final Pair<Timer.Context, LocalDateTime> entry = metricTimerContextMap.get(k);
							if ((entry != null) && (entry.getRight().isBefore(cmpDt))) {
								// stop pending timer after timeout
								entry.getLeft().stop();
								metricTimerContextMap.remove(k);
							}
						});
					}

					log.debug("{} removed {} entries from metrics map - time map has still {} entries & metricTimerContextMap has {} entries" , Thread.currentThread().getName(), removed.get(), timeMap.size(), metricTimerContextMap.size());
				} catch (Throwable ex) {
					ExceptionUtils.handleThrowable(ex);
					log.warn("CleanupTask: Exception caught while trying to remove old metric data", ex);
				}
			}
		}

	}
}
