package com.openexchange.office.rt2.core.management;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.management.MalformedObjectNameException;
import javax.management.ObjectName;

import org.apache.commons.lang3.tuple.Pair;

import com.openexchange.management.ManagementObject;
import com.openexchange.office.rt2.protocol.value.RT2CliendUidType;
import com.openexchange.office.rt2.protocol.value.RT2DocUidType;
import com.openexchange.office.rt2.ws.RT2ChannelId;
import com.openexchange.office.rt2.ws.RT2SessionCountValidator;
import com.openexchange.office.rt2.ws.RT2WSApp;
import com.openexchange.office.rt2.ws.RT2WSChannelDisposer;

public class RT2WebSocketManagement extends ManagementObject<RT2WebSocketMBean> implements RT2WebSocketMBean { 

	private final RT2WSChannelDisposer channelDisposer;
	
	private final RT2WSApp wsApp;
	
	private final RT2SessionCountValidator sessionCountValidator;
	
	private final ObjectName objectName;	
	
	public RT2WebSocketManagement(RT2WSApp wsApp, RT2WSChannelDisposer channelDisposer, RT2SessionCountValidator sessionCountValidator) throws MalformedObjectNameException {		
		super(RT2WebSocketMBean.class);
		this.channelDisposer = channelDisposer;
		this.wsApp = wsApp;
		this.sessionCountValidator = sessionCountValidator;
		this.objectName = initObjectName();
	}
	
	@Override
	public List<String> getAllActiveWSChannels() {
		Set<RT2ChannelId> channelIds = wsApp.getCurrentActiveWebsocketIds();
		List<String> res = new ArrayList<>();		
		channelIds.forEach(id -> res.add(id.getId()));
		return res;
	}

	@Override
	public List<String> getClientUidsOfChannel(String channelId) {
		Set<RT2CliendUidType> clientUids = channelDisposer.getClientsAssociatedToChannel(new RT2ChannelId(channelId));
		List<String> res = new ArrayList<>();
		clientUids.forEach(c -> res.add(c.getValue()));
		return res;		
	}

	@Override
	public Integer getReponsesQueueSize() {
		return channelDisposer.getResponseQueueSize();
	}

	@Override
	public List<String> getDetailInfoAboutAllActiveWSChannels() {
		List<String> res = new ArrayList<>();
		Map<RT2CliendUidType, Pair<RT2DocUidType, RT2ChannelId>> channels = channelDisposer.getChannels();
		channels.forEach((k, v) -> {
			res.add(v.getRight().getId() + "_" + v.getLeft().getValue() + "_" + k.getValue());
		});
		return res;
	}

	@Override
	public void terminateAllWsConnections() {
		wsApp.terminateAllWsConnections();
	}
	
	public Integer getDetailInfoAboutAllActiveWSChannelSize() {
		return channelDisposer.getChannels().size();
	}
	
	@Override
	public List<String> getAllOfflineWSChannels() {
		final List<String> res = new ArrayList<>();
		final Map<RT2ChannelId, Long> channelsOffline = channelDisposer.getChannelsOffline();
		final long now = System.currentTimeMillis();
		channelsOffline.forEach((k, v) -> {
			final long diff = (v == null) ? 0 :  ((now - v) / 1000);
			res.add(k + "=" + diff + "s");
		});
		return res;
	}
	
	@Override
	public Map<Integer, Integer> getCountConnectionsOfUsers() {
		return sessionCountValidator.getCountSessionsOfUsers();
	}

	@Override
	public ObjectName getObjectName() {
		return objectName;
	}
	
    private ObjectName initObjectName() throws MalformedObjectNameException {
        final String directoryName = "WebsocketMonitoring";
        return new ObjectName("com.openexchange.office.rt2", "name", directoryName);
    }

}
