package com.openexchange.office.rt2.core.management;

import java.lang.ref.WeakReference;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import javax.jms.JMSException;
import javax.management.MalformedObjectNameException;
import javax.management.ObjectName;

import org.apache.activemq.ActiveMQConnectionFactory;
import org.apache.activemq.jms.pool.PooledConnectionFactory;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.hazelcast.core.HazelcastInstance;
import com.hazelcast.core.IMap;
import com.openexchange.management.ManagementObject;
import com.openexchange.office.rt2.cache.ClusterLockService;
import com.openexchange.office.rt2.config.RT2ConfigItem;
import com.openexchange.office.rt2.core.RT2DocProcessorExistsTester;
import com.openexchange.office.rt2.core.RT2GarbageCollector;
import com.openexchange.office.rt2.core.RT2NodeInfoService;
import com.openexchange.office.rt2.core.doc.DocProcessorClientInfo;
import com.openexchange.office.rt2.core.doc.RT2DocProcessor;
import com.openexchange.office.rt2.core.doc.RT2DocProcessorManager;
import com.openexchange.office.rt2.core.sequence.QueueProcessorDisposer;
import com.openexchange.office.rt2.core.sequence.QueueProcessorDisposer.MessageQueuesDispatcherInfo;
import com.openexchange.office.rt2.core.sequence.SequenceDocProcessor;
import com.openexchange.office.rt2.jms.EnhActiveMQSSLConnectionFactory;
import com.openexchange.office.rt2.protocol.value.RT2CliendUidType;
import com.openexchange.office.rt2.protocol.value.RT2DocUidType;
import com.openexchange.office.rt2.proxy.RT2DocInfoRegistry;
import com.openexchange.office.rt2.proxy.RT2DocProxy;
import com.openexchange.office.rt2.proxy.RT2DocProxyRegistry;
import com.openexchange.office.tools.osgi.ServiceLookupRegistry;

public class RT2BackendManagement extends ManagementObject<RT2BackendMBean> implements RT2BackendMBean {

    //-------------------------------------------------------------------------
	private static final int MAX_QUEUE_SIZE_FOR_MSGS_FOR_DOCPROCESSOR = 1000;

    //-------------------------------------------------------------------------
    private static final Logger log = LoggerFactory.getLogger(RT2BackendManagement.class);
        
    private static DateTimeFormatter formatter = DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss");

    private final ObjectName objectName;

    private final RT2DocProcessorManager rt2DocProcessorManager;

    private final RT2DocProxyRegistry rt2DocRegistry;

    private final RT2DocInfoRegistry rt2DocInfoRegistry;
    
    private final PooledConnectionFactory pooledConnectionFactory;   
    
    private final RT2GarbageCollector garbageCollector;
    
    private final RT2NodeInfoService rt2NodeInfo;
    
    private final QueueProcessorDisposer queueProcessorDisposer;
    
    private final RT2DocProcessorExistsTester docProcExistsTester;
    
    private final HazelcastInstance hzInstance;
    
    private final ClusterLockService clusterLockService;
    
    public RT2BackendManagement(RT2DocProcessorManager rt2DocProcessorManager, RT2DocProxyRegistry rt2DocRegistry, RT2DocInfoRegistry rt2DocInfoRegistry,
    							PooledConnectionFactory pooledConnectionFactory, RT2GarbageCollector garbageCollector, RT2NodeInfoService rt2NodeInfo,
    							QueueProcessorDisposer queueProcessorDisposer, RT2DocProcessorExistsTester docProcExistsTester,
    							HazelcastInstance hzInstance, ClusterLockService clusterLockService) throws MalformedObjectNameException {
        super(RT2BackendMBean.class);
        this.rt2DocProcessorManager = rt2DocProcessorManager;
        this.rt2DocRegistry = rt2DocRegistry;
        this.rt2DocInfoRegistry = rt2DocInfoRegistry;
        this.pooledConnectionFactory = pooledConnectionFactory;
        this.garbageCollector = garbageCollector;
        this.rt2NodeInfo = rt2NodeInfo;
        this.queueProcessorDisposer = queueProcessorDisposer;
        this.docProcExistsTester = docProcExistsTester;
        this.hzInstance = hzInstance;
        this.clusterLockService = clusterLockService;
        this.objectName = initObjectName();        
    }
    
    @Override
	public String getHazelcastUid() {
    	return rt2NodeInfo.getNodeUUID();
	}

	@Override
    public Map<String, String> getRT2DocProxies() {
        Map<String, String> res = new HashMap<>();
        for (RT2DocProxy docProxy : rt2DocRegistry.listAllDocProxies()) {
            StringBuilder strBuilderValue = new StringBuilder();
            strBuilderValue.append("docUid: ");
            strBuilderValue.append(docProxy.getDocUID());
            strBuilderValue.append(", clientUid: ");
            strBuilderValue.append(docProxy.getClientUID());

            StringBuilder strBuilderKey = new StringBuilder();
            strBuilderKey.append("proxyId: ");
            strBuilderKey.append(docProxy.getProxyID());

            res.put(strBuilderKey.toString(), strBuilderValue.toString());
        }
        return res;
    }

    @Override
    public List<String> getRT2DocProxyIds() {
        List<String> res = new ArrayList<>();
        for (RT2DocProxy docProxy : rt2DocRegistry.listAllDocProxies()) {
            res.add(docProxy.getProxyID());
        }
        return res;
    }

    @Override
    public int getCountRT2DocProxies() {
        return getRT2DocProxies().size();
    }

    @Override
    public Map<String, String> getDocProcessors() {
        Map<String, String> res = new HashMap<>();
        for (WeakReference<RT2DocProcessor> ref : rt2DocProcessorManager.getWeakReferenceToDocProcessors()) {
            RT2DocProcessor docProc = ref.get();
            if (docProc != null) {
                StringBuilder strBuilderKey = new StringBuilder();
                strBuilderKey.append("docUid: ");
                strBuilderKey.append(docProc.getDocUID());

                StringBuilder strBuilderValue = new StringBuilder();
                for (DocProcessorClientInfo clientInfo : docProc.getClientsInfo()) {
                    strBuilderValue.append(", ");
                    strBuilderValue.append(clientInfo.toString());
                }
                res.put(strBuilderKey.toString(), strBuilderValue.toString());
            }
        }
        return res;
    }

    @Override
    public List<String> getDocProcessorIds() {
        List<String> res = new ArrayList<>();
        for (WeakReference<RT2DocProcessor> ref : rt2DocProcessorManager.getWeakReferenceToDocProcessors()) {
            RT2DocProcessor docProc = ref.get();
            if (docProc != null) {
                res.add(docProc.getDocUID().getValue());
            }
        }
        return res;
    }

    @Override
    public int getCountRT2DocProcessors() {
        return getDocProcessors().size();
    }

    private ObjectName initObjectName() throws MalformedObjectNameException {
        final String directoryName = "BackendMonitoring";
        return new ObjectName("com.openexchange.office.rt2", "name", directoryName);
    }

    @Override
    public ObjectName getObjectName() {
        return objectName;
    }

    @Override
    public void clearAll() {
        rt2DocRegistry.clear();
        rt2DocInfoRegistry.clear();
        rt2DocProcessorManager.getWeakReferenceToDocProcessors().forEach(r -> rt2DocProcessorManager.docProcessorDisposed(r.get()));
    }

    @Override
    public long getGCTimeout() {
        return RT2ConfigItem.get().getRT2GCOfflineTresholdInMS();
    }

    @Override
    public long getGCFrequency() {
        return RT2ConfigItem.get().getRT2GCFrequencyInMS();
    }

    @Override
    public void setGCTiming(Long timeout, Long frequency) {
        RT2ConfigItem.get().setRT2GCOfflineTresholdInMS(timeout);
        RT2ConfigItem.get().setRT2GCFrequencyInMS(frequency);

        final RT2GarbageCollector gcInstance = ServiceLookupRegistry.get().getService(RT2GarbageCollector.class);
        if (null != gcInstance) {
            try {
                gcInstance.stop();
                gcInstance.start();
            } catch (InterruptedException e) {
                Thread.currentThread().interrupt();
            } catch (Exception e) {
                log.error("Exception caught trying to change garbage collector timing", e);
            }
        }
    }

	@Override
	public String getDcsConnectionUrl() throws JMSException {
		return ((ActiveMQConnectionFactory) pooledConnectionFactory.getConnectionFactory()).getBrokerURL();
	}
	
	@Override
	public Map<String, Integer> getMessageCountWaitingForAckOfDoc() {
		Map<String, Integer> res = new HashMap<>();
		rt2DocProcessorManager.getWeakReferenceToDocProcessors().forEach(p -> {
			SequenceDocProcessor seqDocProc = (SequenceDocProcessor) p.get();
			if (seqDocProc != null) {
				seqDocProc.getACKProcessor().getClientStatesMsgBackupSize().forEach((k,v) -> res.put(seqDocProc.getDocUID().getValue() + "_" + k.toString(), v));				
			}
		});
		return res;
	}

	@Override
	public int getCountQueuedMessagesOfProcessor() {
		return queueProcessorDisposer.getMessageQueueSize();
	}

	@Override
	public int getCountCurrentUsedProcessors() {
		return queueProcessorDisposer.getCurrentUsedDocProcessors();
	}

	@Override
	public void triggerRT2GC() {
		garbageCollector.doGC();
	}

	@Override
	public Set<String> getDocProxyMarkedForRemove() {
		return docProcExistsTester.getMarkedForRemove().stream().map(p -> p.getValue()).collect(Collectors.toSet());
	}
	
	@Override
	public boolean removeDocProxy(String clientUid, String docUid) {
		String id = RT2DocProxy.calcID(new RT2CliendUidType(clientUid.trim()), new RT2DocUidType(docUid.trim()));
		RT2DocProxy docProxy = rt2DocRegistry.getDocProxy(id);
		if (docProxy != null) {
			rt2DocRegistry.deregisterDocProxy(docProxy, true);
			return true;
		}
		return false;
	}

	@Override
	public boolean removeDocProxy(String docProxyId) {
		RT2DocProxy docProxy = rt2DocRegistry.getDocProxy(docProxyId.trim());
		if (docProxy != null) {
			rt2DocRegistry.deregisterDocProxy(docProxy, true);
			return true;
		}
		return false;
	}

	@Override
	public boolean removeDocProcessor(String docUid) {
		RT2DocProcessor docProc = rt2DocProcessorManager.getDocProcessor(new RT2DocUidType(docUid.trim()));
		if (docProc != null) {
			rt2DocProcessorManager.docProcessorDisposed(docProc);
			return true;
		}
		return false;
	}

	@Override
	public List<String> getMsgsOfDocProxy(String clientUid, String docUid) {
		String id = RT2DocProxy.calcID(new RT2CliendUidType(clientUid.trim()), new RT2DocUidType(docUid.trim()));
		RT2DocProxy docProxy = rt2DocRegistry.getDocProxy(id);
		if (docProxy != null) {
			return docProxy.formatMsgsLogInfo();
		}
		return Arrays.asList("Not found!");
	}

	@Override
	public List<String> getMsgsOfDocProxy(String proxyId) {
		RT2DocProxy docProxy = rt2DocRegistry.getDocProxy(proxyId.trim());
		if (docProxy != null) {
			return docProxy.formatMsgsLogInfo();
		}
		return Arrays.asList("Not found!");		
	}
	
	@Override
	public List<String> getMsgsOfDocProcessor(String docUid) {
		RT2DocProcessor docProc = rt2DocProcessorManager.getDocProcessor(new RT2DocUidType(docUid.trim()));
		if (docProc != null) {
			return docProc.formatMsgsLogInfo();
		}
		return Arrays.asList("Not found!");
	}

	@Override
	public List<String> getMsgsOfDocProcessor(String clientUid, String docUid) {
		RT2DocProcessor docProc = rt2DocProcessorManager.getDocProcessor(new RT2DocUidType(docUid.trim()));
		if (docProc != null) {
			return docProc.formatMsgsLogInfoForClient(new RT2CliendUidType(clientUid.trim()));
		}
		return Arrays.asList("Not found!");
	}

	@Override
	public void setQueueSizeForMsgsOfDocProcessor(String docUid, int count) {
		RT2DocProcessor docProc = rt2DocProcessorManager.getDocProcessor(new RT2DocUidType(docUid.trim()));
		if (docProc != null) {
			count = Math.max(0, Math.min(MAX_QUEUE_SIZE_FOR_MSGS_FOR_DOCPROCESSOR, count));
			docProc.resetQueueSizeForLogInfo(count);
		}
	}

	@Override
	public Set<String> getCurrentJmsConnections() throws Exception {
		Set<String> res = new HashSet<>();		
		EnhActiveMQSSLConnectionFactory amqConnFact = (EnhActiveMQSSLConnectionFactory) pooledConnectionFactory.getConnectionFactory();
		amqConnFact.getConnections().forEach(c -> {
			try {
				res.add("ClientId: " + c.getClientID().toString() + ", BrokerId: " + c.getBrokerInfo().getBrokerId() + ", BrokerUrl: " + c.getBrokerInfo().getBrokerURL() + ", BrokerName: " + c.getBrokerInfo().getBrokerName());
			} catch (JMSException e) {
				log.info(e.getMessage());
			}
		});
		return res;
	}
	
	@Override
	public Map<String, String> getMessageProcessorRunningThreadTime() {
		Map<String, String> res = new HashMap<>();
		queueProcessorDisposer.getMessageQueuesDispatcherInfos().stream().forEach(m -> res.put(m.getDocUid().toString(), m.getRunningtime().toString()
	            .substring(2)
	            .replaceAll("(\\d[HMS])(?!$)", "$1 ")
	            .toLowerCase()));
		return res;
	}

	@Override
	public boolean terminateMessageProcessorThread(String docUid) {
		Set<MessageQueuesDispatcherInfo> messageQueuesDispatcherInfos = queueProcessorDisposer.getMessageQueuesDispatcherInfos();
		for (MessageQueuesDispatcherInfo info : messageQueuesDispatcherInfos) {
			if (info.getDocUid().getValue().equals(docUid)) {
				info.interrupt();
				return true;
			}
		}
		return false;
	}
	
	@Override
	public List<String> getStackTraceOfMessageProcessorThread(String docUid) {
		List<String> res = new ArrayList<>();
		Set<MessageQueuesDispatcherInfo> messageQueuesDispatcherInfos = queueProcessorDisposer.getMessageQueuesDispatcherInfos();
		for (MessageQueuesDispatcherInfo info : messageQueuesDispatcherInfos) {
			if (info.getDocUid().getValue().equals(docUid)) {
				StackTraceElement [] stackTraceArray = info.getStacktraceOfThread();
				for (StackTraceElement ele : stackTraceArray) {
					res.add(ele.toString());
				}
			}
		}
		return res;
	}

//	@Override
//    public List<String> getAllHazelcastRT2LockObjects() {
//		List<String> res = new ArrayList<>();
//		Collection<DistributedObject> distObjects = hzInstance.getDistributedObjects();
//		for (DistributedObject distObj : distObjects) {			
//		    if(distObj.getName().startsWith(ClusterLockService.LOCK_PREFIX)  && (distObj instanceof ILock)) {
//		    	ILock lockObj = (ILock) distObj;
//		    	res.add(lockObj.getName() + ":" + lockObj.getLockCount());
//		    }
//		}
//		return res;
//	}
	
	@Override
    public List<String> getRegisteredRT2LockObjects() {
    	return new ArrayList<>(clusterLockService.getLocks());
    }

	@Override
	public void unlockRT2LockObject(String docUid) {
		IMap<String, String> hzLockMap = hzInstance.getMap(ClusterLockService.LOCK_MAP_NAME);
		if (hzLockMap.isLocked(ClusterLockService.LOCK_PREFIX + docUid)) {
			hzLockMap.unlock(ClusterLockService.LOCK_PREFIX + docUid);
		}
	}

	@Override
	public void destroyRT2LockObject(String docUid) {
		unlockRT2LockObject(docUid);
		IMap<String, String> hzLockMap = hzInstance.getMap(ClusterLockService.LOCK_MAP_NAME);
		hzLockMap.remove(ClusterLockService.LOCK_PREFIX + docUid);
	}

	@Override
	public void runClusterLockUnlockThread() {
		clusterLockService.runClusterLockUnlockThread();
	}

	@Override
	public void runDocLockRemoveThread() {
		clusterLockService.runDocLockRemoveThread();
	}

	@Override
	public Map<String, Integer> getNotAckedMessageCount(String docUid) {
		Map<String, Integer> res = new HashMap<>();
		SequenceDocProcessor rt2DocProcessor = (SequenceDocProcessor) rt2DocProcessorManager.getDocProcessor(new RT2DocUidType(docUid));
		if (rt2DocProcessor != null) { 
			rt2DocProcessor.getACKProcessor().getClientStatesMsgBackupSize().entrySet().stream().forEach(p -> res.put(p.getKey().getValue(), p.getValue()));
		}
		return res;
	}

	@Override
	public Integer getNackFrequenceOfServer() {
		return RT2ConfigItem.get().getNackFrequenceOfServer();
	}

	@Override
	public void updateNackFrequenceOfServer(Integer value) {
		RT2ConfigItem.get().updateNackFrequenceOfServer(value);
	}

	@Override
	public List<String> getAtomicLongsToGc() {
		Map<RT2DocUidType, LocalDateTime> atomicLongsToGc = garbageCollector.getAtomicLongsToGc();
		final List<String> res = new ArrayList<>();
		atomicLongsToGc.forEach((k, v) -> {
			res.add("DocUid: " + k.getValue() + ", date: " + formatter.format(v));
		});
		return res;		
	}

	@Override
	public List<String> getAtomicLongsToBeVerified() {
		Set<RT2DocUidType> atomicLongsToBeVerified = garbageCollector.getAtomicLongsToBeVerified();
		final List<String> res = new ArrayList<>();
		atomicLongsToBeVerified.forEach(t -> {
			res.add(t.getValue());
		});
		return res;
	}
	
//    public Map<RT2DocUidType, LocalDateTime> getAtomicLongsToGc() {
//		return new HashMap<>(atomicLongsToGc);
//	}
//
//	//-------------------------------------------------------------------------	
//	public Set<RT2DocUidType> getAtomicLongsToBeVerified() {
//		return new HashSet<>(atomicLongsToBeVerified);
//	}
	
}
