/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.rt2.core.doc;

import java.lang.ref.WeakReference;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.collect.EvictingQueue;
import com.openexchange.office.documents.Document;
import com.openexchange.office.rt2.cache.RT2DocInfo;
import com.openexchange.office.rt2.core.RT2LogInfo;
import com.openexchange.office.rt2.core.doc.RT2DocProcessorLogInfo.Direction;
import com.openexchange.office.rt2.core.sequence.IInSequenceProxy;
import com.openexchange.office.rt2.core.sequence.IOutSequenceProxy;
import com.openexchange.office.rt2.exception.RT2TypedException;
import com.openexchange.office.rt2.logging.IMessagesLoggable;
import com.openexchange.office.rt2.protocol.RT2Message;
import com.openexchange.office.rt2.protocol.RT2MessageFactory;
import com.openexchange.office.rt2.protocol.RT2MessageGetSet;
import com.openexchange.office.rt2.protocol.value.RT2CliendUidType;
import com.openexchange.office.rt2.protocol.value.RT2DocUidType;
import com.openexchange.office.rt2.protocol.value.RT2MessageType;
import com.openexchange.office.rt2.protocol.value.RT2RecipientListType;
import com.openexchange.office.rt2.protocol.value.RT2SessionIdType;
import com.openexchange.office.rt2.proxy.RT2DocInfoRegistry;
import com.openexchange.office.tools.error.ErrorCode;
import com.openexchange.office.tools.osgi.ServiceLookupRegistryService;
import com.openexchange.office.tools.user.UserData;

//=============================================================================
/** implements the back bone of a RT2 document ...
 *  This instance exists one times for every document within the cluster.
 *  It knows all states about those document and control it's life cycle.
 */
public abstract class RT2DocProcessor extends AsyncMultiExchangeProcessor implements IInSequenceProxy, IOutSequenceProxy, Document, IMessagesLoggable
{
	private static final int DEFAULT_COUNT_MESSAGES_STORED = 20;

	private static final Logger log = LoggerFactory.getLogger(RT2DocProcessor.class);
	
    private RT2DocUidType m_sDocUID = null;

    private WeakReference< RT2DocInfo > m_rDocInfo = null;

    private WeakReference< RT2DocInfoRegistry > m_rDocRegistry = null;
    
    private EvictingQueue<RT2DocProcessorLogInfo> msgs = EvictingQueue.create(DEFAULT_COUNT_MESSAGES_STORED);

    private Object sync = new Object();

    public RT2DocProcessor() {    	
        this.m_rDocRegistry = new WeakReference<RT2DocInfoRegistry>(ServiceLookupRegistryService.getInstance().getService(RT2DocInfoRegistry.class));
    }

    //-------------------------------------------------------------------------
    public abstract void dispose();

    //-------------------------------------------------------------------------
	public abstract String getDocTypeIdentifier();

    //-------------------------------------------------------------------------
	public abstract List<DocProcessorClientInfo> getClientsInfo();
	
	//-------------------------------------------------------------------------
	public abstract void updateClientsInfo(Collection<RT2CliendUidType> existingClients);
	
	//-------------------------------------------------------------------------
	
	public abstract List<UserData> getUserDatasCloned();
	
	//-------------------------------------------------------------------------	
	public abstract int getPendingOperationsCount();

	//-------------------------------------------------------------------------
	public synchronized void setDocUID (final RT2DocUidType docUID)
	{
		m_sDocUID = docUID;
		synchronized (sync) {
			msgs.add(new RT2DocProcessorLogInfo(Direction.NEW_INSTANCE, "RT2DocProcessor::constructor", null, docUID));
		}
        final RT2DocStateNotifier aDocStateNotifier = ServiceLookupRegistryService.getInstance().getService(RT2DocStateNotifier.class);
        aDocStateNotifier.notifyEvent(RT2DocStateNotifier.DOCSTATE_EVENT_CREATED, this);
	}

	//-------------------------------------------------------------------------	
	public Map<RT2SessionIdType, RT2CliendUidType> getSessionIdToClientUids() {
		Map<RT2SessionIdType, RT2CliendUidType> res = new HashMap<>();
		for (UserData userData : getUserDatasCloned()) {
			res.put(new RT2SessionIdType(userData.getSessionId()), new RT2CliendUidType(userData.getClientUid()));
		}
		return res;
	}
		
    //-------------------------------------------------------------------------
    public void deregisterClient (final RT2CliendUidType clientUID)
    {
        log.debug("deregister client '"+clientUID+"' ...");
        clientRemoved(clientUID);
    }

    //-------------------------------------------------------------------------
    protected void addMessageForQueueToLog(final RT2Message msg) {
    	if (msg == null)
    		return;

    	synchronized(sync) {
   			msgs.add(new RT2DocProcessorLogInfo(Direction.QUEUE, msg));
    	}
    }

    //-------------------------------------------------------------------------	
	public void resetQueueSizeForLogInfo(int count) {
		synchronized (sync) {
			final EvictingQueue<RT2DocProcessorLogInfo> newQueue = EvictingQueue.create(count);
			newQueue.addAll(msgs);
			msgs = newQueue;
		}
	}

    //-------------------------------------------------------------------------
    @Override
    public void process(final RT2Message request) throws Exception {
    	synchronized (sync) {
    		msgs.add(new RT2DocProcessorLogInfo(Direction.FROM_JMS, request));
		}    	
        if (RT2MessageType.REQUEST_JOIN.equals(request.getType())) {
            log.debug("register client '" + request.getClientUID() + "' ...");
            clientAdded(request.getClientUID());
        }
        //no else !
        enqueueMessage(request);
    }

    //-------------------------------------------------------------------------
    /** send out a normal response ...
     *
     *  @param  response [IN]
     *          the response to be routed back to client.
     */
    public void sendResponseToClient (final RT2CliendUidType clientUID, final RT2Message response)
    {
        log.debug("Send response ''{}'' to client with id ''{}'' ", response.getHeader() , clientUID);
        
        response.setDocUID(getDocUID());

        synchronized (sync) {
        	msgs.add(new RT2DocProcessorLogInfo(Direction.TO_JMS, response));
        }
        processOutMessage (response);

        // as leave response e.g. deregister our recipient channel ...
        // we have to call this methods after processOutMessage !
        switch (response.getType()) {
        	case RESPONSE_EMERGENCY_LEAVE: 
            case RESPONSE_LEAVE:
                deregisterClient(clientUID);
                break;
            default:
        }
    }

    //-------------------------------------------------------------------------
	/** send out a normal response ...
	 *  ... use recipient list of given message to address it.
	 *
	 *	@param	aResponse [IN]
	 *			the response to be routed back to client.
	 */
	public /* no synchronized */ void sendResponseToRecipients (final RT2Message aResponse)
		throws Exception
	{
	    log.debug("send reponse ''{}'' to recipients", aResponse.getHeader());

		final RT2RecipientListType lRecipients = aResponse.getRecipients();
		for (final RT2CliendUidType sRecipient : lRecipients.getValue()) {
	        sendResponseToClient (sRecipient, aResponse);
		}
	}

    //-------------------------------------------------------------------------
    /** send out an error response ...
     *
     *  If no error code nor an exception is given an UNKNOEN_ERROR is send.
     *
     *  @param  sClientUID [IN, OPTIONAL]
     *          the client UID where this message has to be send to
     *          If it's not given - it's retrieved from the original request.
     *          But at least any client UID must be given - otherwise error message
     *          can't be send.
     *
     *  @param  aOrgRequest [IN, OPTIONAL]
     *          the original request where this error occurred and
     *          where the error response has to be send for
     *          If no such request is given a new error response
     *          is generated from scratch.
     *
     *  @param  aError [IN, OPTIONAL]
     *          the error code to be send
     *
     *  @param  aException [IN, OPTIONAL]
     *          the exception to be analyzed
     */
    public /* no synchronized */ void sendErrorResponseToClient (final RT2CliendUidType     sClientUID ,
                                                                 final RT2Message aOrgRequest,
                                                                 final ErrorCode  aError     ,
                                                                 final Throwable  aException )
    {
        final RT2Message aErrorResponse = RT2MessageFactory.createResponseFromMessage(aOrgRequest, RT2MessageType.RESPONSE_GENERIC_ERROR);
              RT2CliendUidType sSendTo = sClientUID;
              ErrorCode aSendError = null;

        if (aOrgRequest != null) {
            sSendTo = aOrgRequest.getClientUID();
        }

        if (aError != null) {
            aSendError = aError;
        } else {
            if (aException != null) {
                if (RT2TypedException.class.isAssignableFrom(aException.getClass ())) {
                    final RT2TypedException aRTEx = (RT2TypedException) aException;
                    aSendError = aRTEx.getError();
                } else {
                    aSendError = ErrorCode.GENERAL_UNKNOWN_ERROR;
                }
            } else {
                aSendError = ErrorCode.GENERAL_UNKNOWN_ERROR;
            }
        }

        log.debug("send error response ''{}'' to client ''{}''", aSendError, sSendTo);

        // set error and clear seq-nr - an error response cannot use the cloned one
        RT2MessageGetSet.setError(aErrorResponse, aSendError);
        RT2MessageGetSet.clearSeqNumber(aErrorResponse);

        sendMessageWOSeqNrTo(sSendTo, aErrorResponse);
    }

    //-------------------------------------------------------------------------
    @Override
	public RT2DocUidType getDocUID ()
	{
		return m_sDocUID;
	}

    //-------------------------------------------------------------------------
	/**	@return access to the cache info object bound to this document
	 */
	protected synchronized RT2DocInfo mem_DocInfo ()
	    throws Exception
	{
		RT2DocInfo aInfo = null;

		if (m_rDocInfo != null)
			aInfo = m_rDocInfo.get ();

		if (aInfo == null) {
		    aInfo = m_rDocRegistry.get().getDocInfo(m_sDocUID);
		    m_rDocInfo = new WeakReference<> (aInfo);
		}
		return aInfo;
	}

    //-------------------------------------------------------------------------
    public List<String> formatMsgsLogInfo() {
    	synchronized (sync) {
	    	List<RT2DocProcessorLogInfo> res = new ArrayList<>();
	    	res.addAll(this.msgs);
	    	Collections.sort(res);
	    	return res.stream().map(m -> m.toString()).collect(Collectors.toList());
    	}
    }

    //-------------------------------------------------------------------------
    public List<String> formatMsgsLogInfoForClient(RT2CliendUidType clientUid) {
    	synchronized (sync) {
	    	List<RT2DocProcessorLogInfo> res = new ArrayList<>();
	    	res.addAll(this.msgs);
	    	Collections.sort(res);
	    	return res.stream().filter(m -> (m.getClientUid() != null) && m.getClientUid().equals(clientUid)) 
	    			            .map(m -> m.toString())
	    			            .collect(Collectors.toList());
    	}
    }    
    
    //-------------------------------------------------------------------------
    protected List<RT2LogInfo> getMsgsAsList() {
    	synchronized (sync) {
    		return new ArrayList<>(msgs);
		}    	
    }
    
    //-------------------------------------------------------------------------	
	@Override
	public int hashCode() {
		final int prime = 31;
		int result = 1;
		result = prime * result + ((m_sDocUID == null) ? 0 : m_sDocUID.hashCode());
		return result;
	}

    //-------------------------------------------------------------------------	
	@Override
	public boolean equals(Object obj) {
		if (this == obj)
			return true;
		if (obj == null)
			return false;
		if (getClass() != obj.getClass())
			return false;
		RT2DocProcessor other = (RT2DocProcessor) obj;
		if (m_sDocUID == null) {
			if (other.m_sDocUID != null)
				return false;
		} else if (!m_sDocUID.equals(other.m_sDocUID))
			return false;
		return true;
	}

	@Override
	public RT2CliendUidType getClientUID() {
		return null;
	}

}
