/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.rt2.core.doc;

import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.atomic.AtomicReference;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.openexchange.office.rt2.core.sequence.SequenceDocProcessor;
import com.openexchange.office.rt2.protocol.RT2Message;
import com.openexchange.office.rt2.protocol.RT2Protocol;
import com.openexchange.office.tools.osgi.ServiceLookupRegistry;

//=============================================================================
public abstract class BaseDocProcessor extends SequenceDocProcessor implements IDocState
{
	enum ProcessingState
	{
		CREATED,
		INITIALIZING,
		OPEN,
		DISPOSED
	}

    //-------------------------------------------------------------------------
	private static final Logger log = LoggerFactory.getLogger(BaseDocProcessor.class);

    //-------------------------------------------------------------------------
	private static final Map<String, String> PREDEFINED_DEFAULT_METHODS_MAP;
	static
	{
		// predefined message types to be mapped to defined methods
		final HashMap<String, String> tmp = new HashMap<>();
        tmp.put(RT2Protocol.REQUEST_JOIN,               "onJoin"                 );
        tmp.put(RT2Protocol.REQUEST_LEAVE,              "onLeave"                );
		tmp.put(RT2Protocol.REQUEST_OPEN_DOC,           "onOpenDoc"              );
		tmp.put(RT2Protocol.REQUEST_CLOSE_DOC,          "onCloseDoc"             );
        tmp.put(RT2Protocol.REQUEST_SWITCH_DOCSTATE,    "onSwitchDocState"       );
		tmp.put(RT2Protocol.REQUEST_APPLY_OPS,          "onApplyOperations"      );
		tmp.put(RT2Protocol.REQUEST_SAVE_DOC,           "onSaveDoc"              );
		tmp.put(RT2Protocol.REQUEST_SYNC,               "onSync"                 );
		tmp.put(RT2Protocol.REQUEST_SYNC_STABLE,        "onSyncStable"           );
		tmp.put(RT2Protocol.REQUEST_EDITRIGHTS,         "onEditRights"           );
		tmp.put(RT2Protocol.REQUEST_UNAVAILABILITY,     "onUnavailable"          );
		tmp.put(RT2Protocol.REQUEST_EMERGENCY_LEAVE,    "onEmergencyLeave"       );

		PREDEFINED_DEFAULT_METHODS_MAP = Collections.unmodifiableMap(tmp);
	}

    //-------------------------------------------------------------------------
	private AtomicReference<ProcessingState> m_aProcState            = new AtomicReference<>(ProcessingState.CREATED);

    //-------------------------------------------------------------------------
    private MessageDispatcher                m_aCustomMsgDispatcher  = new MessageDispatcher(this.getClass(), (String)null);

    //-------------------------------------------------------------------------
    private MessageDispatcher                m_aDefaultMsgDispatcher = new MessageDispatcher(this.getClass(), PREDEFINED_DEFAULT_METHODS_MAP.values());

    //-------------------------------------------------------------------------
	public BaseDocProcessor() throws Exception
	{
	}

    //-------------------------------------------------------------------------
	@Override
	public boolean isDisposed()
    {
        return (m_aProcState.get() == ProcessingState.DISPOSED);
    }

    //-------------------------------------------------------------------------
	protected void setDisposed()
	{
        m_aProcState.set(ProcessingState.DISPOSED);
	}

	//-------------------------------------------------------------------------+
	@Override
	public void dispose()
    {
        final RT2DocStateNotifier aDocStateNotifier = ServiceLookupRegistry.get().getService(RT2DocStateNotifier.class);
        aDocStateNotifier.notifyEvent(RT2DocStateNotifier.DOCSTATE_EVENT_DISPOSED, this);

        super.dispose();
    }

    //-------------------------------------------------------------------------
	@Override
	public void handleRequest(final RT2Message aRawRequest) throws Exception
	{
		addMessageForQueueToLog(aRawRequest);
		
	    switch (aRawRequest.getType()) {
	        case REQUEST_OPEN_DOC:
	            {
	                if (m_aProcState.compareAndSet(ProcessingState.CREATED, ProcessingState.INITIALIZING)) {
	                    onFirstOpenDoc(aRawRequest);
	                    //  make sure to set processing state in all cases
	                    m_aProcState.set(ProcessingState.OPEN);
	                }
	                else if (m_aProcState.get() == ProcessingState.OPEN) {
	                    onOpenDoc(aRawRequest);
	                } else {
	                    waitForOpen();
	                    onOpenDoc(aRawRequest);
	                }
	            }
	            break;
	        default:
	            {
	                boolean bProcessed = processMessage(aRawRequest);
	                if (!bProcessed)
	                {
	                	log.warn("Message of type {} not successfully processed.", aRawRequest.getType());
	                }
	            }
	    }
	}
	
    //-------------------------------------------------------------------------
	protected void waitForOpen() throws Exception
	{
		while (m_aProcState.get() != ProcessingState.OPEN)
		{
			synchronized (m_aProcState)
			{
				m_aProcState.wait();
			}
		}
	}

    //-------------------------------------------------------------------------
	protected boolean processMessage(final RT2Message aMsg) throws Exception
	{
		final String sMsgType = aMsg.getType().getValue();

		// Dispatch message to the implementation method
		// using the default msg dispatcher or a custom msg
		// dispatcher.
		boolean bProcessed = false;
		if (PREDEFINED_DEFAULT_METHODS_MAP.containsKey(sMsgType))
			bProcessed = m_aDefaultMsgDispatcher.callMethod(this, PREDEFINED_DEFAULT_METHODS_MAP.get(sMsgType), aMsg);
		else
			bProcessed = m_aCustomMsgDispatcher.callMethod(this, aMsg);

		return bProcessed;
	}

	//=============================================================================
	// Default methods for predefined messages
	//=============================================================================

    //-------------------------------------------------------------------------
    public abstract boolean onJoin(final RT2Message aMsg) throws Exception;

    //-------------------------------------------------------------------------
    public abstract boolean onLeave(final RT2Message aMsg) throws Exception;

    //-------------------------------------------------------------------------
	public abstract boolean onFirstOpenDoc(final RT2Message aMsg) throws Exception;

    //-------------------------------------------------------------------------
	public abstract boolean onOpenDoc(final RT2Message aMsg) throws Exception;

    //-------------------------------------------------------------------------
	public abstract boolean onCloseDoc(final RT2Message aMsg) throws Exception;

    //-------------------------------------------------------------------------
	public abstract boolean onApplyOperations(final RT2Message aMsg) throws Exception;

    //-------------------------------------------------------------------------
	public abstract boolean onSaveDoc(final RT2Message aMsg) throws Exception;

    //-------------------------------------------------------------------------
	public abstract boolean onSync(final RT2Message aMsg) throws Exception;

    //-------------------------------------------------------------------------
	public abstract boolean onSyncStable(final RT2Message aMsg) throws Exception;

    //-------------------------------------------------------------------------
	public abstract boolean onEditRights(final RT2Message aMsg) throws Exception;

    //-------------------------------------------------------------------------
	public abstract boolean onUnavailable(final RT2Message aMsg) throws Exception;

    //-------------------------------------------------------------------------
	public abstract boolean onEmergencyLeave(final RT2Message aMsg) throws Exception;
}
