/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.rt2.core.doc;

import java.util.List;
import java.util.Optional;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.openexchange.office.document.AdvisoryLockInfoService;
import com.openexchange.office.rt2.protocol.RT2Message;
import com.openexchange.office.rt2.protocol.RT2MessageGetSet;
import com.openexchange.office.rt2.protocol.value.RT2CliendUidType;
import com.openexchange.office.session.SessionService;
import com.openexchange.office.tools.common.string.StringHelper;
import com.openexchange.office.tools.files.FileHelper;
import com.openexchange.office.tools.osgi.ServiceLookupRegistryService;
import com.openexchange.office.tools.user.UserData;
import com.openexchange.session.Session;

public class AdvisoryLockHelper {

    private static final Logger LOG = LoggerFactory.getLogger(AdvisoryLockHelper.class);
    private static final String ADVISORY_LOCK_RESET_ACTION = "reset";

    public AdvisoryLockHelper() {
        super();
    }

    // -------------------------------------------------------------------------
    public static void clearAdvisoryLock(UserData userData) throws Exception {
        final ServiceLookupRegistryService services = ServiceLookupRegistryService.getInstance();
        final AdvisoryLockInfoService advisoryLockInfoService = services.getService(AdvisoryLockInfoService.class);
        final String folderId = userData.getFolderId();
        final String fileId = userData.getFileId();

        final Session session = ServiceLookupRegistryService.getInstance().getService(SessionService.class).getSession4Id(userData.getSessionId());
        advisoryLockInfoService.clearAdvisoryLock(session, folderId, fileId);
    }

    // -------------------------------------------------------------------------
    public static boolean checkAndApplyAdvisoryLockAction(RT2Message openDocMsg, String nameOfSystem) throws Exception {
        final ServiceLookupRegistryService services = ServiceLookupRegistryService.getInstance();
        final AdvisoryLockInfoService advisoryLockInfoService = services.getService(AdvisoryLockInfoService.class);
        final String sessionId = RT2MessageGetSet.getSessionID(openDocMsg);
        final Session session = ServiceLookupRegistryService.getInstance().getService(SessionService.class).getSession4Id(sessionId);
        final String advisoryLockAction = RT2MessageGetSet.getAdvisoryLockAction(openDocMsg);

        if ((session != null) && advisoryLockAction.equalsIgnoreCase(ADVISORY_LOCK_RESET_ACTION)) {
            final String folderId = RT2MessageGetSet.getFolderID(openDocMsg);
            final String fileId = RT2MessageGetSet.getFileID(openDocMsg);
            boolean result = advisoryLockInfoService.resetAdvisoryLock(session, nameOfSystem, folderId, fileId);
            LOG.info("Advisory lock info reset for doc-uid {} referencing file-id {} by client-uid {}", openDocMsg.getDocUID(), fileId, openDocMsg.getClientUID());
            return result;
        } else if (session == null) {
            LOG.info("Advisory lock info cannot process advisory lock action due to unknown session for client-uid {}", openDocMsg.getClientUID());
        }
        return false;
    }

    // -------------------------------------------------------------------------
    public static boolean isLastUserWithWriteAccess(List<UserData> completeUserDataList, final RT2CliendUidType clientUidToGo) {
        if ((clientUidToGo == null) || (StringUtils.isEmpty(clientUidToGo.getValue())))
            return false;

        final String clientUidToCheck = clientUidToGo.getValue();
        final Optional<UserData> userDataToQuit = completeUserDataList.stream().filter(u -> isSameClientUid(u, clientUidToCheck)).findFirst();
        if (userDataToQuit.isPresent()) {
            return (canUserWriteToFile(userDataToQuit.get()) &&
                    completeUserDataList.stream().filter(u -> !isSameClientUid(u, clientUidToCheck)).noneMatch(u -> canUserWriteToFile(u)));
        }
        return false;
    }

    // -------------------------------------------------------------------------
    private static boolean canUserWriteToFile(final UserData userData) {
        if ((userData == null) || StringUtils.isEmpty(userData.getSessionId()) || StringUtils.isEmpty(userData.getFileId()))
            return false;

        try {
            final Session session = ServiceLookupRegistryService.getInstance().getService(SessionService.class).getSession4Id(userData.getSessionId());
            if (session != null) {
                return FileHelper.canWriteToFile(null, session, userData.getFileId());
            }

            return false;
        } catch (Exception e) {
            return false;
        }
    }

    // -------------------------------------------------------------------------
    private static boolean isSameClientUid(UserData userData, String clientUid) {
        return ((userData != null) && (StringHelper.isNotEmptyAndEqual(userData.getClientUid(), clientUid)));
    }

}
