/*
/*
*
*    OPEN-XCHANGE legal information
*
*    All intellectual property rights in the Software are protected by
*    international copyright laws.
*
*
*    In some countries OX, OX Open-Xchange, open xchange and OXtender
*    as well as the corresponding Logos OX Open-Xchange and OX are registered
*    trademarks of the OX Software GmbH group of companies.
*    The use of the Logos is not covered by the GNU General Public License.
*    Instead, you are allowed to use these Logos according to the terms and
*    conditions of the Creative Commons License, Version 2.5, Attribution,
*    Non-commercial, ShareAlike, and the interpretation of the term
*    Non-commercial applicable to the aforementioned license is published
*    on the web site http://www.open-xchange.com/EN/legal/index.html.
*
*    Please make sure that third-party modules and libraries are used
*    according to their respective licenses.
*
*    Any modifications to this package must retain all copyright notices
*    of the original copyright holder(s) for the original code used.
*
*    After any such modifications, the original and derivative code shall remain
*    under the copyright of the copyright holder(s) and/or original author(s)per
*    the Attribution and Assignment Agreement that can be located at
*    http://www.open-xchange.com/EN/developer/. The contributing author shall be
*    given Attribution for the derivative code and a license granting use.
*
*     Copyright (C) 2016-2020 OX Software GmbH
*     Mail: info@open-xchange.com
*
*
*     This program is free software; you can redistribute it and/or modify it
*     under the terms of the GNU General Public License, Version 2 as published
*     by the Free Software Foundation.
*
*     This program is distributed in the hope that it will be useful, but
*     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
*     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
*     for more details.
*
*     You should have received a copy of the GNU General Public License along
*     with this program; if not, write to the Free Software Foundation, Inc., 59
*     Temple Place, Suite 330, Boston, MA 02111-1307 USA
*
*/

package com.openexchange.office.rt2.core.control;

import java.lang.ref.WeakReference;
import java.util.LinkedList;
import java.util.Queue;
import java.util.concurrent.atomic.AtomicBoolean;

import org.apache.commons.lang3.Validate;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.openexchange.exception.ExceptionUtils;
import com.openexchange.logging.MDCEnabledThreadGroup;

public abstract class TaskProcessor<T extends Task> implements Runnable
{
    private static final Logger log = LoggerFactory.getLogger(TaskProcessor.class);

	//-------------------------------------------------------------------------
	private final static long DEFAULT_WAIT_TIME                           = 250L;

	//-------------------------------------------------------------------------
	private final AtomicBoolean                   m_aStarted              = new AtomicBoolean(false);
	private final AtomicBoolean                   m_aShutdown             = new AtomicBoolean(false);
	private Queue<T>                              m_aTaskQueue            = null;
	private final WeakReference<ITaskListener<T>> m_aListener;
	private Thread                                m_aQueueProcessorThread;

	//-------------------------------------------------------------------------
	public TaskProcessor(final ITaskListener<T> aListener)
	{
		m_aListener  = new WeakReference<>(aListener);
		m_aTaskQueue = createTaskQueue();
	}

	//-------------------------------------------------------------------------
	protected Queue<T> createTaskQueue()
	{
		return new LinkedList<>();
	}

	//-------------------------------------------------------------------------
	public boolean isStarted()
	{
		return m_aStarted.get();
	}

	//-------------------------------------------------------------------------
	public int getPendingTasks()
	{
		synchronized (m_aTaskQueue)
		{
			return m_aTaskQueue.size();
		}
	}

	//-------------------------------------------------------------------------
	public void start()
	{
		if (m_aStarted.compareAndSet(false, true))
		{
			synchronized (m_aTaskQueue)
			{
				m_aQueueProcessorThread = new Thread(new MDCEnabledThreadGroup(), this, getThreadName());
				m_aQueueProcessorThread.start();
			}

			log.trace("RT2: TaskProcessor started.");
		}
	}

	//-------------------------------------------------------------------------
	public void shutdown()
		throws Exception
	{
		if (isStarted() && m_aShutdown.compareAndSet(false, true))
		{
			Thread aQueueProcessorThread = null;
			synchronized (m_aTaskQueue)
			{
				m_aTaskQueue.clear();
				m_aTaskQueue.notifyAll();

				aQueueProcessorThread = m_aQueueProcessorThread;
				m_aQueueProcessorThread = null;
			}

			m_aStarted.set(false);

			if (null != aQueueProcessorThread)
				aQueueProcessorThread.join(2000);

			log.trace("RT2: TaskProcessor shutdown.");
		}
	}

	//-------------------------------------------------------------------------
	public void addTask(final T aTask)
		throws Exception
	{
		Validate.notNull(aTask);

		boolean bShutdown = false;

		synchronized (m_aTaskQueue)
		{
			bShutdown = m_aShutdown.get();
			if (!bShutdown)
			{
				log.debug("RT2: TaskProcessor adds task [{}]", aTask.getTaskID());

				m_aTaskQueue.add(aTask);
				m_aTaskQueue.notifyAll();
			}
		}

		if (bShutdown)
		{
			log.warn("RT2: TaskProcessor addTask called with shutdown true!");
		}
	}

	//-------------------------------------------------------------------------
	public void removeAll()
		throws Exception
	{
		boolean bShutdown = false;
		synchronized (m_aTaskQueue)
		{
			bShutdown = m_aShutdown.get();
			if (!bShutdown)
				m_aTaskQueue.clear();
		}

		if (bShutdown)
			log.warn("RT2: TaskProcessor removeAll called with shutdown true!");
	}

	//-------------------------------------------------------------------------
	@Override
	public void run()
	{
        while (!m_aShutdown.get())
        {
            T aCurrentTask = null;
			try
			{
				synchronized (m_aTaskQueue)
				{
					aCurrentTask = m_aTaskQueue.peek();
					if ((aCurrentTask == null) || !canProcessTask(aCurrentTask))
					{
						m_aTaskQueue.wait(getWaitingTime());
						continue;
					}
					m_aTaskQueue.poll();
				}

				try
				{
					aCurrentTask.process();

					final ITaskListener<T> aListener = m_aListener.get();
					if (null != aListener)
						aListener.taskCompleted(aCurrentTask);
				}
				catch (Throwable t)
				{
					ExceptionUtils.handleThrowable(t);

					final String sTaskDump = aCurrentTask.toString();
					log.error("RT2: Throwable received in TaskProcessor for task [" + sTaskDump + "] run - state unknown!", t);
				}
			}
			catch (InterruptedException e)
			{
				Thread.currentThread().interrupt();
			}
			catch (Throwable t)
			{
				ExceptionUtils.handleThrowable(t);
				String sTaskDump = (aCurrentTask == null) ? "unknown" : aCurrentTask.toString();
				log.error("RT2: Throwable received in TaskProcessor for task [" + sTaskDump + "] run - state unknown!", t);
			}
        }
    }

	//-------------------------------------------------------------------------
	protected long getWaitingTime()
	{
		return DEFAULT_WAIT_TIME;
	}

	//-------------------------------------------------------------------------
	protected boolean canProcessTask(final T aTask)
	{
		return (null != aTask);
	}

	//-------------------------------------------------------------------------
	protected abstract String getThreadName();

}
