/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */
package com.openexchange.office.rt2.core;

import java.lang.ref.WeakReference;
import java.time.LocalDateTime;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.apache.commons.lang3.time.DurationFormatUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.codahale.metrics.Counter;
import com.codahale.metrics.MetricRegistry;
import com.google.common.util.concurrent.ThreadFactoryBuilder;
import com.hazelcast.core.DistributedObject;
import com.hazelcast.core.DistributedObjectUtil;
import com.hazelcast.core.HazelcastInstance;
import com.hazelcast.core.IAtomicLong;
import com.openexchange.exception.ExceptionUtils;
import com.openexchange.java.ConcurrentHashSet;
import com.openexchange.log.LogProperties;
import com.openexchange.office.rt2.cache.ClusterLockException;
import com.openexchange.office.rt2.cache.ClusterLockService;
import com.openexchange.office.rt2.cache.ClusterLockService.ClusterLock;
import com.openexchange.office.rt2.cache.RT2DocInfo;
import com.openexchange.office.rt2.config.RT2ConfigItem;
import com.openexchange.office.rt2.core.doc.IDocProcessorContainer;
import com.openexchange.office.rt2.core.doc.IRT2DocProcessorManager;
import com.openexchange.office.rt2.core.doc.RT2DocProcessor;
import com.openexchange.office.rt2.core.doc.RT2DocProcessorManager;
import com.openexchange.office.rt2.hazelcast.RT2DocOnNodeMap;
import com.openexchange.office.rt2.protocol.value.RT2DocUidType;
import com.openexchange.office.rt2.proxy.EDocProxyState;
import com.openexchange.office.rt2.proxy.RT2DocInfoRegistry;
import com.openexchange.office.rt2.proxy.RT2DocProxy;
import com.openexchange.office.rt2.proxy.RT2DocProxyRegistry;
import com.openexchange.office.tools.common.weakref.WeakRefUtils;
import com.openexchange.office.tools.logging.annotation.LogMethodCallHelper;
import com.openexchange.office.tools.osgi.ServiceLookupRegistry;


//=============================================================================
/**
 */
public class RT2GarbageCollector
{

	public static final String REFCOUNT_CLIENT_PREFIX  = "rt2cache.atomic.";
	public static final String REFCOUNT_CLIENT_POSTFIX = ".refcount.clients.a";

    //-------------------------------------------------------------------------
	private static final Logger log = LoggerFactory.getLogger(RT2GarbageCollector.class);

	private final RT2DocProcessorManager docProcContainer;

	private final RT2NodeInfoService nodeInfo;

	private final ClusterLockService clusterLockService;

	private final ScheduledExecutorService finalRt2DocProcessorRemoveScheduler = Executors.newSingleThreadScheduledExecutor(new RT2ThreadFactoryBuilder("RT2GarbageCollectorFinalProcRemove-%d").build());

	private final HazelcastInstance hazelcastInstance;

	private final Counter docProxyOfflineCounter;
	private final Counter docProxyRemovedCounter;
	private final Counter docProcRemovedCounter;
	private final Counter refCountClientsRemovedCounter;
	private final Counter docOnNodeMapEntryRemovedCounter;

	private Pattern docUidOfAtomicLongNamePattern = Pattern.compile("rt2cache.atomic.(.*).refcount.clients.a");

	private Map<RT2DocUidType, LocalDateTime> atomicLongsToGc = new HashMap<>();
	private Set<RT2DocUidType> atomicLongsToBeVerified = new ConcurrentHashSet<>();
	private Set<RT2DocUidType> docUidsToClear = new ConcurrentHashSet<>();
	private long minWaitTimeBeforeGcAtomicLong = 5 * 60; // 5 minutes in seconds

    //-------------------------------------------------------------------------
	public RT2GarbageCollector (HazelcastInstance hazelcastInstance, RT2DocProcessorManager docProcContainer, MetricRegistry metricRegistry, RT2NodeInfoService nodeInfo, ClusterLockService clusterLockService) {
		this.docProcContainer = docProcContainer;
		this.nodeInfo = nodeInfo;
		this.clusterLockService = clusterLockService;
		this.hazelcastInstance = hazelcastInstance;
		this.docProxyOfflineCounter = metricRegistry.counter(MetricRegistry.name("GarbageCollector", "docProxy", "offline"));
		this.docProxyRemovedCounter = metricRegistry.counter(MetricRegistry.name("GarbageCollector", "docProxy", "removed"));
		this.docProcRemovedCounter = metricRegistry.counter(MetricRegistry.name("GarbageCollector", "docProcessor", "removed"));
		this.refCountClientsRemovedCounter = metricRegistry.counter(MetricRegistry.name("GarbageCollector", "refCountClients", "removed"));
		this.docOnNodeMapEntryRemovedCounter = metricRegistry.counter(MetricRegistry.name("GarbageCollector", "docOnNodeMap", "removed"));
	}

    //-------------------------------------------------------------------------
    RT2GarbageCollector(RT2DocProcessorManager docProcContainer, RT2NodeInfoService nodeInfo, ClusterLockService clusterLockService, HazelcastInstance hazelcastInstance, Counter docProxyOfflineCounter,
			Counter docProxyRemovedCounter, Counter docProcRemovedCounter, Counter refCountClientsRemovedCounter, Counter docOnNodeMapEntryRemovedCounter,
			long minWaitTimeBeforeGcAtomicLong) {
		this.docProcContainer = docProcContainer;
		this.nodeInfo = nodeInfo;
		this.clusterLockService = clusterLockService;
		this.hazelcastInstance = hazelcastInstance;
		this.docProxyOfflineCounter = docProxyOfflineCounter;
		this.docProxyRemovedCounter = docProxyRemovedCounter;
		this.docProcRemovedCounter = docProcRemovedCounter;
		this.refCountClientsRemovedCounter = refCountClientsRemovedCounter;
		this.docOnNodeMapEntryRemovedCounter = docOnNodeMapEntryRemovedCounter;
		this.minWaitTimeBeforeGcAtomicLong = minWaitTimeBeforeGcAtomicLong;
	}

	//-------------------------------------------------------------------------
    public Map<RT2DocUidType, LocalDateTime> getAtomicLongsToGc() {
		return new HashMap<>(atomicLongsToGc);
	}

	//-------------------------------------------------------------------------
	public Set<RT2DocUidType> getAtomicLongsToBeVerified() {
		return new HashSet<>(atomicLongsToBeVerified);
	}

	//-------------------------------------------------------------------------
    public synchronized void start ()
        throws Exception
	{
	    if (m_aScheduler != null)
	        return;

	    LogMethodCallHelper.logMethodCall(log, this, "start");

        final Runnable aRunner = () -> {impl_doGC ();};

        final long                     nFrequencyInMS = RT2ConfigItem.get().getRT2GCFrequencyInMS();

        ThreadFactory namedThreadFactory = new ThreadFactoryBuilder().setNameFormat("RT2GarbageCollector-%d").build();
        final ScheduledExecutorService aScheduler     = Executors.newScheduledThreadPool(1, namedThreadFactory);
        aScheduler.scheduleAtFixedRate(aRunner, nFrequencyInMS, nFrequencyInMS, TimeUnit.MILLISECONDS);
        m_aScheduler = aScheduler;
        LogMethodCallHelper.logMethodCallRes(log, this.getClass(), "start", Void.class);
	}

    //-------------------------------------------------------------------------
    public synchronized void stop ()
        throws Exception
    {
        if (m_aScheduler == null)
            return;

        final ScheduledExecutorService aScheduler = m_aScheduler;
        m_aScheduler = null;
        LogMethodCallHelper.logMethodCall(log, this, "stop");

        aScheduler.shutdownNow();
        finalRt2DocProcessorRemoveScheduler.shutdown();

        final boolean bOK = aScheduler.awaitTermination(30000, TimeUnit.MILLISECONDS);
        if ( ! bOK) {
            log.warn("... RT2-GC not stopped in time !");
            // no exception please - GC is stopped in shutdown of process ...
            // so thread will be stopped anyway ...
        } else {
            LogMethodCallHelper.logMethodCallRes(log, this.getClass(), "stop", Void.class);
        }
    }

    //-------------------------------------------------------------------------

	public boolean doGcForDocUid(RT2DocUidType docUid, boolean rememberWhenException) {
        final ClusterLock clusterLock = clusterLockService.getLock(docUid);
        boolean locked = false;
        boolean lockExceptionOccured = false;
        try {
            log.debug("Lock called by garbage collector for doc with id [{}]", docUid);
            locked = clusterLock.lock();
            nodeInfo.deregisterDocOnNode(docUid);
            final RT2DocInfo aDocInfo = mem_DocInfoRegistry().getDocInfo(docUid);
            aDocInfo.destroyRefCount4Clients();
            return true;
        } catch (ClusterLockException e) {
        	lockExceptionOccured = true;
		} finally {
        	if (locked) {
        		clusterLock.unlock();
        	}
		}
        if (rememberWhenException && (!locked || lockExceptionOccured)) {
        	docUidsToClear.add(docUid);
        }
        return false;
	}

    //-------------------------------------------------------------------------

    public void doGC() {
    	impl_doGC();
    }

    //-------------------------------------------------------------------------
    public void registerDocRefCountForRemovalVerification(RT2DocUidType refCountOfdocUIDToCheck) {
        atomicLongsToBeVerified.add(refCountOfdocUIDToCheck);
    }

    //-------------------------------------------------------------------------
    private void impl_doGC () {
        try {
            LogProperties.putProperty(LogProperties.Name.RT2_BACKEND_PART, "RT2GarbageCollector");
            LogProperties.putProperty(LogProperties.Name.RT2_BACKEND_UID, nodeInfo.getNodeUUID());
            checkAndGCOfflineOrStaleDocProxies();
            checkAndGCStaleDocResources();
            checkAndGCAtomicLongAsClientRef();
            checkConsitencyOfDocOnNodeMap();
            checkDetectedDocUids();
        } catch (Exception ex) {
            log.error(ex.getMessage(), ex);
        }
    }

    //-------------------------------------------------------------------------
    private void checkDetectedDocUids() {
		final Set<RT2DocUidType> toRemove = new HashSet<>();
		docUidsToClear.forEach(docUid -> {
			if (doGcForDocUid(docUid, false)) {
				toRemove.add(docUid);
			}
		});
		docUidsToClear.removeAll(toRemove);
	}

    //-------------------------------------------------------------------------
    private /* no synchronized */ void checkAndGCOfflineOrStaleDocProxies () {
        LogMethodCallHelper.logMethodCall(log, this, "checkAndGCOfflineOrStaleDocProxies");

        final RT2DocProxyRegistry aDocRegistry         = mem_DocProxyRegistry();
        final List< RT2DocProxy > lDocProxies          = aDocRegistry.listAllDocProxies();
        final long                nOfflineTresholdInMS = RT2ConfigItem.get().getRT2GCOfflineTresholdInMS();
        final long                nNow                 = System.currentTimeMillis();

        for (final RT2DocProxy aDocProxy : lDocProxies) {
            final long nOfflineSinceInMS  = aDocProxy.getOfflineTimeInMS();
            final boolean bOfflineTimeout = (nOfflineSinceInMS > nOfflineTresholdInMS);
            final boolean bStaleDocProxy  = isStaleDocProxy(aDocProxy, nNow, nOfflineTresholdInMS);

            if (bOfflineTimeout || bStaleDocProxy) {
                if (bOfflineTimeout) {
                	docProxyOfflineCounter.inc();
                    final String sOfflineSinceHumanReadable = DurationFormatUtils.formatDuration(nOfflineSinceInMS, "HH:mm:ss", /* pad with zeros */ true);
                    log.debug("Doc proxy {} is offline for {}", aDocProxy, sOfflineSinceHumanReadable);
                } else {
                    final long nStaleDuration = Math.abs(nNow - aDocProxy.getCreationTimeInMS());
                    final String sStaleSinceHumanReadable = DurationFormatUtils.formatDuration(nStaleDuration, "HH:mm:ss", /* pad with zeros */ true);
                    log.debug("Doc proxy {} is stale for {}", aDocProxy, sStaleSinceHumanReadable);
                }

	            if (!bStaleDocProxy) {
	            	try {
	            		aDocProxy.closeHard(false);
	            	} catch (Exception ex) {
	            		// An exception here is not a problem because there is an additional task(checkAndGCStaleDocResources()) which removes the corresponding RT2DocProcessor
	            		log.debug("GC failed for docProxy {}", aDocProxy, ex);
	            	}
	            }
	            docProxyRemovedCounter.inc();
                aDocRegistry.deregisterDocProxy(aDocProxy, true);
            }
        }
    }

    //-------------------------------------------------------------------------
    private /* no synchronized */ boolean isStaleDocProxy(final RT2DocProxy aDocProxy, long nNow, long nOfflineTresholdInMS) {
        Validate.notNull(aDocProxy);

        final long nCreationTime           = aDocProxy.getCreationTimeInMS();
        final EDocProxyState docProxyState = aDocProxy.getDocState();

        return ((docProxyState == EDocProxyState.E_INIT) && (Math.abs(nNow - nCreationTime) > nOfflineTresholdInMS));
    }

    //-------------------------------------------------------------------------
    private /* no synchronized */ void checkAndGCStaleDocResources ()
    {
        LogMethodCallHelper.logMethodCall(log, this, "checkAndGCStaleDocResources");
        final Set<WeakReference<RT2DocProcessor>> aLocalDocProcessors = docProcContainer.getWeakReferenceToDocProcessors();

        if (!aLocalDocProcessors.isEmpty())
        {
            final Set<RT2DocProcessor> aDocProcessorsWOClients = retrieveDocProcessorsWithoutClients(aLocalDocProcessors);
            final RT2DocInfoRegistry       aDocRegistry        = mem_DocInfoRegistry ();
            Set<RT2DocUidType> possibleDocProcessorToRemove = new HashSet<>();
            for (final RT2DocProcessor aDocProc : aDocProcessorsWOClients)
            {
                final RT2DocUidType     sDocUID   = aDocProc.getDocUID();
                final RT2DocInfo aDocInfo  = aDocRegistry.peekDocInfo(sDocUID);
               	destroyRefCount4Clients(sDocUID, aDocInfo);
                possibleDocProcessorToRemove.add(sDocUID);
            }
            if (!possibleDocProcessorToRemove.isEmpty()) {
            	finalRt2DocProcessorRemoveScheduler.schedule(new FinalRt2DocProcessorRemoveThread(possibleDocProcessorToRemove), 5, TimeUnit.MINUTES);
            }
        }
    }

    //-------------------------------------------------------------------------
	private void destroyRefCount4Clients(final RT2DocUidType docUID, final RT2DocInfo docInfo) {
		final ClusterLock clusterLock = clusterLockService.getLock(docUID);
		boolean locked = false;
		try {
			locked = clusterLock.lock();
			if (locked) {
				docInfo.destroyRefCount4Clients();
			}
		} catch (ClusterLockException e) {
			// Ignore, try again next time
		} finally {
			if (locked) {
		        clusterLock.unlock();
			}
		}
	}

    //-------------------------------------------------------------------------
    void checkAndGCAtomicLongAsClientRef ()
    {
        final HazelcastInstance aHzCore = hazelcastInstance;
        if (null != aHzCore) {
        	final Collection<DistributedObject> aDistributedObjects = aHzCore.getDistributedObjects();
            final Set<DistributedObject> aClientRefCountSet  =
            		aDistributedObjects.stream()
                                       .filter(RT2GarbageCollector::isAtomicLong)
                                       .filter(RT2GarbageCollector::isClientRefCountObject)
                                       .collect(Collectors.toSet());

            // check distributed object list for gc-able ref-counts
            for (final DistributedObject aObject : aClientRefCountSet)
            {
                try
                {
                    checkAndDestroyAtomicLongSafely((IAtomicLong)aObject);
                }
                catch (Throwable t)
                {
                    ExceptionUtils.handleThrowable(t);
                    String distributeObjectname = ((null == aObject) ? "null" : aObject.getName());
                    log.error("RT2: GC failed for client ref-count instance " + distributeObjectname  + " found with count = 0", t);
                }
            }

            // insert the registered atomic long names in the gc map
            for (final RT2DocUidType docUIDRefCount : atomicLongsToBeVerified) {
                final String hzAtomicLongName = RT2DocInfo.generateNameFromDocUid(docUIDRefCount.getValue());
                IAtomicLong hzRefCountForDoc = aHzCore.getAtomicLong(hzAtomicLongName);

                try {
                    checkAndDestroyAtomicLongSafely(hzRefCountForDoc);
                } catch (Throwable t) {
                    ExceptionUtils.handleThrowable(t);
                    log.error("RT2: GC failed for client ref-count instance " + hzAtomicLongName  + " found with count = 0", t);
                }
            }
        }
    }

    //-------------------------------------------------------------------------
    public static boolean isClientRefCountObject(final DistributedObject distributedObject)
    {
        return DistributedObjectUtil.getName(distributedObject).startsWith(REFCOUNT_CLIENT_PREFIX);
    }

    //-------------------------------------------------------------------------
    public static boolean isAtomicLong(final DistributedObject distributedObject)
    {
        return (distributedObject instanceof IAtomicLong);
    }

    //-------------------------------------------------------------------------
    private void checkAndDestroyAtomicLongSafely(IAtomicLong aClientRefCount)
    {
    	final String docUID = getDocUIDFromClientRefCountName(aClientRefCount);
        if (StringUtils.isNotEmpty(docUID) && shouldBeVerifiedOrGarbageCollected(aClientRefCount)) {
        	RT2DocUidType docUidObj = new RT2DocUidType(docUID);
        	ClusterLock clusterLock = clusterLockService.getLock(new RT2DocUidType(docUID));
            try {
            	clusterLock.lock();
                long refCount = aClientRefCount.get();
                boolean bCanBeGC = (refCount == 0);
                if (!bCanBeGC && ((refCount >= RT2Constants.REF_COUNT_TRESHHOLD) || refCount < 0)) {
                	if (atomicLongsToGc.containsKey(docUidObj)) {
                		LocalDateTime insertTime = atomicLongsToGc.get(docUidObj);
                		if (insertTime.plusSeconds(minWaitTimeBeforeGcAtomicLong).isBefore(LocalDateTime.now())) {
                			bCanBeGC = true;
                			atomicLongsToGc.remove(docUidObj);
                            atomicLongsToBeVerified.remove(docUidObj);
                		}
                	} else {
                		atomicLongsToGc.put(docUidObj, LocalDateTime.now());
                	}
                } else {
                	atomicLongsToGc.remove(docUidObj);
                    atomicLongsToBeVerified.remove(docUidObj);
                }
                if (bCanBeGC) {
					Matcher m = docUidOfAtomicLongNamePattern.matcher(aClientRefCount.getName());
					if (m.find()) {
						LogProperties.putProperty(LogProperties.Name.RT2_DOC_UID, m.group(1));
						log.info("Removing AtomicLong with name {}", aClientRefCount.getName());
					}
                	refCountClientsRemovedCounter.inc();
                	aClientRefCount.destroy();
                }
        	} catch (ClusterLockException e) {
        		// Ignore, try again next time
        	} finally {
            	clusterLock.unlock();
				LogProperties.remove(LogProperties.Name.RT2_DOC_UID);
            }
        } else {
        	// verification completed, remove entry as we see a normal value for the ref-count
        	if (docUID != null) {
	        	RT2DocUidType docUidObj = new RT2DocUidType(docUID);
	            atomicLongsToBeVerified.remove(docUidObj);
	            atomicLongsToGc.remove(docUidObj);
        	}
        }
    }

    //-------------------------------------------------------------------------
    private static boolean shouldBeVerifiedOrGarbageCollected(IAtomicLong atomicLong) {
        long refCount = atomicLong.get();
        return (refCount <= 0) || (refCount >= RT2Constants.REF_COUNT_TRESHHOLD);
    }

    //-------------------------------------------------------------------------
    private static String getDocUIDFromClientRefCountName(final IAtomicLong aClientRefCount)
    {
        String sResult = null;

        final String sName  = aClientRefCount.getName();
        final int    nStart = REFCOUNT_CLIENT_PREFIX.length();
        final int    nEnd   = sName.indexOf(".", nStart);

        if ((nEnd > nStart) && (nEnd < sName.length()))
            sResult = sName.substring(nStart, nEnd);

        return sResult;
    }


    //-------------------------------------------------------------------------
    private /* no synchronized */ void checkConsitencyOfDocOnNodeMap() {
    	RT2DocOnNodeMap aDocOnNodeMap = mem_DocOnNodeMap();
    	Set<String> docIds = aDocOnNodeMap.getDocsOfMember();
    	IDocProcessorContainer docProcMngr = mem_DocProcessorManager();
    	for (String str : docIds) {
    		RT2DocUidType docUid = new RT2DocUidType(str);
    		if (!docProcMngr.contains(docUid)) {
				LogProperties.putProperty(LogProperties.Name.RT2_DOC_UID, docUid.getValue());
				log.info("Removing docOnNodeMap entry with key {}", docUid.getValue());
				ClusterLock clusterLock = clusterLockService.getLock(docUid);
        		try {
    				clusterLock.lock();
    				if (aDocOnNodeMap.remove(str) == null) {
    					LogProperties.putProperty(LogProperties.Name.RT2_DOC_UID, str);
    					log.error("Cannot remove entry with docUid {} from DocOnNodeMap", str);
    				}
    				final RT2DocInfo   aDocInfo   = mem_DocInfoRegistry().getDocInfoWithoutCreate(docUid);
    				if (aDocInfo != null) {
    					aDocInfo.destroyRefCount4Clients();
    					docOnNodeMapEntryRemovedCounter.inc();
    				} else {
    					LogProperties.putProperty(LogProperties.Name.RT2_DOC_UID, str);
    					log.warn("No RefCount4Clients for document with id {} fround!", str);
    				}
        		} catch (ClusterLockException e) {
            		// Ignore, try again next time
            	} finally {
   					clusterLock.unlock();
                    LogProperties.removeProperty(LogProperties.Name.RT2_DOC_UID);
    			}
    			LogProperties.remove(LogProperties.Name.RT2_DOC_UID);
    		}
    	}
    }

    //-------------------------------------------------------------------------
    private /* no synchronized */ Set<RT2DocProcessor> retrieveDocProcessorsWithoutClients (final Set<WeakReference<RT2DocProcessor>> aLocalDocProcessors)
    {
        return aLocalDocProcessors.stream()
                                  .map(WeakRefUtils::getHardRef)
                                  .filter(Objects::nonNull)
                                  .filter(docProc -> CollectionUtils.isEmpty(docProc.getClientsInfo()))
                                  .collect(Collectors.toSet());
    }

    //-------------------------------------------------------------------------
    private synchronized RT2DocInfoRegistry mem_DocInfoRegistry ()
    {
        final RT2DocInfoRegistry aDocRegistry = ServiceLookupRegistry.get ().getService(RT2DocInfoRegistry.class);
        return aDocRegistry;
    }

    //-------------------------------------------------------------------------
    private synchronized RT2DocProxyRegistry mem_DocProxyRegistry ()
    {
        final RT2DocProxyRegistry aDocRegistry = ServiceLookupRegistry.get ().getService(RT2DocProxyRegistry.class);
        return aDocRegistry;
    }

    //-------------------------------------------------------------------------
    private synchronized RT2DocOnNodeMap mem_DocOnNodeMap() {
    	final RT2DocOnNodeMap aDocOnNodeMap = ServiceLookupRegistry.get().getService(RT2DocOnNodeMap.class);
    	return aDocOnNodeMap;
    }

    private synchronized IDocProcessorContainer mem_DocProcessorManager() {
    	final IDocProcessorContainer docProcessorManager = ServiceLookupRegistry.get().getService(IRT2DocProcessorManager.class);
    	return docProcessorManager;
    }

    //-------------------------------------------------------------------------
    private ScheduledExecutorService m_aScheduler = null;

    private class FinalRt2DocProcessorRemoveThread implements Runnable {

    	private final Set<RT2DocUidType> possibleDocProcessorToRemove;

    	public FinalRt2DocProcessorRemoveThread(Set<RT2DocUidType> possibleDocProcessorToRemove) {
    		this.possibleDocProcessorToRemove = possibleDocProcessorToRemove;
    	}

		@Override
		public void run() {
			final Set<WeakReference<RT2DocProcessor>> aLocalDocProcessors = docProcContainer.getWeakReferenceToDocProcessors();
			final Set<RT2DocProcessor> aDocProcessorsWOClients = retrieveDocProcessorsWithoutClients(aLocalDocProcessors);
			final RT2DocInfoRegistry       aDocRegistry        = mem_DocInfoRegistry ();
			for (RT2DocProcessor docProc : aDocProcessorsWOClients) {
				if (possibleDocProcessorToRemove.contains(docProc.getDocUID())) {
	                final RT2DocUidType     sDocUID   = docProc.getDocUID();
	                final RT2DocInfo aDocInfo  = aDocRegistry.peekDocInfo(sDocUID);
					destroyRefCount4Clients(sDocUID, aDocInfo);
					docProcContainer.docProcessorDisposed(docProc);
					LogProperties.putProperty(LogProperties.Name.RT2_DOC_UID, sDocUID.getValue());
					log.info("Removed doc processor with docUid {} because there is no client for the last 5 minutes.", sDocUID);
					LogProperties.remove(LogProperties.Name.RT2_DOC_UID);
					docProcRemovedCounter.inc();
				}
			}
		}
    }
}
