package com.openexchange.office.rt2.cache;

import java.time.LocalTime;
import java.util.HashSet;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.hazelcast.core.HazelcastInstance;
import com.hazelcast.core.IMap;
import com.openexchange.office.rt2.core.RT2ThreadFactoryBuilder;
import com.openexchange.office.rt2.protocol.value.RT2DocUidType;

public class ClusterLockService {

	public static final String LOCK_PREFIX = "rt2_";
	
	public static final String LOCK_MAP_NAME = "rt2DocLockMap";
	
	private static final Logger log = LoggerFactory.getLogger(ClusterLockService.class);
	
	private final IMap<String, String> hzLockMap;

	private long maxLockTimeInSeconds;
	
	private long maxLockExistTimeInMinutes;
	
	private ConcurrentHashMap<String, ClusterLock> lockMap = new ConcurrentHashMap<>();
	
	private final ScheduledExecutorService docLockRemoveScheduledExecutorService = Executors.newScheduledThreadPool(1, new RT2ThreadFactoryBuilder("DocLockRemoveThread-%d").build());
	private final ScheduledExecutorService clusterLockUnlockThreadScheduledExecService = Executors.newScheduledThreadPool(1, new RT2ThreadFactoryBuilder("ClusterLockUnlockThread-%d").build());
	
	public ClusterLockService(HazelcastInstance hzInstance) {
		this(hzInstance, 60, 30);
	}

	public ClusterLockService(HazelcastInstance hzInstance, long maxLockTimeInSeconds, long maxLockExistTimeInMinutes) {
		this.maxLockTimeInSeconds = maxLockTimeInSeconds;
		this.maxLockExistTimeInMinutes = maxLockExistTimeInMinutes;
		this.hzLockMap = hzInstance.getMap(LOCK_MAP_NAME);
		clusterLockUnlockThreadScheduledExecService.scheduleWithFixedDelay(new ClusterLockUnlockThread(), 1, 1, TimeUnit.MINUTES);
		docLockRemoveScheduledExecutorService.scheduleWithFixedDelay(new DocLockRemoveThread(), 1, 1, TimeUnit.MINUTES);
	}
	
	public void stop() {
		clusterLockUnlockThreadScheduledExecService.shutdown();
		docLockRemoveScheduledExecutorService.shutdown();
	}
	
    public ClusterLock getLock(RT2DocUidType docUid)  {
    	String lockName = LOCK_PREFIX + docUid.getValue();
        log.debug("ClusterLockImpl: tryLock for id {}", docUid);
        return getLock(lockName);
    }

    private ClusterLock getLock(String name) {
    	ClusterLock newLock = new ClusterLock(name);
    	ClusterLock prevLock = lockMap.putIfAbsent(name, newLock);
    	return prevLock == null ? newLock : prevLock;
    }
        
    public Set<String> getLocks() {
    	return lockMap.values().stream().map(lock -> lock.toString()).collect(Collectors.toSet());
    }    
    
    private class ClusterLockUnlockThread implements Runnable {

		@Override
		public void run() {
			final LocalTime now = LocalTime.now();
			lockMap.values().forEach(c -> {
				if (now.minusSeconds(maxLockTimeInSeconds).isAfter(c.getLastLockTime())) {
					log.warn("Executing forceUnlock on lock with id {}", c.getName());						
					if (hzLockMap.isLocked(c.getName())) {
						hzLockMap.unlock(c.getName());
					} else {
						log.warn("Doc with id {} should be unlocked with force, but it is not locked!", c.getName());
					}
				}
			});				
		}    	
    }
    
    private class DocLockRemoveThread implements Runnable {

		@Override
		public void run() {
			final LocalTime now = LocalTime.now(); 
			Set<String> toDel = new HashSet<>();
			lockMap.values().forEach(c -> {
				if (now.minusMinutes(maxLockExistTimeInMinutes).isAfter(c.getLastAccessTime())) {
					toDel.add(c.getName());			
				}
			});
//				hzLockMap.removeAll(e -> toDel.contains(e.getKey()));
			lockMap.keySet().removeIf(e -> toDel.contains(e));
		}    	
    }
        
    public class ClusterLock {
    	private final String name;
//    	private final ILock lock;
    	private final LocalTime creationTime;
    	private LocalTime lastAccessTime;
    	private LocalTime lastLockTime;    	

    	public ClusterLock(String name) {
			this.name = name;
			this.creationTime = LocalTime.now();
			this.lastAccessTime = LocalTime.now();			
		}
    	
    	public boolean lock() throws ClusterLockException {
    		return lock(10000);
    	}

    	public boolean lock(long waitTime) throws ClusterLockException {
    		return lock(waitTime, 1);
    	}
    	
    	private boolean lock(long waitTime, int tryIdx)  throws ClusterLockException {
    		log.debug("Lock with id {}, instance {}", name, this);
    		boolean res = false;
    		this.lastAccessTime = LocalTime.now();    		
    		try {
    			res = (hzLockMap.tryLock(name, waitTime, TimeUnit.MILLISECONDS));
    			if (res) {
        			lastLockTime = LocalTime.now();
    			}
    		} catch (InterruptedException ex) {
    			res = false;
    		}
    		if (!res) {
    			if (tryIdx < 3) {
    				log.warn("Wait time {} for lock {} exceeded, tries {}, trying again", waitTime, name, tryIdx);
    				lock(waitTime, ++tryIdx);
    			} else {
    				log.warn("Wait time {} for lock {} exceeded, tries {}, force unlock", waitTime, name, tryIdx);
    				hzLockMap.forceUnlock(name);
    				lock(waitTime, 1);
    			}
    		}
    		return res;
    	}

    	public void unlock() {
			log.debug("Unlock called with id {}, instance {}", name, this);
    		if (lastLockTime != null) {
    			log.debug("Unlocking lock with id {}", name);
	    		this.lastAccessTime = LocalTime.now();
    		    this.lastLockTime = null;
	    		hzLockMap.unlock(name);
    		}    		
    	}

    	public boolean isLocked() {
    		return hzLockMap.isLocked(name);
    	}
    	
		public String getName() {
			return name;
		}

		public LocalTime getLastAccessTime() {
			return lastAccessTime;
		}

		public LocalTime getLastLockTime() {
			return lastLockTime;
		}

		public LocalTime getCreationTime() {
			return creationTime;
		}

		@Override
		public String toString() {
			boolean unlock = false;
			LocalTime localLastLockTime = getLastLockTime();
			if (localLastLockTime != null) {
				unlock = LocalTime.now().minusSeconds(maxLockTimeInSeconds).isAfter(localLastLockTime);
			}
			boolean remove = false;
			LocalTime localLastAccessTime = getLastAccessTime();
			if (localLastAccessTime != null) {
				remove = LocalTime.now().minusMinutes(maxLockExistTimeInMinutes).isAfter(localLastAccessTime);	
			}			
			boolean locked = hzLockMap.isLocked(name);
			return "ClusterLock [name=" + name + ", creationTime=" + creationTime
					+ ", lastAccessTime=" + (localLastAccessTime == null ? "<unset>" : localLastAccessTime) 
					+ ", lastLockTime=" + (localLastLockTime == null ? "<unset>" : localLastLockTime) 
					+ ", locked=" + locked + ", unlock=" + unlock + ", remove=" + remove + "]";
		}
    }

    public void runClusterLockUnlockThread() {
    	new ClusterLockUnlockThread().run();
    }
    
    public void runDocLockRemoveThread() {
    	new DocLockRemoveThread().run();
    }
}
